/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */


CREATE OR REPLACE FUNCTION __update_not_overridden_table_autovacuum_params(
    full_tablename VARCHAR,
    expected_autovacuum_vacuum_threshold INTEGER,
    expected_autovacuum_vacuum_scale_factor FLOAT,
    new_autovacuum_vacuum_threshold INTEGER,
    new_autovacuum_vacuum_scale_factor FLOAT
)
RETURNS BOOLEAN as
$$
DECLARE
	rec RECORD;
	is_partitioned BOOLEAN;
	updated_partition BOOLEAN;
	updated BOOLEAN = FALSE;
BEGIN
    SELECT relkind = 'p'
      INTO is_partitioned
      FROM pg_class
     WHERE (relnamespace::regnamespace::text || '.' || relname) = full_tablename;

    IF is_partitioned IS NULL THEN
        RAISE EXCEPTION 'Table "%" does not exist', full_tablename;
    ELSIF is_partitioned THEN
        FOR rec in (SELECT child.relnamespace::regnamespace::text || '.' || child.relname AS partition_name
                      FROM pg_inherits
                      JOIN pg_class parent            ON pg_inherits.inhparent = parent.oid
                      JOIN pg_class child             ON pg_inherits.inhrelid   = child.oid
                     WHERE (parent.relnamespace::regnamespace::text || '.' || parent.relname) = full_tablename) LOOP
            SELECT * INTO updated_partition FROM __update_not_overridden_table_autovacuum_params(
                rec.partition_name,
                expected_autovacuum_vacuum_threshold := expected_autovacuum_vacuum_threshold,
                expected_autovacuum_vacuum_scale_factor := expected_autovacuum_vacuum_scale_factor,
                new_autovacuum_vacuum_threshold := new_autovacuum_vacuum_threshold,
                new_autovacuum_vacuum_scale_factor := new_autovacuum_vacuum_scale_factor
            );
            updated = updated OR updated_partition;
        END LOOP;
    ELSE
        -- non-partitioned table
        WITH table_params AS (
            SELECT json_object_agg(option_name, option_value) AS json
              FROM pg_class c
              JOIN pg_namespace n ON c.relnamespace = n.oid,
                   pg_options_to_table(c.reloptions)
             WHERE (n.nspname || '.' || c.relname) = full_tablename
        ) SELECT x.*
            INTO rec
            FROM table_params, json_to_record(table_params.json) AS x(autovacuum_vacuum_threshold int,
                                                                      autovacuum_vacuum_scale_factor float);

        IF rec.autovacuum_vacuum_threshold IS NOT DISTINCT FROM expected_autovacuum_vacuum_threshold AND
           rec.autovacuum_vacuum_scale_factor IS NOT DISTINCT FROM expected_autovacuum_vacuum_scale_factor THEN
           EXECUTE 'ALTER TABLE ' || full_tablename
                || ' SET (autovacuum_vacuum_threshold    = ' || new_autovacuum_vacuum_threshold || ', '
                || '      autovacuum_vacuum_scale_factor = ' || new_autovacuum_vacuum_scale_factor || ')';
           updated = TRUE;
        END IF;
    END IF;
    RETURN updated;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


SELECT __update_not_overridden_table_autovacuum_params(
    'sf.dir_current',
    expected_autovacuum_vacuum_threshold := 1000000,
    expected_autovacuum_vacuum_scale_factor := 0.0,
    new_autovacuum_vacuum_threshold := 10000,
    new_autovacuum_vacuum_scale_factor := 0.025
);
SELECT __update_not_overridden_table_autovacuum_params(
    'sf.dir_history',
    expected_autovacuum_vacuum_threshold := 100000,
    expected_autovacuum_vacuum_scale_factor := 0.0,
    new_autovacuum_vacuum_threshold := 10000,
    new_autovacuum_vacuum_scale_factor := 0.025
);

SELECT __update_not_overridden_table_autovacuum_params(
    'sf.file_current',
    expected_autovacuum_vacuum_threshold := 10000000,
    expected_autovacuum_vacuum_scale_factor := 0.0,
    new_autovacuum_vacuum_threshold := 10000,
    new_autovacuum_vacuum_scale_factor := 0.025
);
SELECT __update_not_overridden_table_autovacuum_params(
    'sf.file_history',
    expected_autovacuum_vacuum_threshold := 10000000,
    expected_autovacuum_vacuum_scale_factor := 0.0,
    new_autovacuum_vacuum_threshold := 10000,
    new_autovacuum_vacuum_scale_factor := 0.025
);

SELECT __update_not_overridden_table_autovacuum_params(
    'sf.job_result_current',
    expected_autovacuum_vacuum_threshold := 5000000,
    expected_autovacuum_vacuum_scale_factor := 0.0,
    new_autovacuum_vacuum_threshold := 10000,
    new_autovacuum_vacuum_scale_factor := 0.025
);
SELECT __update_not_overridden_table_autovacuum_params(
    'sf.job_result_history',
    expected_autovacuum_vacuum_threshold := 5000000,
    expected_autovacuum_vacuum_scale_factor := 0.0,
    new_autovacuum_vacuum_threshold := 10000,
    new_autovacuum_vacuum_scale_factor := 0.025
);

SELECT __update_not_overridden_table_autovacuum_params(
    'sf.tag_value_current',
    expected_autovacuum_vacuum_threshold := 5000000,
    expected_autovacuum_vacuum_scale_factor := 0.0,
    new_autovacuum_vacuum_threshold := 10000,
    new_autovacuum_vacuum_scale_factor := 0.025
);
SELECT __update_not_overridden_table_autovacuum_params(
    'sf.tag_value_history',
    expected_autovacuum_vacuum_threshold := 5000000,
    expected_autovacuum_vacuum_scale_factor := 0.0,
    new_autovacuum_vacuum_threshold := 10000,
    new_autovacuum_vacuum_scale_factor := 0.025
);
