/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */

CREATE SCHEMA sf;
CREATE SCHEMA fs_entry_queues;

CREATE DOMAIN sf.blkcnt_t AS bigint;
CREATE DOMAIN sf.non_negative_int AS integer
    CONSTRAINT non_negative_int_check CHECK ((0 <= VALUE));
CREATE DOMAIN sf.depth_t AS sf.non_negative_int;
CREATE DOMAIN sf.gid_t AS bigint;
CREATE DOMAIN sf.uint64 AS numeric(20,0)
    CONSTRAINT uint64_check CHECK (((0)::numeric <= VALUE));
CREATE DOMAIN sf.ino_t AS sf.uint64;
CREATE DOMAIN sf.nlinks_t AS bigint;
CREATE DOMAIN sf.off_t AS bigint;
CREATE DOMAIN sf.perms_t AS integer;
CREATE DOMAIN sf.time_t AS timestamp with time zone;
CREATE DOMAIN sf.type_t AS integer;
CREATE DOMAIN sf.uid_t AS bigint;

-- function below is used by custom hash operator below
CREATE OR REPLACE FUNCTION modulo_division_hash(value BIGINT, seed BIGINT)
RETURNS BIGINT AS $$
    -- this number is UINT64CONST(0x49a0f4dd15e5a8e3) from
    -- https://github.com/postgres/postgres/blob/REL_13_2/src/include/common/hashfn.h#L83
    -- which is called here:
    -- https://github.com/postgres/postgres/blob/REL_13_2/src/backend/partitioning/partbounds.c#L4585
    SELECT value::BIGINT - 5305509591434766563;
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;

CREATE FUNCTION modulo_division_hash(value INT, seed INT)
RETURNS BIGINT AS $$
   SELECT value::BIGINT - 5305509591434766563;
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;

CREATE FUNCTION modulo_division_hash(value SMALLINT, seed SMALLINT)
RETURNS BIGINT AS $$
   SELECT value::BIGINT - 5305509591434766563;
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;

-- This operator is used by tables that are partitioned by volume_id
CREATE OPERATOR FAMILY partition_custom_integer_hash_ops USING hash;

CREATE OPERATOR CLASS partition_custom_bigint_hash_op FOR TYPE BIGINT USING hash
   FAMILY partition_custom_integer_hash_ops AS
   OPERATOR 1 = (BIGINT, BIGINT),
   FUNCTION 2 modulo_division_hash(BIGINT, BIGINT);

CREATE OPERATOR CLASS partition_custom_int4_hash_op FOR TYPE INT USING hash
   FAMILY partition_custom_integer_hash_ops AS
   OPERATOR 1 = (BIGINT, INT),
   FUNCTION 2 modulo_division_hash(INT, INT);

CREATE OPERATOR CLASS partition_custom_int2_hash_op FOR TYPE SMALLINT USING hash
   FAMILY partition_custom_integer_hash_ops AS
   OPERATOR 1 = (BIGINT, SMALLINT),
   FUNCTION 2 modulo_division_hash(SMALLINT, SMALLINT);


CREATE TABLE sf.dir (
    id bigint NOT NULL,
    valid tstzrange NOT NULL,
    volume_id bigint NOT NULL,
    parent_id bigint,
    ancestor_ids bigint[] NOT NULL,
    depth sf.depth_t NOT NULL,
    inode sf.ino_t,
    size sf.off_t,
    blocks sf.blkcnt_t,
    ctime sf.time_t,
    atime sf.time_t,
    mtime sf.time_t,
    uid sf.uid_t,
    gid sf.gid_t,
    perms sf.perms_t,
    path text NOT NULL,
    name text NOT NULL,
    rec_aggrs jsonb NOT NULL,
    local_aggrs jsonb,
    history_id bigint,
    sync_time sf.time_t,
    out_of_sync_time sf.time_t DEFAULT NULL::timestamp with time zone,
    errors jsonb,
    custom_fs_attrs_id bigint
) PARTITION BY LIST (upper(valid));;
ALTER TABLE sf.dir ALTER COLUMN parent_id SET (n_distinct_inherited=-0.1);

CREATE SEQUENCE sf.object_id_seq
    START WITH 1
    CACHE 1000;

CREATE TABLE sf.dir_current PARTITION OF sf.dir (
    id DEFAULT nextval('sf.object_id_seq'::regclass),
    CONSTRAINT dir_current_depth_is_zero_only_for_root_path_ck CHECK (((0 < (depth)::integer) OR (((depth)::integer = 0) AND (path = ''::text) AND (name = ''::text)))),
    CONSTRAINT dir_current_parent_id_is_null_only_for_root_path_ck CHECK (((parent_id IS NOT NULL) OR ((depth)::integer = 0)))
) FOR VALUES IN ('infinity'::timestamptz)
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);


ALTER TABLE sf.dir_current ALTER COLUMN parent_id SET (n_distinct=-0.1);

CREATE TABLE sf.file (
    id bigint NOT NULL,
    valid tstzrange NOT NULL,
    volume_id bigint NOT NULL,
    parent_id bigint NOT NULL,
    inode sf.ino_t NOT NULL,
    size sf.off_t NOT NULL,
    blocks sf.blkcnt_t NOT NULL,
    ctime sf.time_t NOT NULL,
    atime sf.time_t NOT NULL,
    mtime sf.time_t NOT NULL,
    uid sf.uid_t NOT NULL,
    gid sf.gid_t NOT NULL,
    type sf.type_t NOT NULL,
    perms sf.perms_t NOT NULL,
    name text NOT NULL,
    target text,
    history_id bigint,
    sync_time sf.time_t,
    nlinks sf.nlinks_t,
    custom_fs_attrs_id bigint
)
PARTITION BY LIST (upper(valid));
ALTER TABLE sf.file ALTER COLUMN parent_id SET (n_distinct_inherited=-0.002);

CREATE TABLE sf.file_current PARTITION OF sf.file (
    id DEFAULT nextval('sf.object_id_seq'::regclass)
) FOR VALUES IN ('infinity'::timestamptz)
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);
ALTER TABLE sf.file_current ALTER COLUMN parent_id SET (n_distinct=-0.002);


CREATE TABLE sf.calculated_aggrs (
    id bigint,
    rec_aggrs jsonb,
    local_aggrs jsonb,
    volume_id bigint NOT NULL
);

CREATE TABLE sf.cifs_sid_mapping (
    cifs_sid text NOT NULL,
    name text NOT NULL
);

CREATE SEQUENCE sf.custom_fs_attrs_id_seq
    START WITH 1
    CACHE 1;


CREATE TABLE sf.custom_fs_attrs (
    id bigint NOT NULL DEFAULT nextval('sf.custom_fs_attrs_id_seq'::regclass),
    volume_id bigint NOT NULL,
    attrs jsonb NOT NULL,
    is_current boolean NOT NULL
) PARTITION BY LIST (is_current);

CREATE TABLE sf.custom_fs_attrs_current PARTITION OF sf.custom_fs_attrs (
    is_current DEFAULT TRUE
)
FOR VALUES IN (TRUE)
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);


CREATE TABLE sf.custom_fs_attrs_history PARTITION OF sf.custom_fs_attrs (
    is_current DEFAULT FALSE
)
FOR VALUES IN (FALSE)
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);


CREATE SEQUENCE sf.history_object_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


CREATE TABLE sf.dir_history PARTITION OF sf.dir (
    history_id DEFAULT nextval('sf.history_object_id_seq'::regclass) NOT NULL
) DEFAULT -- default means all this is good partition for all rows that do not have any other partition. We use it to allow rows with UPPER(valid) == NULL
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);
ALTER TABLE sf.dir_history ALTER COLUMN parent_id SET (n_distinct=-0.1);


CREATE SEQUENCE sf.dir_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1000;


CREATE TABLE sf.dir_metadata (
    id bigint DEFAULT nextval('sf.dir_metadata_id_seq'::regclass) NOT NULL,
    dir_id bigint NOT NULL,
    exclude_dirs text[] DEFAULT ARRAY[]::text[] NOT NULL,
    exclude_files text[] DEFAULT ARRAY[]::text[] NOT NULL,
    tree_out_of_sync_time timestamp with time zone,
    volume_id bigint NOT NULL
);


CREATE TABLE sf.dirs_with_recently_changed_recaggrs (
    fs_entry_id bigint,
    volume_id bigint
);

CREATE TABLE sf.file_history PARTITION OF sf.file (
    history_id DEFAULT nextval('sf.history_object_id_seq'::regclass) NOT NULL
)
DEFAULT -- partition for all rows with upper(valid) != infinity
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);
ALTER TABLE sf.file_history ALTER COLUMN parent_id SET (n_distinct=-0.002);


ALTER TABLE sf.file_history OWNER TO starfish;

CREATE SEQUENCE sf.win_gid_seq
    START WITH 6000000000
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


CREATE TABLE sf.gid_mapping (
    volume_id bigint NOT NULL,
    gid sf.uid_t DEFAULT nextval('sf.win_gid_seq'::regclass) NOT NULL,
    name text NOT NULL,
    sid text
);


CREATE TABLE sf.gid_to_uid_mapping (
    volume_id bigint NOT NULL,
    gid sf.gid_t NOT NULL,
    uid sf.uid_t NOT NULL
);

CREATE SEQUENCE sf.job_name_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE sf.job_name (
    id bigint NOT NULL DEFAULT nextval('sf.job_name_id_seq'::regclass),
    name text NOT NULL
);


CREATE SEQUENCE sf.job_result_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1000;


CREATE TABLE sf.job_result (
    id bigint NOT NULL DEFAULT nextval('sf.job_result_id_seq'::regclass),
    fs_entry_id bigint NOT NULL,
    inode sf.ino_t NOT NULL,
    mtime sf.time_t,
    ctime sf.time_t,
    run_time timestamp with time zone NOT NULL,
    name_id bigint NOT NULL,
    result jsonb NOT NULL,
    volume_id bigint,
    job_id bigint,
    is_current boolean NOT NULL
)
PARTITION BY LIST (is_current);



CREATE TABLE sf.job_result_current PARTITION OF sf.job_result (
    is_current DEFAULT TRUE
)
FOR VALUES IN (TRUE)
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);
ALTER TABLE sf.job_result_current ALTER COLUMN volume_id SET NOT NULL;


CREATE TABLE sf.job_result_history PARTITION OF sf.job_result (
    is_current DEFAULT FALSE
)
FOR VALUES IN (FALSE)
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);
ALTER TABLE sf.job_result_history ALTER COLUMN volume_id SET NOT NULL;


CREATE TABLE sf.refresh_aggregates (
    id bigint,
    depth bigint,
    volume_id bigint NOT NULL
);


CREATE SEQUENCE sf.tag_name_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


CREATE TABLE sf.tag_name (
    id bigint NOT NULL DEFAULT nextval('sf.tag_name_id_seq'::regclass),
    name text NOT NULL,
    namespace_id bigint NOT NULL
);


CREATE SEQUENCE sf.tag_namespace_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

CREATE TABLE sf.tag_namespace (
    id bigint NOT NULL DEFAULT nextval('sf.tag_namespace_id_seq'::regclass),
    name text NOT NULL,
    inheritable boolean DEFAULT true NOT NULL,
    pinnable boolean DEFAULT true NOT NULL,
    action character varying
);

INSERT INTO sf.tag_namespace (name, inheritable) VALUES
    ('', true),
    ('__zone', false),
    ('__archive', false);


CREATE TABLE sf.tag_value (
    fs_entry_id bigint NOT NULL,
    name_id bigint NOT NULL,
    value jsonb,
    volume_id bigint NOT NULL,
    is_current boolean NOT NULL
) PARTITION BY LIST (is_current);

CREATE TABLE sf.tag_value_current PARTITION OF sf.tag_value (
    is_current DEFAULT TRUE
)
FOR VALUES IN (TRUE)
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);


CREATE TABLE sf.tag_value_history PARTITION OF sf.tag_value (
    is_current DEFAULT FALSE
)
FOR VALUES IN (FALSE)
PARTITION BY HASH (volume_id partition_custom_bigint_hash_op);


CREATE SEQUENCE sf.win_uid_seq
    START WITH 5000000000
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


CREATE TABLE sf.uid_mapping (
    volume_id bigint NOT NULL,
    uid sf.uid_t DEFAULT nextval('sf.win_uid_seq'::regclass) NOT NULL,
    name text NOT NULL,
    sid text
);


CREATE TABLE sf.zone_namespace (
    zone_id bigint NOT NULL,
    namespace_id bigint NOT NULL
);


ALTER TABLE sf.custom_fs_attrs ALTER COLUMN id SET DEFAULT nextval('sf.custom_fs_attrs_id_seq'::regclass);
ALTER TABLE sf.custom_fs_attrs_current ALTER COLUMN id SET DEFAULT nextval('sf.custom_fs_attrs_id_seq'::regclass);
ALTER TABLE sf.custom_fs_attrs_history ALTER COLUMN id SET DEFAULT nextval('sf.custom_fs_attrs_id_seq'::regclass);

ALTER TABLE sf.dir_current ALTER COLUMN history_id SET DEFAULT nextval('sf.history_object_id_seq'::regclass);
ALTER TABLE sf.dir_current ALTER COLUMN out_of_sync_time SET DEFAULT NULL::timestamp with time zone;
ALTER TABLE sf.dir_history ALTER COLUMN out_of_sync_time SET DEFAULT NULL::timestamp with time zone;
ALTER TABLE sf.file_current ALTER COLUMN history_id SET DEFAULT nextval('sf.history_object_id_seq'::regclass);
ALTER TABLE sf.job_name ALTER COLUMN id SET DEFAULT nextval('sf.job_name_id_seq'::regclass);
ALTER TABLE sf.tag_name ALTER COLUMN id SET DEFAULT nextval('sf.tag_name_id_seq'::regclass);
ALTER TABLE sf.tag_namespace ALTER COLUMN id SET DEFAULT nextval('sf.tag_namespace_id_seq'::regclass);

ALTER TABLE sf.cifs_sid_mapping
    ADD CONSTRAINT cifs_sid_mapping_pkey PRIMARY KEY (cifs_sid);

ALTER TABLE sf.custom_fs_attrs_current
    ADD CONSTRAINT custom_fs_attrs_current_pkey PRIMARY KEY (volume_id, id);

ALTER TABLE sf.custom_fs_attrs_history
    ADD CONSTRAINT custom_fs_attrs_history_pkey PRIMARY KEY (volume_id, id);

ALTER TABLE sf.dir_current
    ADD CONSTRAINT dir_current_pk PRIMARY KEY (id, volume_id);

ALTER TABLE sf.dir_history
    ADD CONSTRAINT dir_history_pk PRIMARY KEY (history_id, volume_id);

ALTER TABLE sf.dir_metadata
    ADD CONSTRAINT dir_metadata_dir_id_unq UNIQUE (dir_id);

ALTER TABLE sf.file_current
    ADD CONSTRAINT file_current_pk PRIMARY KEY (id, volume_id);

ALTER TABLE sf.file_current
    ADD CONSTRAINT file_current_volume_id_parent_id_name_unq UNIQUE (volume_id, parent_id, name);

ALTER TABLE sf.file_history
    ADD CONSTRAINT file_history_pk PRIMARY KEY (history_id, volume_id);

ALTER TABLE sf.gid_mapping
    ADD CONSTRAINT gid_mapping_pkey PRIMARY KEY (volume_id, gid);

ALTER TABLE sf.gid_to_uid_mapping
    ADD CONSTRAINT gid_to_uid_mapping_pkey PRIMARY KEY (volume_id, gid, uid);

ALTER TABLE sf.job_name
    ADD CONSTRAINT job_name_name_unique UNIQUE (name);

ALTER TABLE sf.job_name
    ADD CONSTRAINT job_name_pkey PRIMARY KEY (id);

ALTER TABLE sf.job_result_current
    ADD CONSTRAINT job_result_current_pk PRIMARY KEY (volume_id, id);

ALTER TABLE sf.job_result_current
    ADD CONSTRAINT job_result_current_volume_id_fs_entry_id_name_id_unq UNIQUE (volume_id, fs_entry_id, name_id);

ALTER TABLE sf.job_result_history
    ADD CONSTRAINT job_result_history_pk PRIMARY KEY (volume_id, id);

ALTER TABLE sf.dir_metadata
    ADD CONSTRAINT scan_metadata_pk PRIMARY KEY (id);

ALTER TABLE sf.tag_name
    ADD CONSTRAINT tag_name_name_and_namespace_id_unique UNIQUE (name, namespace_id);

ALTER TABLE sf.tag_name
    ADD CONSTRAINT tag_name_pkey PRIMARY KEY (id);

ALTER TABLE sf.tag_namespace
    ADD CONSTRAINT tag_namespace_name_unique UNIQUE (name);

ALTER TABLE sf.tag_namespace
    ADD CONSTRAINT tag_namespace_pkey PRIMARY KEY (id);

ALTER TABLE sf.tag_value_current
    ADD CONSTRAINT tag_value_current_pkey PRIMARY KEY (volume_id, fs_entry_id, name_id);

ALTER TABLE sf.tag_value_history
    ADD CONSTRAINT tag_value_history_pkey PRIMARY KEY (volume_id, fs_entry_id, name_id);

ALTER TABLE sf.uid_mapping
    ADD CONSTRAINT uid_mapping_pkey PRIMARY KEY (volume_id, uid);

ALTER TABLE sf.zone_namespace
    ADD CONSTRAINT zone_namespace_pkey PRIMARY KEY (zone_id, namespace_id);


CREATE OR REPLACE FUNCTION sf.get_extension(basename text)
RETURNS text AS
$$
    -- reimplementation of get_extension (from sfutils/extension.py):
    -- - extension doesn't start with '.'
    -- - ignore first '.' if basename starts with '.'
    -- - convert to lowercase
    SELECT CASE
        WHEN position('.' in reverse(basename)) = length(basename) THEN '' -- the only dot is at beginning
        ELSE LOWER(RIGHT(basename, - length(basename) + position('.' in reverse(basename)) - 1))
    END
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf.get_extension_no_lower(basename text)
RETURNS text AS
$$
    -- reimplementation of get_extension (from sfutils/extension.py):
    -- - extension doesn't start with '.'
    -- - ignore first '.' if basename starts with '.'
    -- - convert to lowercase
    SELECT CASE
        WHEN position('.' in reverse(basename)) = length(basename) THEN '' -- the only dot is at beginning
        ELSE RIGHT(basename, - length(basename) + position('.' in reverse(basename)) - 1)
    END
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


CREATE UNIQUE INDEX calculated_aggrs_id ON sf.calculated_aggrs USING btree (volume_id, id);
CREATE INDEX custom_fs_attrs_current_attrs_idx ON sf.custom_fs_attrs_current USING gin (attrs jsonb_path_ops);
CREATE INDEX custom_fs_attrs_history_attrs_idx ON sf.custom_fs_attrs_history USING gin (attrs jsonb_path_ops);
CREATE INDEX dir_current_atime_ix ON sf.dir_current USING btree (atime);
CREATE INDEX dir_current_ctime_ix ON sf.dir_current USING btree (ctime);
CREATE INDEX dir_current_custom_fs_attrs_id_idx ON sf.dir_current USING btree (custom_fs_attrs_id) WHERE (custom_fs_attrs_id IS NOT NULL);
CREATE INDEX dir_current_depth_idx ON sf.dir_current USING btree (depth);
CREATE INDEX dir_current_errors_gin_idx ON sf.dir_current USING gin (errors jsonb_path_ops) WHERE (errors IS NOT NULL);
CREATE INDEX dir_current_gid_ix ON sf.dir_current USING btree (gid);
CREATE INDEX dir_current_lowercase_name_ix ON sf.dir_current USING btree (lower(name) text_pattern_ops);
CREATE INDEX dir_current_mtime_ix ON sf.dir_current USING btree (mtime);
CREATE INDEX dir_current_out_of_sync_time_idx ON sf.dir_current USING btree (out_of_sync_time) WHERE (out_of_sync_time IS NOT NULL);
CREATE INDEX dir_current_parentid_idx ON sf.dir_current USING btree (parent_id);
CREATE INDEX dir_current_perms_idx ON sf.dir_current USING btree (perms);
CREATE INDEX dir_current_uid_ix ON sf.dir_current USING btree (uid);
CREATE INDEX dir_current_valid_lower_idx ON sf.dir_current USING btree (lower(valid));
CREATE INDEX dir_current_valid_upper_idx ON sf.dir_current USING btree (upper(valid));
CREATE INDEX dir_current_ancestor_ids_idx on sf.dir_current USING GIN (ancestor_ids);
CREATE UNIQUE INDEX dir_current_volume_id_path_unq_idx ON sf.dir_current USING btree (volume_id, path text_pattern_ops);
CREATE INDEX dir_history_custom_fs_attrs_id_idx ON sf.dir_history USING btree (custom_fs_attrs_id) WHERE (custom_fs_attrs_id IS NOT NULL);
CREATE INDEX dir_history_depth_idx ON sf.dir_history USING btree (depth);
CREATE INDEX dir_history_id_idx ON sf.dir_history USING btree (id);
CREATE INDEX dir_history_lowercase_name_idx ON sf.dir_history USING btree (lower(name) text_pattern_ops);
CREATE INDEX dir_history_perms_idx ON sf.dir_history USING btree (perms) WHERE ((perms)::integer <> ALL (ARRAY[511, 509, 493, 448, 1533, 1517, 1528]));
CREATE INDEX dir_history_valid_lower_idx ON sf.dir_history USING btree (lower(valid));
CREATE INDEX dir_history_valid_upper_idx ON sf.dir_history USING btree (upper(valid));
CREATE INDEX dir_history_volume_id_path_idx ON sf.dir_history USING btree (volume_id, path text_pattern_ops);
CREATE INDEX dir_history_ancestor_ids_idx on sf.dir_history USING GIN (ancestor_ids);
CREATE INDEX dirs_with_recently_changed_recaggrs_volume_id_idx ON sf.dirs_with_recently_changed_recaggrs USING btree (volume_id);
CREATE INDEX file_current_atime_idx ON sf.file_current USING btree (atime);
CREATE INDEX file_current_ctime_idx ON sf.file_current USING btree (ctime);
CREATE INDEX file_current_custom_fs_attrs_id_idx ON sf.file_current USING btree (custom_fs_attrs_id) WHERE (custom_fs_attrs_id IS NOT NULL);
CREATE INDEX file_current_extension_idx ON sf.file_current USING btree (sf.get_extension(name));
CREATE INDEX file_current_gid_idx ON sf.file_current USING btree (gid);
CREATE INDEX file_current_inode_idx ON sf.file_current USING btree (inode);
CREATE INDEX file_current_lowercase_name_idx ON sf.file_current USING btree (lower(name) text_pattern_ops);
CREATE INDEX file_current_mtime_idx ON sf.file_current USING btree (mtime);
CREATE INDEX file_current_nlinks_idx ON sf.file_current USING btree (nlinks) WHERE ((nlinks)::bigint > 1);
CREATE INDEX file_current_parentid_idx ON sf.file_current USING btree (parent_id);
CREATE INDEX file_current_perms_idx ON sf.file_current USING btree (perms);
CREATE INDEX file_current_size_idx ON sf.file_current USING btree (size);
CREATE INDEX file_current_type_idx ON sf.file_current USING btree (type) WHERE ((type)::integer <> 32768);
CREATE INDEX file_current_uid_idx ON sf.file_current USING btree (uid);
CREATE INDEX file_current_valid_lower_idx ON sf.file_current USING btree (lower(valid));
CREATE INDEX file_current_valid_upper_idx ON sf.file_current USING btree (upper(valid));
CREATE INDEX file_history_custom_fs_attrs_id_idx ON sf.file_history USING btree (custom_fs_attrs_id) WHERE (custom_fs_attrs_id IS NOT NULL);
CREATE INDEX file_history_id_idx ON sf.file_history USING btree (id);
CREATE INDEX file_history_nlinks_idx ON sf.file_history USING btree (nlinks) WHERE ((nlinks)::bigint > 1);
CREATE INDEX file_history_parentid_idx ON sf.file_history USING btree (parent_id);
CREATE INDEX file_history_perms_idx ON sf.file_history USING btree (perms) WHERE ((perms)::integer <> ALL (ARRAY[420, 436, 384, 511]));
CREATE INDEX file_history_type_idx ON sf.file_history USING btree (type) WHERE ((type)::integer <> 32768);
CREATE INDEX file_history_valid_lower_idx ON sf.file_history USING btree (lower(valid));
CREATE INDEX file_history_valid_upper_idx ON sf.file_history USING btree (upper(valid));
CREATE INDEX file_history_volumeid_idx ON sf.file_history USING btree (volume_id);
CREATE INDEX gid_mapping_lower_groupname ON sf.gid_mapping USING btree (lower(name));
CREATE INDEX gid_to_uid_mapping_idx ON sf.gid_to_uid_mapping USING btree (gid);
CREATE INDEX job_name_name ON sf.job_name USING btree (name text_pattern_ops);
CREATE INDEX job_result_current_job_id_idx ON sf.job_result_current USING btree (job_id) WHERE (job_id IS NOT NULL);
CREATE INDEX job_result_current_name_id_idx ON sf.job_result_current USING btree (name_id);
CREATE INDEX job_result_current_result_idx ON sf.job_result_current USING gin (result jsonb_path_ops);
CREATE INDEX job_result_history_job_id_idx ON sf.job_result_history USING btree (job_id) WHERE (job_id IS NOT NULL);
CREATE INDEX job_result_history_name_id_idx ON sf.job_result_history USING btree (name_id);
CREATE INDEX job_result_history_result_idx ON sf.job_result_history USING gin (result jsonb_path_ops);
CREATE INDEX job_result_history_volume_id_fs_entry_id_name_id_idx ON sf.job_result_history USING btree (volume_id, fs_entry_id, name_id);
CREATE UNIQUE INDEX sid_to_gid_index ON sf.gid_mapping USING btree (volume_id, sid);
CREATE UNIQUE INDEX sid_to_uid_index ON sf.uid_mapping USING btree (volume_id, sid);
CREATE INDEX tag_name_name ON sf.tag_name USING btree (name text_pattern_ops);
CREATE INDEX tag_value_current_name_id_idx ON sf.tag_value_current USING btree (name_id);
CREATE INDEX tag_value_current_volume_id_name_id_idx ON sf.tag_value_current USING btree (volume_id, name_id);
CREATE INDEX tag_value_history_name_id_idx ON sf.tag_value_history USING btree (name_id);
CREATE INDEX tag_value_history_volume_id_name_id_idx ON sf.tag_value_history USING btree (volume_id, name_id);
CREATE INDEX uid_mapping_lower_username ON sf.uid_mapping USING btree (lower(name));
CREATE INDEX uid_to_gid_mapping_idx ON sf.gid_to_uid_mapping USING btree (uid, volume_id);
ALTER TABLE sf.job_result_current
    ADD CONSTRAINT job_result_current_name_id_fk FOREIGN KEY (name_id) REFERENCES sf.job_name(id) ON DELETE CASCADE;
ALTER TABLE sf.job_result_history
    ADD CONSTRAINT job_result_history_name_id_fk FOREIGN KEY (name_id) REFERENCES sf.job_name(id) ON DELETE CASCADE;
ALTER TABLE sf.dir_metadata
    ADD CONSTRAINT dir_metadata_volumeid_parentid_fk FOREIGN KEY (volume_id, dir_id) REFERENCES sf.dir_current(volume_id, id) ON DELETE CASCADE;
ALTER TABLE sf.tag_name
    ADD CONSTRAINT tag_name_namespace_id_fk FOREIGN KEY (namespace_id) REFERENCES sf.tag_namespace(id) ON DELETE CASCADE;
ALTER TABLE sf.tag_value_history
    ADD CONSTRAINT tag_value_name_id_fk FOREIGN KEY (name_id) REFERENCES sf.tag_name(id) ON DELETE CASCADE;
ALTER TABLE sf.tag_value_current
    ADD CONSTRAINT tag_value_name_id_fk FOREIGN KEY (name_id) REFERENCES sf.tag_name(id) ON DELETE CASCADE;
ALTER TABLE sf.zone_namespace
    ADD CONSTRAINT zone_namespace_namespace_id_fkey FOREIGN KEY (namespace_id) REFERENCES sf.tag_namespace(id) ON DELETE CASCADE;

ALTER TABLE sf_archive.restore_job ADD COLUMN requested_by VARCHAR; -- from migration 0097

ALTER TABLE sf_volumes.volume DROP COLUMN check_noatime; -- from migration 0098

CREATE TABLE use_partitioning (  -- from migration 0101
    value BOOLEAN NOT NULL
);
