/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */

/***********************************************************************************************************************************
RAW Functions
***********************************************************************************************************************************/

DROP TYPE IF EXISTS sf_internal.load_events_ret_type;
CREATE TYPE sf_internal.load_events_ret_type AS (events_processed BIGINT, events_ignored BIGINT, events_failed BIGINT);

CREATE OR REPLACE FUNCTION generate_sfid(
    volume_id BIGINT,
    fs_entry_id BIGINT,
    history_id BIGINT
    )
RETURNS TEXT as
$$
DECLARE
	_sep    TEXT;
	_version   TEXT;
	_sfid_result TEXT;
BEGIN
    IF (volume_id > 0 and fs_entry_id > 0 and history_id > 0) THEN
        _sep = ':';
        _version = 'sfid';
        _sfid_result = _version || _sep || volume_id || _sep || fs_entry_id || _sep || history_id;
        RETURN _sfid_result;
    ELSE
        RAISE EXCEPTION 'volume_id, fs_entry_id and history id should all be integer greater than 0';
    END IF;
END;
$$ LANGUAGE plpgsql STRICT IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION urlencode(
    in_str varchar)
RETURNS varchar as
$$
DECLARE
   _i      int4;
   _temp   varchar;
   _hex    varchar;
   _ascii  int4;
   _result varchar;
BEGIN
   _result = '';
   FOR _i IN 1 .. length(in_str) LOOP
       _temp := substr(in_str, _i, 1);
       IF _temp ~ '[0-9a-zA-Z:/@._?#-]+' THEN
           _result := _result || _temp;
       ELSE
           _hex := encode(_temp::bytea, 'hex');
           _temp := '';
           WHILE LENGTH(_hex) > 0 LOOP
               _temp := _temp || '%' || substring(_hex, 1, 2);
               _hex := substring(_hex, 3, 999);
           END LOOP;
           _result := _result || upper(_temp);
       END IF;
   END LOOP;
   RETURN _result;
END;
$$ LANGUAGE plpgsql STRICT IMMUTABLE PARALLEL SAFE;



/***********************************************************************************************************************************
Correctly escapes LIKE operator special characters
***********************************************************************************************************************************/

CREATE OR REPLACE FUNCTION escape_like(
    text varchar
)
RETURNS varchar as
$$
BEGIN
    RETURN regexp_replace(text, '(%|_|\\)', '\\\1', 'g');
END
$$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;



/***********************************************************************************************************************************
RETURNS patterns to filter subdirs for a given path
***********************************************************************************************************************************/

CREATE OR REPLACE FUNCTION subtree_pattern(
    root_path varchar
)
RETURNS varchar as
$$
BEGIN
    -- this function does not include root
    -- _% - anything with at least 1 character (to not include volume root)
    RETURN escape_like(root_path) || (CASE WHEN root_path = '' THEN '_%' ELSE '/%' END);
END
$$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION tstzrange_contains(
    range tstzrange,
    ts timestamptz)
RETURNS bool AS
$$
-- The condition is written in a not obvious way to increase chances that PostgreSQL
-- will use indexes on lower(range), and upper(range) in a most efficient way.
SELECT (ts >= lower(range) AND (lower_inc(range) OR ts > lower(range)))
   AND (ts <= upper(range) AND (upper_inc(range) OR ts < upper(range)))
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION tstzrange_intersects_with_bounds(
    range tstzrange,
    lower_bound timestamptz,
    upper_bound timestamptz)
RETURNS bool AS
$$
-- The condition is written in a not obvious way to increase chances that PostgreSQL
-- will use indexes on lower(range), and upper(range) in a most efficient way.
SELECT (upper(range) <= upper_bound AND upper(range) >= lower_bound)
   OR  (lower(range) >= lower_bound AND lower(range) <= upper_bound)
   OR  (lower(range) <= lower_bound AND upper(range) >= upper_bound)
   OR  (lower(range) >= lower_bound AND upper(range) <= upper_bound)
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION tstzrange_contains_upper(
    range tstzrange,
    upper_bound timestamptz)
RETURNS bool AS
$$
-- This assumes that upper_bound is upper of some tstzrange with [)
-- and that range is also [) range.
SELECT upper_bound > lower(range) AND upper_bound <= upper(range)
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


/***********************************************************************************************************************************
insert_stubs_for_missing_path_tokens
NOT THREAD SAFE!
***********************************************************************************************************************************/
CREATE OR REPLACE FUNCTION sf_internal.insert_stubs_for_missing_path_tokens(
    vol_id BIGINT,
    entry_path TEXT,
    stub_sync_time sf.time_t
) RETURNS BIGINT AS $$
DECLARE
    path_tokens TEXT[];
    path_tokens_len BIGINT;
    existing_ancestor_idx BIGINT;
    existing_ancestor sf.dir_current;
    curr_path TEXT;
    idx BIGINT;
    stub_id BIGINT;
    ancestor_ids BIGINT[];
    inserted_stubs_cnt BIGINT = 0;
BEGIN
    IF stub_sync_time IS NULL THEN
        RAISE EXCEPTION 'Can not add a stub with null stub_sync_time: '
                        'entry_path: %, vol_id: %, stub_sync_time: %',
                            entry_path, vol_id, stub_sync_time;
    END IF;
    IF entry_path = '' THEN
        path_tokens = ARRAY[]::TEXT[];
        path_tokens_len = 0;
    ELSE
        path_tokens = regexp_split_to_array(entry_path, '/');
        path_tokens_len = array_length(path_tokens, 1);
    END IF;
    FOR idx IN REVERSE path_tokens_len .. 0 LOOP
        curr_path = array_to_string(path_tokens[1:idx], '/');
        SELECT * INTO existing_ancestor FROM sf.dir_current WHERE path = curr_path AND volume_id = vol_id;
        IF FOUND THEN
            existing_ancestor_idx = idx;
            EXIT;
        END IF;
    END LOOP;
    IF existing_ancestor IS NULL THEN
        stub_id = nextval('sf.object_id_seq');
        ancestor_ids = ARRAY[stub_id]::BIGINT[];
        -- Set OOS Marker which is: oos_time=sync_time+1
        INSERT INTO sf.dir_current (id, volume_id, parent_id, ancestor_ids, path, name, depth, valid, sync_time, out_of_sync_time,
                                    rec_aggrs, local_aggrs)
            VALUES (stub_id, vol_id, NULL, ancestor_ids, '', '', 0, tstzrange(stub_sync_time, 'infinity', '[)'), stub_sync_time,
                    stub_sync_time + INTERVAL '1 second', '{}', '{}')
            RETURNING * INTO existing_ancestor;
        existing_ancestor_idx = 0;
        inserted_stubs_cnt = inserted_stubs_cnt + 1;
    END IF;
    FOR idx IN existing_ancestor_idx + 1 .. path_tokens_len LOOP
        stub_id = nextval('sf.object_id_seq');
        curr_path = array_to_string(path_tokens[1:idx], '/');
        ancestor_ids = existing_ancestor.ancestor_ids || stub_id;
        INSERT INTO sf.dir_current (id, volume_id, parent_id, ancestor_ids, path, name, depth, valid, sync_time, out_of_sync_time,
                                    rec_aggrs, local_aggrs)
            VALUES (stub_id, vol_id, existing_ancestor.id, ancestor_ids, curr_path, path_tokens[idx], existing_ancestor.depth + 1,
                    tstzrange(stub_sync_time, 'infinity', '[)'), stub_sync_time, stub_sync_time + INTERVAL '1 second' , '{}', '{}')
            RETURNING * INTO existing_ancestor;
        inserted_stubs_cnt = inserted_stubs_cnt + 1;
    END LOOP;

    RETURN inserted_stubs_cnt;
END;
$$ LANGUAGE plpgsql VOLATILE PARALLEL UNSAFE;


/***********************************************************************************************************************************
Fill missing parents
NOT THREAD SAFE!
***********************************************************************************************************************************/

CREATE OR REPLACE FUNCTION sf_internal.fill_missing_parents(
    vol_id BIGINT
) RETURNS BIGINT AS $$
DECLARE
    missing_parent_dir record;
    missing_paths BIGINT = 0;
    inserted_stubs BIGINT;
BEGIN

    FOR missing_parent_dir IN
        SELECT * FROM fill_missing_parents_parent_missing_parent
    LOOP
        SELECT sf_internal.insert_stubs_for_missing_path_tokens(vol_id, missing_parent_dir.path,
                                                                missing_parent_dir.min_sync_time)
            INTO inserted_stubs;
        missing_paths = missing_paths + inserted_stubs;
    END LOOP;

    RETURN missing_paths;
END;
$$ LANGUAGE plpgsql VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.update_custom_fs_attrs(
    customFsAttrId BIGINT,
    volumeId BIGINT,
    newCustomFsAttrs JSONB,
    bHistory BOOLEAN
) RETURNS BIGINT AS $$
DECLARE
    rCurrentCustomFsAttrs sf.custom_fs_attrs_current;
    ret BIGINT;
BEGIN
    IF newCustomFsAttrs IS NULL THEN
        RETURN customFsAttrId;
    END IF;

    SELECT * INTO rCurrentCustomFsAttrs FROM sf.custom_fs_attrs_current WHERE volume_id = volumeId AND id = customFsAttrId;
    IF newCustomFsAttrs = rCurrentCustomFsAttrs.attrs THEN
        RETURN rCurrentCustomFsAttrs.id;
    END IF;
    IF rCurrentCustomFsAttrs IS NOT NULL THEN
        IF bHistory THEN
            INSERT INTO sf.custom_fs_attrs_history(volume_id, id, attrs)
                VALUES (rCurrentCustomFsAttrs.volume_id, rCurrentCustomFsAttrs.id, rCurrentCustomFsAttrs.attrs);
        END IF;
        DELETE FROM sf.custom_fs_attrs_current WHERE volume_id = volumeId AND id = customFsAttrId;
    END IF;
    IF newCustomFsAttrs IS NOT NULL AND newCustomFsAttrs != '{}'::JSONB THEN
        INSERT INTO sf.custom_fs_attrs_current(volume_id, attrs) VALUES (volumeId, newCustomFsAttrs) RETURNING id INTO ret;
        RETURN ret;
    END IF;
    RETURN null;
END;
$$ LANGUAGE plpgsql VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.insert_file_into_history(
    file sf.file_current,
    calculated_sync_time timestamptz
) RETURNS void AS $$
BEGIN
    file.valid = tstzrange(LOWER(file.valid), calculated_sync_time, '[)');
    -- COALESCE is needed here because history_id column was added to current tables in migration 0012 and can be NULL
    file.history_id = COALESCE(file.history_id, nextval('sf.history_object_id_seq'));
    INSERT INTO sf.file_history VALUES (file.*);
END;
$$ LANGUAGE plpgsql VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.insert_dir_into_history(
    dir sf.dir_current,
    calculated_sync_time timestamptz
) RETURNS void AS $$
BEGIN
    dir.valid = tstzrange(LOWER(dir.valid), calculated_sync_time, '[)');
    -- COALESCE is needed here because history_id column was added to current tables in migration 0012 and can be NULL
    dir.history_id = COALESCE(dir.history_id, nextval('sf.history_object_id_seq'));
    INSERT INTO sf.dir_history VALUES (dir.*);
END;
$$ LANGUAGE plpgsql VOLATILE PARALLEL UNSAFE;

CREATE OR REPLACE FUNCTION sf_internal.is_historically_significant_custom_fs_attrs_change(
    vol_id BIGINT,
    old_custom_fs_attrs_id BIGINT,
    new_custom_fs_attrs JSONB
) RETURNS BOOLEAN AS $$
DECLARE
    old_custom_fs_attrs JSONB;
BEGIN
    IF (new_custom_fs_attrs IS NOT NULL) THEN
        -- load old attrs
        IF (old_custom_fs_attrs_id IS NOT NULL) THEN
            SELECT attrs INTO old_custom_fs_attrs FROM sf.custom_fs_attrs_current WHERE volume_id = vol_id AND id = old_custom_fs_attrs_id;
            -- old_custom_fs_attrs can still be NULL if custom_fs_attrs has been manually removed from DB to save space
        END IF;

        IF (new_custom_fs_attrs = '{}'::JSONB) THEN
            -- This optimisation do not store empty '{}' attrs in database.
            -- No custom_fs_attrs row is treated as empty dict.
            IF (old_custom_fs_attrs IS NOT NULL) THEN
                -- Old version will have to be deleted
                RETURN TRUE;
            END IF;
        ELSE
            IF (old_custom_fs_attrs IS NULL OR old_custom_fs_attrs != new_custom_fs_attrs) THEN
                -- Old version will have to be changed to new one
                RETURN TRUE;
            END IF;
        END IF;
    END IF;
    RETURN FALSE;
END;
$$ LANGUAGE plpgsql STABLE PARALLEL SAFE;

CREATE OR REPLACE FUNCTION sf_internal.is_historically_significant_change(
    current_row sf.dir_current,
    new_custom_fs_attrs JSONB,
    new_row sf.dir_current
) RETURNS BOOLEAN AS $$
DECLARE
    tmp sf.dir_current;
BEGIN
    IF (sf_internal.is_historically_significant_custom_fs_attrs_change(current_row.volume_id, current_row.custom_fs_attrs_id, new_custom_fs_attrs)) THEN
        RETURN TRUE;
    END IF;

    tmp = current_row;
    tmp.atime = new_row.atime;
    -- Block count is ignored, because on Isilon it often changes without any other metadata
    tmp.blocks = new_row.blocks;
    -- Do not compare fields which are updated by sf_internal.process_event
    tmp.history_id = new_row.history_id;
    tmp.custom_fs_attrs_id = new_row.custom_fs_attrs_id;
    tmp.valid = new_row.valid;
    tmp.sync_time = new_row.sync_time;
    tmp.out_of_sync_time = new_row.out_of_sync_time;
    -- changes in entry errors should not result in new version of entry
    tmp.errors = new_row.errors;

    RETURN tmp.* IS DISTINCT FROM new_row.*;
END;
$$ LANGUAGE plpgsql STABLE PARALLEL SAFE;

CREATE OR REPLACE FUNCTION sf_internal.is_historically_significant_change(
    current_row sf.file_current,
    new_custom_fs_attrs JSONB,
    new_row sf.file_current
) RETURNS BOOLEAN AS $$
DECLARE
    tmp sf.file_current;
BEGIN
    IF (sf_internal.is_historically_significant_custom_fs_attrs_change(current_row.volume_id, current_row.custom_fs_attrs_id, new_custom_fs_attrs)) THEN
        RETURN TRUE;
    END IF;

    tmp = current_row;
    tmp.atime = new_row.atime;
    -- Block count is ignored, because on Isilon it often changes without any other metadata
    tmp.blocks = new_row.blocks;
    tmp.nlinks = new_row.nlinks;
    -- Do not compare fields which are updated by sf_internal.process_event
    tmp.history_id = new_row.history_id;
    tmp.custom_fs_attrs_id = new_row.custom_fs_attrs_id;
    tmp.valid = new_row.valid;
    tmp.sync_time = new_row.sync_time;

    RETURN tmp.* IS DISTINCT FROM new_row.*;
END;
$$ LANGUAGE plpgsql STABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf_internal.can_update_oos_time(
    current_row sf.dir_current,
    event sf_internal.event_base,
    calculated_sync_time timestamptz,
    oos_accuracy INTERVAL
) RETURNS BOOLEAN AS $$
DECLARE
BEGIN
    IF (event.out_of_sync_time IS NULL) THEN
        -- can nullify oos_time?
        IF (current_row.out_of_sync_time IS NOT NULL) THEN

            -- Condition `out_of_sync_time = sync_time + 1` checks if this row was created, because of
            -- fill_missing_parents. So, if it is:
            -- * True: in this case it is resonable to nullify oos, because stub is currently being overwritten
            --         with correct data.
            -- * False: this row might have been created by event monitor; so, it is not safe to nullify oos;
            --          we might forget to make stat in future crawl and do not update recently changed metadata.
            IF (current_row.sync_time IS NOT NULL) THEN
                -- Check if oos marker is already set on a row -In that case lets nullify oos
                IF (current_row.out_of_sync_time = current_row.sync_time + INTERVAL '1 second' ) THEN
                    RETURN TRUE;
                END IF;
            END IF;

            RETURN current_row.out_of_sync_time + oos_accuracy < calculated_sync_time;
        ELSE
            RETURN FALSE;
        END IF;
    END IF;

    IF (event.out_of_sync_time IS NOT NULL) THEN
        IF (event.out_of_sync_time + oos_accuracy < COALESCE(event.sync_time, current_row.sync_time)) THEN
            -- should not set to stale value
            RETURN FALSE;
        END IF;

        IF (current_row.out_of_sync_time IS NOT NULL) THEN
            IF (current_row.out_of_sync_time > event.out_of_sync_time) THEN
                -- should not decrease oos_time?
                RETURN FALSE;
            END IF;
        END IF;
        RETURN TRUE;
    END IF;

END;
$$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf_internal.carries_file_attr_changes(
    event sf_internal.event_base
) RETURNS BOOLEAN AS $$
DECLARE
    tmp sf_internal.event_base;
    empty_event sf_internal.event_base;
BEGIN
    empty_event.seq_num = NULL;

    tmp = event;
    -- required fields in event
    tmp.seq_num = NULL;
    tmp.volume_id = NULL;
    tmp.event_type = NULL;
    tmp.type = NULL;
    -- fields not related to file/dir attrs
    tmp.sync_time = NULL;
    tmp.out_of_sync_time = NULL;
    tmp.tree_out_of_sync_time = NULL;
    tmp.depth = NULL;
    tmp.path = NULL;
    tmp.name = NULL;
    tmp.parent_path = NULL;
    tmp.new_full_path = NULL;
    tmp.errors = NULL;

    RETURN tmp IS DISTINCT FROM empty_event;
END;
$$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf_internal.set_tree_out_of_sync_time(
    directory_id BIGINT,
    vol_id BIGINT,
    new_tree_out_of_sync_time timestamptz
) RETURNS BOOLEAN AS $$
DECLARE
    dirMetadata sf.dir_metadata;
BEGIN
    IF new_tree_out_of_sync_time IS NULL THEN
        RAISE EXCEPTION 'Setting tree_out_of_sync_time to NULL is not supported; dir_id: %', directory_id;
    END IF;
    INSERT INTO sf.dir_metadata(dir_id, volume_id, exclude_dirs, exclude_files, tree_out_of_sync_time)
        VALUES (directory_id, vol_id, array[]::text[], array[]::text[], new_tree_out_of_sync_time)
        ON CONFLICT (dir_id, volume_id) DO UPDATE
            SET tree_out_of_sync_time = new_tree_out_of_sync_time
            WHERE sf.dir_metadata.dir_id = directory_id
                AND sf.dir_metadata.tree_out_of_sync_time < new_tree_out_of_sync_time
                OR sf.dir_metadata.tree_out_of_sync_time IS NULL;
    RETURN FOUND;
END;
$$ LANGUAGE plpgsql VOLATILE PARALLEL UNSAFE;


/***********************************************************************************************************************************
    Handling MOVED events
***********************************************************************************************************************************/
CREATE OR REPLACE FUNCTION sf_internal.process_move_file_event
(
    vol_id BIGINT,                    -- volume id
    bHistory BOOLEAN,                 -- track history
    event_row sf_internal.event_base, -- row from event table
    src_parent sf.dir_current,        -- old parent
    dst_parent sf.dir_current,        -- new parent
    dst_name TEXT                     -- new basename
) RETURNS boolean AS $$
DECLARE
    file sf.file_current;             -- file being moved
    overwritten_file sf.file_current; -- file being overwritten
BEGIN

    -- remove overwritten file
    -- we do not check dir existence here, as a file can overwrite only another file
    -- STAR-3726: how to react when we get impossible move of a file onto a (possible nonempty) dir?

    SELECT * INTO overwritten_file
        FROM sf.file_current WHERE
            file_current.parent_id = dst_parent.id
            AND file_current.volume_id = vol_id
            AND file_current.name = dst_name;

    IF FOUND THEN

        IF bHistory THEN
            PERFORM sf_internal.insert_file_into_history(overwritten_file, event_row.sync_time);
        END IF;

        DELETE FROM sf.file_current
            WHERE id = overwritten_file.id AND volume_id = overwritten_file.volume_id;

        EXECUTE 'SELECT sf_internal.remove_from_extra_tables_vol_' || overwritten_file.volume_id || '($1, $2, $3)'
        USING ARRAY[overwritten_file.id]::BIGINT[], ARRAY[overwritten_file.custom_fs_attrs_id]::BIGINT[], bHistory;

    END IF;

    -- perform actual move

    SELECT * INTO file
        FROM sf.file_current WHERE
            file_current.parent_id = src_parent.id
            AND file_current.volume_id = vol_id
            AND file_current.name = event_row.name;

    IF not FOUND THEN
        RETURN FALSE;
    END IF;

    IF bHistory THEN
        PERFORM sf_internal.insert_file_into_history(file, event_row.sync_time);
    END IF;
    file.history_id = nextval('sf.history_object_id_seq');
    file.sync_time = event_row.sync_time;
    file.valid = tstzrange(event_row.sync_time, 'infinity', '[)');

    UPDATE sf.file_current
       SET valid = file.valid,
           sync_time = file.sync_time,
           history_id = file.history_id,
           parent_id = dst_parent.id,
           name = dst_name
     WHERE id = file.id AND volume_id = file.volume_id;

    RETURN TRUE;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;



CREATE OR REPLACE FUNCTION sf_internal.process_move_dir_event
(
    vol_id BIGINT,                    -- volume id
    bHistory BOOLEAN,                 -- track history
    event_row sf_internal.event_base, -- row from event table
    src_parent sf.dir_current,        -- old parent
    dst_parent sf.dir_current,        -- new parent
    dst_name TEXT                     -- new basename
) RETURNS boolean AS $$
DECLARE
    src_dir sf.dir_current;            -- dir being moved
    subdir sf.dir_current;            -- subdir being moved
    dst_path TEXT;                    -- new path
    overwritten_dir sf.dir_current;   -- file being overwritten
BEGIN

    -- get src_dir

    SELECT * INTO src_dir
        FROM sf.dir_current WHERE
            dir_current.parent_id = src_parent.id
            AND dir_current.volume_id = vol_id
            AND dir_current.name = event_row.name;

    IF not FOUND THEN
        RETURN FALSE;
    END IF;

    IF dst_parent.path = '' THEN
        dst_path = dst_name;
    ELSE
        dst_path = dst_parent.path || '/' || dst_name;
    END IF;

    -- remove overwritten empty dir
    -- we do not check file existence here not dir emptiness, as a dir can overwrite only empty dir
    -- STAR-3726: how to react when we get impossible move of a dir onto nonempty dir?

    SELECT * INTO overwritten_dir
        FROM sf.dir_current WHERE
            dir_current.volume_id = vol_id
            AND dir_current.path = dst_path;

    IF FOUND THEN

        IF bHistory THEN
            PERFORM sf_internal.insert_dir_into_history(overwritten_dir, event_row.sync_time);
        END IF;

        DELETE FROM sf.dir_current
            WHERE id = overwritten_dir.id AND volume_id = overwritten_dir.volume_id;

        EXECUTE 'SELECT sf_internal.remove_from_extra_tables_vol_' || overwritten_dir.volume_id || '($1, $2, $3)'
        USING ARRAY[overwritten_dir.id], ARRAY[overwritten_dir.custom_fs_attrs_id], bHistory;

    END IF;

    -- update children

    FOR subdir IN SELECT * FROM sf.dir_current WHERE
        volume_id = vol_id
        AND path LIKE subtree_pattern(src_dir.path)
    LOOP
        IF bHistory THEN
            PERFORM sf_internal.insert_dir_into_history(subdir, event_row.sync_time);
        END IF;
        subdir.history_id = nextval('sf.history_object_id_seq');
        subdir.sync_time = event_row.sync_time;
        subdir.valid = tstzrange(event_row.sync_time, 'infinity', '[)');

        UPDATE sf.dir_current
           SET valid = subdir.valid,
               sync_time = subdir.sync_time,
               history_id = subdir.history_id,
               path = overlay(subdir.path placing dst_path from 1 for char_length(src_dir.path)),
               depth = subdir.depth - src_parent.depth + dst_parent.depth,
               ancestor_ids = dst_parent.ancestor_ids || subdir.ancestor_ids[array_length(src_parent.ancestor_ids, 1) + 1:]
         WHERE id = subdir.id AND volume_id = subdir.volume_id;
    END LOOP;

    -- update src_dir

    IF bHistory THEN
        PERFORM sf_internal.insert_dir_into_history(src_dir, event_row.sync_time);
    END IF;
    src_dir.history_id = nextval('sf.history_object_id_seq');
    src_dir.sync_time = event_row.sync_time;
    src_dir.valid = tstzrange(event_row.sync_time, 'infinity', '[)');

    UPDATE sf.dir_current
       SET valid = src_dir.valid,
           sync_time = src_dir.sync_time,
           history_id = src_dir.history_id,
           parent_id = dst_parent.id,
           ancestor_ids = dst_parent.ancestor_ids || ARRAY[id] ::BIGINT[],
           name = dst_name,
           path = dst_path,
           depth = dst_parent.depth + 1
     WHERE id = src_dir.id AND volume_id = src_dir.volume_id;

    INSERT INTO sf.refresh_aggregates(id, volume_id, depth) VALUES (src_dir.id, src_dir.volume_id, dst_parent.depth + 1);

    RETURN TRUE;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;



CREATE OR REPLACE FUNCTION sf_internal.load_move_events
(
    vol_id BIGINT,    -- volume id
    bHistory BOOLEAN  -- track history
) RETURNS record AS $$
DECLARE
    events_processed BIGINT = 0;      -- number of events processed
    events_ignored BIGINT = 0;        -- number of events ignored
    event_row sf_internal.event_base; -- row from the event table
    src_parent sf.dir_current;        -- old parent
    dst_parent sf.dir_current;        -- new parent
    new_full_path_splitted TEXT[];
    dst_parent_path TEXT;
    dst_name TEXT;
    ret record;
BEGIN
    -- WARNING: this does not update target metadata (eg. ctime)

    FOR event_row IN SELECT * FROM move_events ORDER BY seq_num
    LOOP
        events_processed = events_processed + 1;

        IF event_row.path = event_row.new_full_path THEN
            events_ignored = events_ignored + 1;
            CONTINUE;
        END IF;

        new_full_path_splitted = regexp_split_to_array(event_row.new_full_path, '/');
        dst_parent_path = array_to_string(new_full_path_splitted[1:array_length(new_full_path_splitted, 1) - 1], '/');
        dst_name = new_full_path_splitted[array_length(new_full_path_splitted, 1)];

        SELECT * INTO src_parent
            FROM sf.dir_current WHERE
                dir_current.path = event_row.parent_path
                AND dir_current.volume_id = vol_id;

        IF not FOUND THEN
            events_ignored = events_ignored + 1;
            CONTINUE;
        END IF;

        SELECT * INTO dst_parent
            FROM sf.dir_current WHERE
                dir_current.path = dst_parent_path
                AND dir_current.volume_id = vol_id;

        IF not FOUND THEN
            events_ignored = events_ignored + 1;
            CONTINUE;
        END IF;

        IF event_row.type = 16384 THEN
            IF NOT sf_internal.process_move_dir_event(vol_id, bHistory, event_row, src_parent, dst_parent, dst_name) THEN
                events_ignored = events_ignored + 1;
                CONTINUE;
            END IF;
        ELSE
            IF NOT sf_internal.process_move_file_event(vol_id, bHistory, event_row, src_parent, dst_parent, dst_name) THEN
                events_ignored = events_ignored + 1;
                CONTINUE;
            END IF;
        END IF;

        INSERT INTO sf.refresh_aggregates(id, volume_id, depth) VALUES (src_parent.id, src_parent.volume_id, src_parent.depth), (dst_parent.id, dst_parent.volume_id, dst_parent.depth);

    END LOOP;

    SELECT events_processed, events_ignored INTO ret;
    RETURN ret;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.move_dirs_with_recaggrs_changed_to_history(
    move_dir_to_history_seconds BIGINT,
    crawl_end_time BIGINT,
    current_volume_id BIGINT,
    worker_num BIGINT,
    workers_count BIGINT
) RETURNS record AS $$
DECLARE
    total_processed_dirs BIGINT = 0;
    total_moved_to_history_dirs BIGINT = 0;
    dir_current_row sf.dir_current;
    crawl_end_time_tstz timestamptz;
    ret record;
BEGIN
    crawl_end_time_tstz = to_timestamp(crawl_end_time);

    -- all ids of dirs that recently has rec_aggrs updated for this volume
    CREATE TEMP TABLE dir_ids AS
            -- pop work items for this volume
            WITH volume_dirs_with_recently_changed_recaggrs AS (
                DELETE FROM sf.dirs_with_recently_changed_recaggrs d
                    WHERE d.volume_id = current_volume_id
                      AND d.fs_entry_id % workers_count = worker_num
                RETURNING *
            )
        SELECT DISTINCT fs_entry_id AS id FROM volume_dirs_with_recently_changed_recaggrs vd;
    ANALYZE dir_ids (id);

    SELECT COUNT(id) FROM dir_ids INTO total_processed_dirs;

    -- all ids of dirs that have to be stored in history for this volume
    CREATE TEMP TABLE history_change_ids AS
        SELECT dir_ids.id FROM dir_ids
            JOIN sf.dir_current dir_current
                ON dir_ids.id = dir_current.id
               AND dir_current.volume_id = current_volume_id  -- should allow partition prunning
            WHERE crawl_end_time_tstz - LOWER(dir_current.valid) > (move_dir_to_history_seconds * INTERVAL '1 seconds')
              AND dir_current.size IS NOT NULL;
    ANALYZE history_change_ids;

    SELECT COUNT(id) FROM history_change_ids INTO total_moved_to_history_dirs;

    -- copy current to history
    -- in local perf tests this takes 2 sec for 100K entries
    -- inlining insert_dir_into_history gives 4x boost here
    INSERT INTO sf.dir_history
        (id, valid, volume_id, parent_id, ancestor_ids, depth, inode, size, blocks,
         ctime, atime, mtime, uid, gid, perms, path, name, rec_aggrs, local_aggrs, history_id, sync_time,
         out_of_sync_time, errors, custom_fs_attrs_id)
        SELECT
        dir.id,
        tstzrange(LOWER(dir.valid), crawl_end_time_tstz, '[)') AS valid,
        volume_id, parent_id, ancestor_ids, depth, inode, size, blocks, ctime, atime, mtime, uid, gid, perms, path, name, rec_aggrs, local_aggrs,
        COALESCE(dir.history_id, nextval('sf.history_object_id_seq')) AS history_id,
        sync_time, out_of_sync_time, errors, custom_fs_attrs_id
    FROM sf.dir_current dir, history_change_ids hci
    WHERE dir.id = hci.id
      AND dir.volume_id = current_volume_id;

    -- update current
    -- in local perf tests this takes 4 sec for 100K entries
    UPDATE sf.dir_current d
    SET
        valid = tstzrange(crawl_end_time_tstz, 'infinity', '[)'),
        history_id = nextval('sf.history_object_id_seq')
    FROM history_change_ids
    WHERE d.id = history_change_ids.id
      AND d.volume_id = current_volume_id;

    DROP TABLE history_change_ids;
    DROP TABLE dir_ids;

    SELECT total_processed_dirs, total_moved_to_history_dirs INTO ret;
    RETURN ret;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.compare_entries(
    src_cur refcursor,
    dst_cur refcursor,
    max_mtime sf.time_t,
    hash_func TEXT)
RETURNS refcursor AS $$
-- This function is used by cmp_file_trees.py run by sfsync-and-verify.sh
DECLARE
    result_cur refcursor;
    src_row RECORD;
    dst_row RECORD;
    src_hash TEXT;
    dst_hash TEXT;
    src_full_path TEXT;
    dst_full_path TEXT;
    src_hash_mt sf.time_t;
    dst_hash_mt sf.time_t;
    synced_files BIGINT = 0;
    synced_bytes BIGINT = 0;
BEGIN
    -- RETURN NEXT stores whole result in memory which is unacceptable here as the result can be large.
    -- See Note under https://www.postgresql.org/docs/9.6/static/plpgsql-control-structures.html#AEN65768
    CREATE TEMP TABLE _compare_result(
        code TEXT NOT NULL,
        path TEXT,
        src_mtime sf.time_t,
        src_hash TEXT,
        src_hash_mtime sf.time_t,
        dst_mtime sf.time_t,
        dst_hash TEXT,
        dst_hash_mtime sf.time_t,
        synced_files BIGINT,
        synced_bytes BIGINT
    );

    FETCH src_cur INTO src_row;
    FETCH dst_cur INTO dst_row;
    -- row IS NOT NULL returns true if ANY value is NULL.
    -- To test if the row record in null we should use NOT row IS NULL
    WHILE NOT src_row IS NULL OR NOT dst_row IS NULL LOOP
        IF src_row IS NOT NULL THEN
            src_full_path = CASE WHEN src_row.parent_path = '' THEN src_row.fn ELSE src_row.parent_path || '/' || src_row.fn END;
        ELSE
            src_full_path = NULL;
        END IF;
        IF dst_row IS NOT NULL THEN
            dst_full_path = CASE WHEN dst_row.parent_path = '' THEN dst_row.fn ELSE dst_row.parent_path || '/' || dst_row.fn END;
        ELSE
            dst_full_path = NULL;
        END IF;

        IF src_row IS NULL THEN
            INSERT INTO _compare_result(code, path) VALUES ('missing-src', dst_full_path);
            FETCH dst_cur INTO dst_row;
        ELSIF dst_row IS NULL THEN
            IF (max_mtime IS NOT NULL) AND (src_row.mt > max_mtime) THEN
                INSERT INTO _compare_result(code, path, src_mtime) VALUES ('added-in-meantime', src_full_path, src_row.mt);
            ELSE
                INSERT INTO _compare_result(code, path, src_mtime) VALUES ('missing-dst', src_full_path, src_row.mt);
            END IF;
            FETCH src_cur INTO src_row;
        ELSIF src_row.fn = dst_row.fn AND src_row.parent_path = dst_row.parent_path THEN
            -- The result row of the query created in cmp_file_trees.py using FsEntriesPgApi
            -- has an array with a single element
            IF src_row.jobs->0->>'name' <> 'hash' THEN
                RAISE EXCEPTION 'Expected job name is "hash", got "%"', src_row.jobs->0->>'name';
            END IF;
            src_hash_mt = (to_timestamp((src_row.jobs->0->>'mt')::bigint))::sf.time_t;
            src_hash = (src_row.jobs->0->'result'->>hash_func)::text;
            dst_hash_mt = (to_timestamp((dst_row.jobs->0->>'mt')::bigint))::sf.time_t;
            dst_hash = (dst_row.jobs->0->'result'->>hash_func)::text;
            IF (max_mtime IS NOT NULL) AND (src_row.mt > max_mtime OR
                                            src_hash_mt > max_mtime OR
                                            dst_row.mt > max_mtime OR
                                            dst_hash_mt > max_mtime) THEN
                INSERT INTO _compare_result(code, path, src_mtime, dst_mtime)
                    VALUES ('modified-in-meantime', src_full_path, src_row.mt, dst_row.mt);
            ELSIF src_row.mt <> dst_row.mt THEN
                INSERT INTO _compare_result(code, path, src_mtime, dst_mtime)
                    VALUES ('not-synced', src_full_path, src_row.mt, dst_row.mt);
            ELSIF (src_hash IS NULL
                    OR dst_hash IS NULL
                    OR src_row.mt <> src_hash_mt
                    OR dst_row.mt <> dst_hash_mt) THEN
                INSERT INTO _compare_result(code, path, src_mtime, src_hash_mtime, dst_hash_mtime)
                    VALUES ('hash-not-computed', src_full_path, src_row.mt, src_hash_mt, dst_hash_mt);
            ELSIF src_hash <> dst_hash THEN
                INSERT INTO _compare_result(code, path, src_hash, dst_hash)
                    VALUES ('different-hashes', src_full_path, src_hash, dst_hash);
            ELSE
                synced_files = synced_files + 1;
                synced_bytes = synced_bytes + src_row.size;
            END IF;
            FETCH src_cur INTO src_row;
            FETCH dst_cur INTO dst_row;
        -- Entries are ordered by tuple (fn, parent_path) because comparing long prefixes is more expensive
        -- and fn is shorter than parent_path. If src_row is different then dst_row here we need to
        -- decide which cursor should be moved ahead and it's important to compare
        -- 'fn' first and in case it's equal compare 'parent_path'
        ELSIF src_row.fn < dst_row.fn OR (src_row.fn = dst_row.fn AND src_row.parent_path < dst_row.parent_path) THEN
            IF (max_mtime IS NOT NULL) AND (src_row.mt > max_mtime) THEN
                INSERT INTO _compare_result(code, path, src_mtime) VALUES ('added-in-meantime', src_full_path, src_row.mt);
            ELSE
                INSERT INTO _compare_result(code, path, src_mtime) VALUES ('missing-dst', src_full_path, src_row.mt);
            END IF;
            FETCH src_cur INTO src_row;
        ELSE
            INSERT INTO _compare_result(code, path) VALUES ('missing-src', dst_full_path);
            FETCH dst_cur INTO dst_row;
        END IF;
    END LOOP;
    INSERT INTO _compare_result(code, synced_files, synced_bytes) VALUES ('synced-stats', synced_files, synced_bytes);
    OPEN result_cur FOR SELECT * FROM _compare_result;
    RETURN result_cur;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.pg_cancel_backend(
    pid integer,
    reason TEXT)
RETURNS BOOL AS $$
BEGIN
    RAISE WARNING 'Cancelling pg op with pid % due to %', pid, reason;
    RETURN (SELECT pg_cancel_backend(pid));
END;
$$ LANGUAGE plpgsql SECURITY INVOKER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION postgres_major_version() RETURNS INT AS $$
BEGIN
    -- This function should work like PgDbApi.get_server_version()
    RETURN current_setting('server_version_num')::INT / 10000 AS version;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf.get_pg_hostname() RETURNS TEXT AS $$
DECLARE
    pg_hostname TEXT;
BEGIN
    -- Table pg_temp_hostname is temporary so it should be visible only in the session it was created.
    CREATE TEMPORARY TABLE pg_temp_hostname (hostname text);
    -- There is no '/etc/hostname' on CentOS; '/proc/sys/kernel/hostname' should be available
    -- on all Linux distributions.
    COPY pg_temp_hostname FROM '/proc/sys/kernel/hostname';
    SELECT hostname INTO pg_hostname FROM pg_temp_hostname;
    DROP TABLE pg_temp_hostname;
    RETURN pg_hostname;
END;
$$ LANGUAGE plpgsql VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf.physical_size_bucket_to_lower_limit(bucket TEXT)
RETURNS BIGINT AS $$
   SELECT CASE bucket
       WHEN '0B' THEN 0::BIGINT
       WHEN '1B ... <1KiB' THEN 1::BIGINT
       WHEN '1KiB ... <10KiB' THEN 1::BIGINT * 1024
       WHEN '10KiB ... <100KiB' THEN 10::BIGINT * 1024
       WHEN '100KiB ... <1MiB' THEN 100::BIGINT * 1024
       WHEN '1MiB ... <10MiB' THEN 1::BIGINT * 1024 * 1024
       WHEN '10MiB ... <100MiB' THEN 10::BIGINT * 1024 * 1024
       WHEN '100MiB ... <1GiB' THEN 100::BIGINT * 1024 * 1024
       WHEN '1GiB ... <10GiB' THEN 1::BIGINT * 1024 * 1024 * 1024
       WHEN '10GiB ... <100GiB' THEN 10::BIGINT * 1024 * 1024 * 1024
       WHEN '100GiB ... <1TiB' THEN 100::BIGINT * 1024 * 1024 * 1024
       WHEN '1TiB ... <10TiB' THEN 1::BIGINT * 1024 * 1024 * 1024 * 1024
       WHEN '10TiB ... <100TiB' THEN 10::BIGINT * 1024 * 1024 * 1024 * 1024
       WHEN '100TiB ... <1PiB' THEN 100::BIGINT * 1024 * 1024 * 1024 * 1024
       WHEN '1PiB ... <10PiB' THEN 1::BIGINT * 1024 * 1024 * 1024 * 1024 * 1024
       WHEN '10PiB ... <100PiB' THEN 10::BIGINT * 1024 * 1024 * 1024 * 1024 * 1024
       WHEN '100PiB ... <1EiB' THEN 100::BIGINT * 1024 * 1024 * 1024 * 1024 * 1024
       WHEN '>=1EiB' THEN 1024::BIGINT * 1024 * 1024 * 1024 * 1024 * 1024
       ELSE -1
   END
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


-- This function was previously created in migrations. I changed it from plsql to sql and put it here
-- instead of creating new migration.
CREATE OR REPLACE FUNCTION sf.get_extension(basename text)
RETURNS text AS
$$
    -- reimplementation of get_extension (from sfutils/extension.py):
    -- - extension doesn't start with '.'
    -- - ignore first '.' if basename starts with '.'
    -- - convert to lowercase
    SELECT CASE
        WHEN position('.' in reverse(basename)) = length(basename) THEN '' -- the only dot is at beginning
        ELSE LOWER(RIGHT(basename, - length(basename) + position('.' in reverse(basename)) - 1))
    END
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


-- This function was previously created in migrations. I changed it from plsql to sql and put it here
-- instead of creating new migration.
CREATE OR REPLACE FUNCTION sf.get_extension_no_lower(basename text)
RETURNS text AS
$$
    SELECT CASE
        WHEN position('.' in reverse(basename)) = length(basename) THEN '' -- the only dot is at beginning
        ELSE RIGHT(basename, - length(basename) + position('.' in reverse(basename)) - 1)
    END
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf_internal.calculate_processing_stats()
    RETURNS JSONB AS $$
DECLARE
    processing_stats JSONB;
BEGIN
    WITH grouped_stats AS (
        SELECT stat_name || '_' || type AS stat_name,
               COALESCE(SUM(count), 0) AS count,
               COALESCE(SUM(size), 0) AS size
        FROM process_event_temp_processing_stats
        GROUP BY stat_name, type
        ORDER BY stat_name
    ), splitted_stats AS (
	    SELECT stat_name || '_COUNT' AS stat_name,
	           count AS value
	    FROM grouped_stats

	    UNION ALL

	    SELECT stat_name || '_SIZE' AS stat_name,
	           size AS value
	    FROM grouped_stats
    )
    SELECT json_object_agg(stat_name, value)
        INTO processing_stats
    FROM splitted_stats;

    RETURN processing_stats;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;

-- This function is used in query compare_mode. It takes two cursors of files in different point in times
CREATE OR REPLACE FUNCTION sf_internal.compare_entries(
    src_cur refcursor,
    dst_cur refcursor)
RETURNS void AS $$
DECLARE
    src_row RECORD;
    dst_row RECORD;
    status TEXT;
BEGIN
    -- Be carefull, that table is used and removed later in code
    CREATE TEMPORARY TABLE _compare_entries(
        status TEXT NOT NULL,
        parent TEXT,
        src_fn TEXT,
        dst_fn TEXT,
        src_size BIGINT,
        dst_size BIGINT,
        username TEXT,
        mtime TIMESTAMP
    );

    FETCH src_cur INTO src_row;
    FETCH dst_cur INTO dst_row;
    WHILE NOT src_row IS NULL OR NOT dst_row IS NULL LOOP
        IF (src_row IS NULL) OR (src_row.fn > dst_row.fn) OR (src_row.fn = dst_row.fn AND src_row.parent_path > dst_row.parent_path) THEN
            INSERT INTO _compare_entries VALUES ('new', dst_row.parent_path, null, dst_row.fn, null, dst_row.size, dst_row.username, dst_row.mt);
            FETCH dst_cur INTO dst_row;
        ELSIF (dst_row IS NULL) OR (src_row.fn < dst_row.fn) OR (src_row.fn = dst_row.fn AND src_row.parent_path < dst_row.parent_path) THEN
            INSERT INTO _compare_entries VALUES ('removed', src_row.parent_path, src_row.fn, null, src_row.size, null, src_row.username, src_row.mt);
            FETCH src_cur INTO src_row;
        ELSE
            IF src_row.size = dst_row.size AND src_row.mt = dst_row.mt THEN
                status = 'no_change';
            ELSE
                status = 'changed';
            END IF;
            INSERT INTO _compare_entries VALUES (status, src_row.parent_path, src_row.fn, dst_row.fn, src_row.size, dst_row.size, src_row.username, src_row.mt);
            FETCH dst_cur INTO dst_row;
            FETCH src_cur INTO src_row;
        END IF;
    END LOOP;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;
-- This function is used in query compare_mode. It takes two cursors of files in different point in times

-- WARNING! Function below is marked as IMMUTABLE which is not 100% correct,
-- as it is possible to change volume_name. But this function is designed
-- to use in places where it is hard to extract volume_id and then use
-- it. One of the places is Redash where single SQL is executed so it should
-- be immune to any consequences of volume renaming.
-- If any problems with this function are detected then it should be
-- changed to STABLE.
CREATE OR REPLACE FUNCTION sf.volume_id_from_name(
    volume_name TEXT)
RETURNS BIGINT AS
$$
    SELECT id FROM sf_volumes.volume WHERE name = volume_name;
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf.job_name_id_from_name(
    job_name TEXT)
RETURNS BIGINT AS
$$
    SELECT id FROM sf.job_name as jn WHERE jn.name = job_name;
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf.volume_ids_from_zone_name(
    zone_name TEXT)
RETURNS BIGINT[] AS
$$
    SELECT array_agg(volume_id) FROM (
        SELECT DISTINCT tvc.volume_id
        FROM sf.tag_value_current tvc
        INNER JOIN sf.tag_name tn ON tvc.name_id = tn.id
        INNER JOIN sf.tag_namespace space ON tn.namespace_id = space.id
        JOIN sf_auth.zone zname ON tn.name = zname.id::VARCHAR
        WHERE space.name = '__zone'
        AND zname.name = zone_name
    ) t
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;

CREATE OR REPLACE FUNCTION sf.dir_ids_from_zone_name(
    zone_name TEXT)
RETURNS BIGINT[] AS
$$
    SELECT array_agg(fs_entry_id) FROM (
        SELECT DISTINCT tvc.fs_entry_id
        FROM sf.tag_value_current tvc
        INNER JOIN sf.tag_name tn ON tvc.name_id = tn.id
        INNER JOIN sf.tag_namespace space ON tn.namespace_id = space.id
        JOIN sf_auth.zone zname ON tn.name = zname.id::VARCHAR
        WHERE space.name = '__zone'
        AND zname.name = zone_name
    ) t
$$ LANGUAGE SQL IMMUTABLE PARALLEL SAFE;
