#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

SFHOME="${SFHOME:-/opt/starfish}"
SFCLI="${SFCLI:-${SFHOME}/bin/client}"
OK=true
CMD_NUM=0

fatal_error() {
    echo "$1" >&2
    exit 2
}

_run_cmd() {
    CMD_NUM=$((CMD_NUM + 1))

    local filename
    filename="$(printf "%02d" "${CMD_NUM}")-$1"
    shift

    (exec "$@" >"${filename}.out" 2>"${filename}.err") || {
        echo >&2 "COMMAND: \`$*\` RETURN_CODE: $?"
        OK=false
    }
}

_run_cmd_background() {
    CMD_NUM=$((CMD_NUM + 1))

    local filename
    filename="$(printf "%02d" "${CMD_NUM}")-$1"

    "$@" >"${filename}.out" 2>"${filename}.err" &
}

run_sf_cmd() {
    local cmd1="$1"
    local cmd2="${2:-""}"

    _run_cmd "${cmd1}-${cmd2}" "${SFCLI}" "$@"
    _run_cmd "${cmd1}-${cmd2}-json" "${SFCLI}" "$@" --json
}

run_cmd() {
    local cmd1="$1"

    _run_cmd "${cmd1}" "$@"
}

collect_logs() {
    local temp_storage_path

    if [[ "${RUN_SF_CLI_ONLY}" != sf_cli_only ]]; then
        if [[ "${DO_NOT_RUN_IO_TOP:-""}" != true ]]; then
            _run_cmd_background iotop --iter="${BACKGROUND_LEN}" --delay=1 --time --quiet --only
        fi
        _run_cmd_background pidstat -u 1 "${BACKGROUND_LEN}"

        run_cmd uptime
        run_cmd uname -a
        run_cmd lscpu
        run_cmd lsblk --topology
        run_cmd lshw
        run_cmd lsmod

        _run_cmd meminfo cat /proc/meminfo
        _run_cmd slabinfo cat /proc/slabinfo
        _run_cmd sysctl sysctl -a
        # twice --stats to get moar stats
        _run_cmd interfaces ip --stats --stats addr

        _run_cmd processes ps faux
        _run_cmd threads ps axms

        temp_storage_path="$(${SFCLI} check-config --json | jq -r .services.temp.out_data.status.storage_path || true)"
        if [[ -z "${temp_storage_path}" ]] || [[ "${temp_storage_path}" == null ]]; then
            temp_storage_path="${SFHOME}/lib/temp/storage"
        fi
        if [[ -d "${temp_storage_path}" ]]; then
            _run_cmd "temp-storage_tree-dump" ls -Rla "${temp_storage_path}"
        fi
    fi

    if [[ "${FOR_AGENT}" == false ]]; then
        # These command is only run on Starfish core server. There is no point in running
        # it on agents, because result for each agent and core server is the same.
        run_sf_cmd check-config --full --show-agents-diff
        # Create an .out file with a config properties verification
        _run_cmd "check-config--verify-properties" "${SFCLI}" check-config --verify-properties

    fi
    # service-status does not support --json
    _run_cmd service-status "${SFCLI}" service-status
    if [[ "${FOR_AGENT}" == false ]]; then
        # These commands are only run on Starfish core server. There is no point in running
        # them on agents, because result for each agent and core server is the same.
        run_sf_cmd volume list --details --with-mount-opts --wrap 140
        # shellcheck disable=SC2086
        run_sf_cmd scan list --details ${LIMIT_ARG} ${DAYS_ARG} --wrap 140
        run_sf_cmd scan pending --details --wrap 140
        # shellcheck disable=SC2086
        run_sf_cmd job list --details ${LIMIT_ARG} ${DAYS_ARG} --wrap 140
        run_sf_cmd job pending --details --wrap 140
        run_sf_cmd cron list
        run_sf_cmd archive-target list --details
        run_sf_cmd archive list --details
        run_sf_cmd restore list --details
        run_sf_cmd zone list --details
        run_sf_cmd tagset list --details
    fi

    if [[ "${WITH_REDASH}" == true ]]; then
        _run_cmd redash-status "${SFHOME}/bin/redash" status
        _run_cmd redash-settings "${SFHOME}/bin/redash" check_settings
    fi

    wait
}

parse_args() {
    BACKGROUND_LEN=30
    DAYS_ARG=""
    FOR_AGENT=false
    LIMIT_ARG="-n 500"
    WITH_REDASH=true
    RUN_SF_CLI_ONLY=no

    while [ $# -gt 0 ]; do
        case $1 in
        --background-len)
            shift
            BACKGROUND_LEN="${1:-}"
            ;;
        --run-sf-cli-only)
            shift
            RUN_SF_CLI_ONLY="${1:-}"
            ;;
        --days)
            shift
            DAYS_ARG="--days ${1:-}"
            LIMIT_ARG=""
            ;;
        --for-agent)
            FOR_AGENT=true
            ;;
        --no-redash)
            WITH_REDASH=false
            ;;
        *)
            break
            ;;
        esac
        shift
    done

    [ $# = 0 ] || fatal_error "Unrecognized arguments: $*"
}

main() {
    parse_args "$@"

    collect_logs

    if [ $OK != true ]; then
        exit 1
    fi
}

main "$@"
