#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

readonly SSL_SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

# shellcheck source=scripts/installation/_utils.sh
source "${SSL_SCRIPT_DIR}/_utils.sh"

get_all_names() {
    local host
    for host in $(hostname --all-fqdns; hostname --all-ip; hostname); do
        echo "${host}"
    done | sort -u
}

create_ssl_cert() {
    local ssl_dir="$1"
    local ssl_crt_name="$2"
    # take first 64 chars, as CN field in cert has a limit of 64 chars
    local ssl_common_name="${3:0:64}"
    local ssl_days_valid=825
    local ssl_crt_file="${ssl_dir}/${ssl_crt_name}.crt"
    local ssl_key_file="${ssl_dir}/${ssl_crt_name}.key"
    local ssl_conf_file="${ssl_dir}/openssl-${ssl_crt_name}.cnf"

    local host hostnames dns_idx

    mkdir -p "${ssl_dir}"

    if ! file_exists "${ssl_crt_file}"; then
        cp "${SSL_SCRIPT_DIR}/openssl.cnf" "${ssl_conf_file}"
        sed -i "s|^default_keyfile = KEY_FILE$|default_keyfile = ${ssl_key_file//|/\\|}|g" "${ssl_conf_file}"
        sed -i "s|^CN=AGENT$|CN=${ssl_common_name//|/\\|}|g" "${ssl_conf_file}"

        hostnames=$(get_all_names)
        dns_idx=1
        for host in ${hostnames}; do
            echo "DNS.${dns_idx} = ${host}" >> "${ssl_conf_file}"
            dns_idx=$((dns_idx+1))
        done

        log "Generating ${ssl_common_name} SSL certificate and key"
        # based on https://certsimple.com/blog/openssl-csr-command
        openssl req -x509 -sha256 -newkey rsa:2048 -config "${ssl_conf_file}" -keyout "${ssl_key_file}" -out "${ssl_crt_file}" -days "${ssl_days_valid}"
        rm "${ssl_conf_file}"
    else
        log "${ssl_crt_file} already exists, not regenerating"
    fi
}

main() {
    if [[ $# -lt 3 ]]; then
        >&2 echo "Usage: $0 ssl_dir ssl_crt_name ssl_common_name"
        exit 2
    fi

    local ssl_dir="$1"
    local ssl_crt_name="$2"
    local ssl_common_name="$3"

    create_ssl_cert "${ssl_dir}" "${ssl_crt_name}" "${ssl_common_name}"
}

sourced() {
    [[ "${BASH_SOURCE[0]}" != "${0}" ]]
}

if ! sourced; then
    main "$@"
fi
