#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

REDASH_COMMON_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
readonly REDASH_COMMON_DIR

# shellcheck source=scripts/installation/_secret_key.sh
source "${REDASH_COMMON_DIR}/_secret_key.sh"

REDASH_POSTGRES_LISTEN_ADDRESS="127.0.0.1"
REDASH_POSTGRES_STORAGE_TYPE="ssd"
REDASH_POSTGRES_CONFIG_TEMPLATE="redash"
REDASH_POSTGRES_DB_NAME="redash"
REDASH_POSTGRES_USERNAME="redash"

REDASH_REDIS_LISTEN_ADDRESS="127.0.0.1"
REDASH_REDIS_USERNAME="redash"

REDASH1_PG_PORT=5435
REDASH1_PG_VERSION=9.6
REDASH1_REDIS_PORT=6380
REDASH1_DEFAULT_GUNICORN_LISTEN_ADDRESS="localhost:5001"

REDASH8_PG_PORT=5434
REDASH8_PG_VERSION=9.6
REDASH8_REDIS_PORT=6381
REDASH8_DEFAULT_GUNICORN_LISTEN_ADDRESS="localhost:5002"

REDASH10_PG_PORT=5436
REDASH10_PG_VERSION=13
REDASH10_REDIS_PORT=6382
REDASH10_DEFAULT_GUNICORN_LISTEN_ADDRESS="localhost:5003"

export REDASH_DEFAULT_EMAIL="starfish@starfishstorage.com"
# generate random password for Redash of length 20
REDASH_DEFAULT_PASSWORD="$(generate_secret_key 20)"
export REDASH_DEFAULT_PASSWORD

# shellcheck source=scripts/installation/_utils.sh
source "${REDASH_COMMON_DIR}/_utils.sh"

install_redash_postgresql() {
    local pg_version="${1:-${REDASH10_PG_VERSION}}"
    local log_file="$2"

    # --no-create-cluster to add PGDG repos and install Postgres packages, without creating a cluster at all
    # --no-backup to avoid trying to install & configure pgbackrest - Redash DB is not backed up
    local postgres_opts=(
        --pg-version "${pg_version}"
        --log-file "${log_file}"
        --no-create-cluster
        --no-backup
    )

    ALLOW_FOR_INTERNAL_SCRIPT=1 DB_THRESHOLD=0 "${REDASH_COMMON_DIR}/_postgresql.sh" "${postgres_opts[@]}"
}

configure_redash_postgresql() {
    local pg_port="$1"
    local db_path="$2"
    local cluster_name="$3"
    local password="$4"
    local log_file="$5"
    local overwrite="$6"
    local pg_version="${7:-${REDASH10_PG_VERSION}}"

    # --no-backup to avoid trying to install & configure pgbackrest - Redash DB is not backed up
    # --no-install-pkgs as need to run install_redash_postgresql before creating PG cluster for Redash
    local postgres_opts=(
        --pg-version "${pg_version}"
        --log-file "${log_file}"
        --listen-address "${REDASH_POSTGRES_LISTEN_ADDRESS}"
        --port "${pg_port}"
        --username "${REDASH_POSTGRES_USERNAME}"
        --password "${password}"
        --db-path "${db_path}"
        --db-storage-type "${REDASH_POSTGRES_STORAGE_TYPE}"
        --db-config-template "${REDASH_POSTGRES_CONFIG_TEMPLATE}"
        --pg-cluster-name "${cluster_name}"
        --db-name "${REDASH_POSTGRES_DB_NAME}"
        --no-backup
        --no-install-pkgs
    )

    if [[ "${overwrite}" == true ]]; then
        postgres_opts+=(--drop-existing-cluster)
    fi

    ALLOW_FOR_INTERNAL_SCRIPT=1 DB_THRESHOLD=0 "${REDASH_COMMON_DIR}/_postgresql.sh" "${postgres_opts[@]}"
}

install_redash_redis() {
    local redis_port="$1"
    local redis_dir="$2"
    local password="$3"
    local log_file="$4"
    local install_pkgs="$5"
    local redis_opts=(
        --log-file "${log_file}"
        --listen-address "${REDASH_REDIS_LISTEN_ADDRESS}"
        --port "${redis_port}"
        --password "${password}"
        --redis-dir "${redis_dir}"
        --username "${REDASH_REDIS_USERNAME}"
    )

    if [[ "${install_pkgs}" == false ]]; then
        redis_opts+=(--no-install-pkgs)
    fi

    ALLOW_FOR_INTERNAL_SCRIPT=1 "${REDASH_COMMON_DIR}/_redis.sh" "${redis_opts[@]}"
}

get_redash_dir() {
    local redash_version="$1"

    if [[ "${redash_version}" -eq 1 ]]; then
        echo -n "${SFHOME}/redash/1.0.3"
    elif [[ "${redash_version}" -eq 8 ]]; then
        echo -n "${STARFISH_REDASH_DIR}/8.0.2"
    elif [[ "${redash_version}" -ge 10 ]]; then
        # a hack - Redash is using Calendar Versioning now - https://calver.org/
        # which makes major versions increment every year and minor each month
        # we don't want to keep changing the dir that often, so we'll keep the Redash in the old 10.1.0 dir
        # and have a symlink to this dir
        echo -n "${STARFISH_REDASH_DIR}/10.1.0"
    else
        log "Unsupported Redash version ${redash_version}"
        exit 1
    fi
}

get_redash_name_with_version_at_most_10() {
    local redash_version="$1"

    if [[ "${redash_version}" -eq 1 ]]; then
        echo -n redash
    elif [[ "${redash_version}" -eq 8 ]]; then
        echo -n redash8
    elif [[ "${redash_version}" -ge 10 ]]; then
        # a hack - Redash is using Calendar Versioning now - https://calver.org/
        # which makes major versions increment every year and minor each month
        # we don't want to change the names so often so we'll keep redash10 wherever possible
        echo -n redash10
    else
        log "Unsupported Redash version ${redash_version}"
        exit 1
    fi
}

get_redash_pg_port() {
    local redash_version="$1"

    if [[ "${redash_version}" -eq 1 ]]; then
        echo -n "${REDASH1_PG_PORT}"
    elif [[ "${redash_version}" -eq 8 ]]; then
        echo -n "${REDASH8_PG_PORT}"
    elif [[ "${redash_version}" -ge 10 ]]; then
        echo -n "${REDASH10_PG_PORT}"
    else
        log "Unsupported Redash version ${redash_version}"
        exit 1
    fi
}

get_redash_pg_version() {
    local redash_version="$1"

    if [[ "${redash_version}" -eq 1 ]]; then
        echo -n "${REDASH1_PG_VERSION}"
    elif [[ "${redash_version}" -eq 8 ]]; then
        echo -n "${REDASH8_PG_VERSION}"
    elif [[ "${redash_version}" -ge 10 ]]; then
        echo -n "${REDASH10_PG_VERSION}"
    else
        log "Unsupported Redash version ${redash_version}"
        exit 1
    fi
}

get_redash_redis_port() {
    local redash_version="$1"

    if [[ "${redash_version}" -eq 1 ]]; then
        echo -n "${REDASH1_REDIS_PORT}"
    elif [[ "${redash_version}" -eq 8 ]]; then
        echo -n "${REDASH8_REDIS_PORT}"
    elif [[ "${redash_version}" -ge 10 ]]; then
        echo -n "${REDASH10_REDIS_PORT}"
    else
        log "Unsupported Redash version ${redash_version}"
        exit 1
    fi
}

get_redash_gunicorn_address() {
    local redash_version="$1"

    if [[ "${redash_version}" -eq 1 ]]; then
        echo -n "${REDASH1_DEFAULT_GUNICORN_LISTEN_ADDRESS}"
    elif [[ "${redash_version}" -eq 8 ]]; then
        echo -n "${REDASH8_DEFAULT_GUNICORN_LISTEN_ADDRESS}"
    elif [[ "${redash_version}" -ge 10 ]]; then
        echo -n "${REDASH10_DEFAULT_GUNICORN_LISTEN_ADDRESS}"
    else
        log "Unsupported Redash version ${redash_version}"
        exit 1
    fi
}

get_redash_pg_cluster_name() {
    local redash_version="$1"

    # redash for Redash 1, redash8 for Redash 8, redash10 for Redash 10 and later
    get_redash_name_with_version_at_most_10 "${redash_version}"
}

get_redash_service_manager_name() {
    local redash_version="$1"

    # sf-redash for Redash 1, sf-redash8 for Redash 8, sf-redash10 for Redash 10 and later
    echo -n "sf-$(get_redash_name_with_version_at_most_10 "${redash_version}")"
}

get_redash_pkg_name() {
    local redash_version="$1"

    if [[ "${redash_version}" -eq 1 ]]; then
        echo -n redash
    elif [[ "${redash_version}" -eq 8 ]]; then
        echo -n redash8-systemd
    elif [[ "${redash_version}" -ge 10 ]]; then
        echo -n "redash${redash_version}"
    else
        log "Unsupported Redash version ${redash_version}"
        exit 1
    fi
}

get_redash_env_filename() {
    local redash_version="$1"

    echo -n "$(get_redash_dir "${redash_version}")/.env"
}

get_redash_pg_dir() {
    local redash_version="$1"

    echo -n "$(get_redash_dir "${redash_version}")/pg"
}

get_redash_redis_dir() {
    local redash_version="$1"

    echo -n "$(get_redash_dir "${redash_version}")/redis"
}

run_redash_manage() {
    local redash_version="$1"
    shift
    local msg="$1"
    shift
    local manage_path

    manage_path="$(get_redash_dir "${redash_version}")/bin/manage"
    if [[ ! -e "${manage_path}" ]]; then
        # this is backwards compatibility for Redash 1, not sure if supporting upgrading from Redash 1 is even
        # needed anymore, but removing support for Redash 1 is a bigger task than I am willing to do right now
        mkdir --parents "$(dirname "${manage_path}")"
        cat >"${manage_path}" <<'EOF'
readonly BIN_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
readonly REDASH_HOME_DIR="$(readlink -ev "${BIN_DIR}/..")"

cd "${REDASH_HOME_DIR}/src"
exec bin/run python manage.py "$@"
EOF
        chmod 755 "${manage_path}"
    fi

    log "${msg}"

    "$(get_redash_dir "${redash_version}")/bin/manage" "$@"
}

systemd_stop_redash() {
    local redash_service_manager_name="$1"

    systemctl stop "${redash_service_manager_name}.target"
}

supervisord_stop_redash() {
    local redash_service_manager_name="$1"

    supervisorctl stop "${redash_service_manager_name}:*"
}

stop_redash() {
    local redash_version="$1"
    local redash_service_manager_name

    redash_service_manager_name=$(get_redash_service_manager_name "${redash_version}")
    echo "Stopping Redash ${redash_version}"
    run_func_for_service_manager "${redash_service_manager_name}.target" systemd_stop_redash "${redash_service_manager_name}" supervisord_stop_redash "${redash_service_manager_name}"
}

systemd_start_redash() {
    local redash_service_manager_name="$1"

    systemctl start "${redash_service_manager_name}.target"
}

supervisord_start_redash() {
    local redash_service_manager_name="$1"

    supervisorctl start "${redash_service_manager_name}:*"
}

start_redash() {
    local redash_version="$1"
    local redash_service_manager_name

    redash_service_manager_name=$(get_redash_service_manager_name "${redash_version}")
    echo "Starting Redash ${redash_version}"
    run_func_for_service_manager "${redash_service_manager_name}.target" systemd_start_redash "${redash_service_manager_name}" supervisord_start_redash "${redash_service_manager_name}"
}

dump_redash_pg() {
    local redash_version="$1"
    local dump_file="$2"
    local pg_dump

    pg_dump="$(find_pg_96_command pg_dump)"

    echo "Dumping PostgreSQL database of Redash ${redash_version} to ${dump_file}"
    su postgres -c "${pg_dump} --clean --if-exists --dbname=${REDASH_POSTGRES_DB_NAME} --port=$(get_redash_pg_port "${redash_version}") --no-password" | gzip >"${dump_file}"
}

stop_redash_postgres() {
    local redash_version="$1"
    local pg_version pg_cluster_name port

    pg_version="$(get_redash_pg_version "${redash_version}")"
    pg_cluster_name="$(get_redash_pg_cluster_name "${redash_version}")"
    port="$(get_redash_pg_port "${redash_version}")"

    echo "Stopping Postgres for Redash ${redash_version}"
    stop_postgresql_service "${pg_version}" "${pg_cluster_name}" "${port}"
}

disable_redash_postgres_service() {
    local redash_version="$1"
    local pg_version pg_cluster_name service_name

    pg_version="$(get_redash_pg_version "${redash_version}")"
    pg_cluster_name=$(get_redash_pg_cluster_name "${redash_version}")
    service_name=$(get_pg_service_name "${pg_cluster_name}" "${pg_version}")

    echo "Disabling Postgres service for Redash ${redash_version}"
    systemctl disable "${service_name}"
}
