#!/bin/bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

_MIGRATE_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
readonly _MIGRATE_DIR

# shellcheck source=scripts/installation/_common_install.sh
source "${_MIGRATE_DIR}/_common_install.sh"

restore_pg_to_new_redash() {
    local redash_version="$1"
    local dump_file="$2"

    if [[ ${redash_version} == 10 ]]; then
        psql="$(find_pg_13_command psql)"
    else
        psql="$(find_pg_96_command psql)"
    fi

    echo "Restoring PostgreSQL database into Redash ${redash_version} DB from ${dump_file}"
    zcat "${dump_file}" | su postgres -c "${psql} --quiet --tuples-only --echo-errors --dbname=${REDASH_POSTGRES_DB_NAME} --port=$(get_redash_pg_port "${redash_version}") --no-password"
}

cut_password_from_url_in_env_file() {
    local env="$1"
    local var="$2"

    # REDASH_DATABASE_URL=postgresql://user:password@localhost:5435/redash
    # REDASH_REDIS_URL=redis://user:password@localhost:6380/0
    # remove comments to avoid trying to cut out from a commented out line
    # --invert-match "^\s*#" means: remove lines which start with 0 or more whitespace chars followed by `#`
    grep --invert-match "^\s*#" "${env}" | grep "${var}" | cut -f 3 -d : | cut -f 1 -d @
}

cut_pg_password_from_env_file() {
    local env="$1"

    cut_password_from_url_in_env_file "${env}" REDASH_DATABASE_URL
}

cut_redis_password_from_env_file() {
    local env="$1"

    cut_password_from_url_in_env_file "${env}" REDASH_REDIS_URL
}

create_new_redash_env_file() {
    local env_file_from="$1"
    local env_file_to="$2"
    local version_from="$3"
    local version_to="$4"

    echo "Creating ${env_file_to} based on ${env_file_from}"

    sed -e "s|:$(get_redash_pg_port "${version_from}")/|:$(get_redash_pg_port "${version_to}")/|g" \
        -e "s|:$(get_redash_redis_port "${version_from}")/|:$(get_redash_redis_port "${version_to}")/|g" \
        <"${env_file_from}" >"${env_file_to}"
}

upgrade_db() {
    local redash_version="$1"

    # manage.py db upgrade needs Redis running (not sure why to be honest)
    redash_service_manager_name=$(get_redash_service_manager_name "${redash_version}")
    systemctl start "${redash_service_manager_name}-redis.service"
    run_redash_manage "${redash_version}" "Upgrading DB schema" db upgrade
}

disable_redash() {
    local redash_version="$1"
    local redash_version_major="$2"

    local redash_service_manager_name redash_postgres_service_name

    redash_service_manager_name=$(get_redash_service_manager_name "${redash_version_major}")
    if [[ "${redash_version}" == "1" || "${redash_version}" == "8" ]]; then
        supervisorctl stop "${redash_service_manager_name}:*"
        supervisorctl remove "${redash_service_manager_name}"
    else
        systemctl stop "${redash_service_manager_name}.target"
        systemctl disable "${redash_service_manager_name}.target"
    fi

    redash_postgres_service_name=$(get_pg_service_name "$(get_redash_pg_cluster_name "${redash_version_major}")" 9.6)
    systemctl disable "${redash_postgres_service_name}"
}

main() {
    local valid_versions_from=("1" "8" "8-systemd")
    local valid_versions_to=("8-systemd" "10")
    local dump_file redis_pwd postgres_pwd
    local env_file_from env_file_to ver_from ver_from_major ver_to ver_to_major pg_version_to
    local custom_log_file="false"
    local install_pkgs="true"
    local args=("$@")

    fail_if_not_root

    while [[ $# -gt 0 ]]; do
        case $1 in
        --log-file)
            shift
            LOG_FILE="$1"
            custom_log_file="true"
            ;;
        --no-install-pkgs)
            # may be called from post-install RPM/DEB
            # this means we should have all packages installed and we can't use apt/yum as it'll result in a deadlock
            install_pkgs="false"
            ;;
        --from)
            shift
            ver_from="$1"
            ;;
        --to)
            shift
            ver_to="$1"
            ;;
        *)
            break
            ;;
        esac
        shift
    done

    # Simple check for existence of a string (containing no spaces) in a space-separated array.
    # shellcheck disable=SC2076
    if [[ ! " ${valid_versions_from[*]} " =~ " ${ver_from} " ]]; then
        echo "Unsupported from-version: ${ver_from}. Valid values: ${valid_versions_from[*]}"
        exit 0
    fi
    # Simple check for existence of a string (containing no spaces) in a space-separated array.
    # shellcheck disable=SC2076
    if [[ ! " ${valid_versions_to[*]} " =~ " ${ver_to} " ]]; then
        echo "Unsupported to-version: ${ver_to}. Valid values: ${valid_versions_to[*]}"
        exit 0
    fi
    if [[ "${ver_to}" == "${ver_from}" ]]; then
        echo "From-version and to-version have to be different".
        exit 0
    fi
    if [[ "${ver_from}" == "8" && "${ver_to}" == "8-systemd" ]]; then
        echo "This script only supports migrations between major Redash versions".
        exit 0
    fi

    if [[ "${ver_from}" == "8-systemd" ]]; then
        ver_from_major="8"
    else
        ver_from_major="${ver_from}"
    fi
    if [[ "${ver_to}" == "8-systemd" ]]; then
        ver_to_major="8"
    else
        ver_to_major="${ver_to}"
    fi
    redash_dir_from="$(get_redash_dir "${ver_from_major}")"
    if [[ ${ver_from} == "8" ]]; then
        redash_dir_from=${redash_dir_from/redash-systemd/redash}
    fi
    if [[ "${custom_log_file}" == "false" ]]; then
        LOG_FILE="${redash_dir_from}/migrate_redash_${ver_from}_to_${ver_to}.log"
    fi

    if [[ "${ver_to}" == "10" ]]; then
        pg_version_to="13"
    else
        pg_version_to="9.6"
    fi

    env_file_from="$(get_redash_env_filename "${ver_from_major}")"
    if [[ ${ver_from} == "8" ]]; then
        env_file_from=${env_file_from/redash-systemd/redash}
    fi
    env_file_to="$(get_redash_env_filename "${ver_to_major}")"

    if ! file_exists "${env_file_from}"; then
        echo "${env_file_from} does not exist, no need to migrate from Redash ${ver_from}"
        exit 0
    fi
    if file_exists "${env_file_to}"; then
        echo "${env_file_to} already exists, no need to migrate to Redash ${ver_to}"
        exit 0
    fi

    export LOG_FILE
    log "Script called with arguments: $0 " ${args[@]+"${args[@]}"}

    # dump first - it happens that something goes wrong here and it's better to exit without setting up Redis and PG
    # which will make it obvious that the new Redash is not ready
    # otherwise you get directories created, possible services (Postgres and Redis) but without data in the DB
    dump_file="${redash_dir_from}/$(date +'%Y%m%d_%H%M%S')_redash_${ver_from}.sql.gz"
    stop_redash "${ver_from_major}"
    dump_redash_pg "${ver_from_major}" "${dump_file}"
    start_redash "${ver_from_major}"

    redis_pwd="$(cut_redis_password_from_env_file "${env_file_from}")"
    postgres_pwd="$(cut_pg_password_from_env_file "${env_file_from}")"

    if [[ "${install_pkgs}" == true ]]; then
        install_redash_postgresql "${pg_version_to}" "${LOG_FILE}"
    fi

    configure_redash_postgresql "$(get_redash_pg_port "${ver_to_major}")" "$(get_redash_pg_dir "${ver_to_major}")" "$(get_redash_pg_cluster_name "${ver_to_major}")" "${postgres_pwd}" "${LOG_FILE}" false "${pg_version_to}"
    install_redash_redis "$(get_redash_redis_port "${ver_to_major}")" "$(get_redash_redis_dir "${ver_to_major}")" "${redis_pwd}" "${LOG_FILE}" "${install_pkgs}"

    if [[ "${install_pkgs}" = "true" ]]; then
        pkg_install "$(get_redash_pkg_name "${ver_to_major}")"
    fi

    stop_redash "${ver_to_major}"
    restore_pg_to_new_redash "${ver_to_major}" "${dump_file}"
    create_new_redash_env_file "${env_file_from}" "${env_file_to}" "${ver_from_major}" "${ver_to_major}"
    upgrade_db "${ver_to_major}"
    start_redash "${ver_to_major}"

    stop_redash_postgres "${ver_from_major}"
    disable_redash_postgres_service "${ver_from_major}"
    disable_redash "${ver_from}" "${ver_from_major}"
}

main "$@"
