#!/bin/bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

_MIGRATE_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
readonly _MIGRATE_DIR

# shellcheck source=scripts/installation/_common_install.sh
source "${_MIGRATE_DIR}/_common_install.sh"

PREFIX="${RPM_INSTALL_PREFIX:-}"
if [[ "${PREFIX}" = "/" ]]; then
    PREFIX=""
fi
SFHOME="${PREFIX}${SFHOME:-/opt/starfish}"
STARFISH_NGINX_CONF_DIR=$(readlink --verbose --canonicalize-missing "${SFHOME}/nginx/etc/conf.d")
NGINX_REDASH_FILE="${STARFISH_NGINX_CONF_DIR}/starfish/redash8.conf"

REDASH_8_BACKUP_FILE_NAME="redash-8.0.2-backup.tar.gz"
# system_files is a new directory for redash files outside sfhome/redash*/8.0.2
BACKUP_AND_REMOVE_REDASH_ENTRIES=(
    bin
    celery
    .env
    gunicorn.conf
    pg
    redis
    src
    venv
    system_files
)
BACKUP_AND_REMOVE_SYSTEM_ENTRIES=(
    /etc/systemd/system/multi-user.target.wants/postgresql-9.6-redash8.service
    /etc/systemd/system/multi-user.target.wants/postgresql-9.6@redash8.service
    /etc/systemd/system/postgresql-9.6-redash8.service
    /etc/systemd/system/postgresql-9.6@redash8.service
     "${NGINX_REDASH_FILE}"
    /etc/logrotate.d/sf-redash8
    /etc/logrotate.d/sf-redash8-systemd
    /etc/supervisord.d/sf-redash8.ini
    /etc/supervisor/conf.d/sf-redash8.conf
    /usr/lib/systemd/system/sf-redash8.target
    /usr/lib/systemd/system/sf-redash8-server.service
    /usr/lib/systemd/system/sf-redash8-redis.service
    /usr/lib/systemd/system/sf-redash8-celery.service
    /usr/lib/systemd/system/sf-redash8-celery-scheduled.service
)

backup_and_remove_redash_files() {
    local redash_dir="$1"
    local backup_file
    local existing_backup_redash_entries=()

    backup_file="${redash_dir}/${REDASH_8_BACKUP_FILE_NAME}"
    system_files_dir="${redash_dir}/system_files"

    # ( CMD ) is subshell, so that "cd" doesn't effect this shell
    (
        for file in "${BACKUP_AND_REMOVE_SYSTEM_ENTRIES[@]}"; do
            if file_exists "${file}"; then
                destination="${system_files_dir}/$(dirname "${file}")"
                log Copying "${file}" to "${destination}"
                mkdir --parents "${destination}"
                cp --preserve=all "${file}" "${destination}"
            fi
        done
        cd "${redash_dir}"

        for entry in "${BACKUP_AND_REMOVE_REDASH_ENTRIES[@]}"; do
            if path_exists "${entry}"; then
                existing_backup_redash_entries+=("${entry}")
            fi
        done

        log Backing up Redash 8 to "${backup_file}"
        tar -czf "${backup_file}" "${existing_backup_redash_entries[@]}"

        # Config files and files that were created after installing redash
        # are not automatically removed by yum/apt thus trying to remove all files here.
        for entry in "${existing_backup_redash_entries[@]}"; do
            log Removing "${entry}"
            rm -rf "${entry}"
        done
    )
}

dump_db_and_stop_redash_pg() {
    local redash_dir="$1"
    local dump_file

    if is_postgresql_running "$(get_redash_pg_port 8)"; then
        dump_file="${redash_dir}/$(date +'%Y%m%d_%H%M%S')_redash_8.sql.gz"
        log Creating Redash PG dump file "${dump_file}"
        dump_redash_pg 8 "${dump_file}"
        log Stopping Redash PG cluster
        stop_redash_postgres 8
    fi
}

cleanup_redash_pg() {
    local pg_version pg_cluster_name db_path

    pg_version="$(get_redash_pg_version 8)"
    pg_cluster_name="$(get_redash_pg_cluster_name 8)"
    db_path="$(get_redash_pg_dir 8)"

    remove_postgresql_cluster "${pg_version}" "${pg_cluster_name}" "${db_path}"
}

cleanup_system_files() {
    local file

    for file in "${BACKUP_AND_REMOVE_SYSTEM_ENTRIES[@]}"; do
        log "Removing ${file}"
        rm -f "${file}"
    done
}

backup_and_remove_redash() {
    local redash_dir="$1"

    stop_redash 8 || true
    dump_db_and_stop_redash_pg "${redash_dir}"
    backup_and_remove_redash_files "${redash_dir}"
    cleanup_redash_pg
    # Remove files from system at the end because some of them may be needed meanwhile.
    # For example systemd unit files for postgres.
    cleanup_system_files
}

main() {
    local env_file_supervisor env_file_systemd backup_file
    fail_if_not_root

    LOG_FILE="${SFHOME}/redash-systemd/_backup_and_remove_redash_8.log"
    while [[ $# -gt 0 ]]; do
        case $1 in
        --log-file)
            shift
            LOG_FILE="$1"
            ;;
        *)
            break
            ;;
        esac
        shift
    done
    export LOG_FILE

    env_file_systemd="$(get_redash_env_filename 8)"
    env_file_supervisor="${env_file_systemd//redash-systemd/redash}"
    if file_exists "${env_file_supervisor}"; then
        echo "Backing up and removing Redash 8 (supervisor version)"
        backup_and_remove_redash "$(dirname "${env_file_supervisor}")"
        echo "Redash 8 (supervisor version) backup and removal finished"
    else
        echo "${env_file_supervisor} does not exist, no need to backup and remove Redash 8 (supervisor version)"
    fi
    if file_exists "${env_file_systemd}"; then
        echo "Backing up and removing Redash 8 (systemd version)"
        backup_and_remove_redash "$(dirname "${env_file_systemd}")"
        echo "Redash 8 (systemd version) backup and removal finished"
    else
        echo "${env_file_systemd} does not exist, no need to backup and remove Redash 8 (systemd version)"
    fi
}

main "$@"
