/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */


CREATE OR REPLACE FUNCTION sf_reports.generate_age_buckets(bucket_months INTEGER[])
RETURNS TABLE(lower_bound_month INTEGER, label VARCHAR) AS
$$
DECLARE
    bucket INTEGER;
	last_bucket INTEGER := 0;
BEGIN
    FOREACH bucket IN ARRAY bucket_months
    loop
		IF bucket % 12 = 0 AND last_bucket % 12 = 0 THEN
		    label := FORMAT('Previous Years: %s-%s', last_bucket/12, bucket/12);
	    ELSE
		    label := FORMAT('Previous Months: %s-%s', last_bucket, bucket);
		END IF;
        lower_bound_month := last_bucket;
		last_bucket := bucket;
        RETURN NEXT;
    END LOOP;
	IF last_bucket % 12 = 0 THEN
	    label := FORMAT('Previous Years: > %s', last_bucket/12);
    ELSE
	    label := FORMAT('Previous Months: > %s', last_bucket);
	END IF;
	lower_bound_month := last_bucket;
	RETURN NEXT;
END;
$$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;


CREATE TABLE IF NOT EXISTS sf_reports.age_buckets AS SELECT * FROM sf_reports.generate_age_buckets(ARRAY[1,3,6,12,24,36]);


CREATE OR REPLACE FUNCTION sf_reports.set_age_buckets(bucket_months INTEGER[])
RETURNS VOID AS
$$
	DELETE FROM sf_reports.age_buckets;
	INSERT INTO sf_reports.age_buckets SELECT * FROM sf_reports.generate_age_buckets(bucket_months)
$$ LANGUAGE sql volatile PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_reports.months_to_age_bucket(months INTEGER)
RETURNS TEXT AS
$$
	SELECT label FROM sf_reports.age_buckets
	WHERE lower_bound_month <= months ORDER BY lower_bound_month DESC LIMIT 1
$$ LANGUAGE sql IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf_reports.days_to_months(days INTEGER)
RETURNS INTEGER AS
$$
DECLARE
    days_per_month INTEGER := 30;
    days_per_year INTEGER := 365;
    full_years INTEGER;
    months INTEGER;
BEGIN
    full_years = days / days_per_year;
	months = (days % days_per_year) / days_per_month;
	IF months > 11 THEN
		months = 11;
	END if;
	RETURN months + 12 * full_years;
END;
$$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf_reports.days_to_age_bucket(days INTEGER)
RETURNS TEXT AS
$$
BEGIN
	RETURN sf_reports.months_to_age_bucket(sf_reports.days_to_months(days));
END
$$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf_reports.interval_to_total_days(age_interval INTERVAL)
RETURNS INTEGER AS
$$
DECLARE
    days_per_month INTEGER := 30;
    days_per_year INTEGER := 365;
	total_days INTEGER;
BEGIN
	SELECT EXTRACT(days FROM age_interval) +
       EXTRACT(months FROM age_interval) * days_per_month +
       EXTRACT(years FROM age_interval) * days_per_year
    INTO total_days;
	RETURN total_days;
END
$$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;


CREATE OR REPLACE FUNCTION sf_reports.interval_to_age_bucket(age_interval INTERVAL)
RETURNS TEXT AS
$$
BEGIN
	RETURN sf_reports.days_to_age_bucket(sf_reports.interval_to_total_days(age_interval));
END
$$ LANGUAGE plpgsql IMMUTABLE PARALLEL SAFE;
