/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */

DO $$
    BEGIN
        IF NOT EXISTS(SELECT proname FROM pg_catalog.pg_proc WHERE proname = 'get_parameters_for_redash_scan_alert') THEN
            CREATE FUNCTION get_parameters_for_redash_scan_alert()
            RETURNS TABLE(scans_back INT, heartbeat_interval TEXT, volume_interval TEXT, hb_interval INTERVAL, v_interval INTERVAL)
            AS $Body$
                SELECT 3, '12 hours', '3 days', INTERVAL '12 hours', INTERVAL '3 days';
            $Body$
            LANGUAGE sql STABLE PARALLEL SAFE;
        END IF;
    END
$$;

WITH params AS
(
    SELECT scans_back, heartbeat_interval, volume_interval
    FROM get_parameters_for_redash_scan_alert()
),
scans AS
(
    SELECT id, volume_id, type, state_name, creation_time, heartbeat
    FROM sf_scans.scan
    -- test scans_back (repeatedly failed scans)
    UNION ALL SELECT -1, -1, 'diff', 'failed', null, null
    UNION ALL SELECT -2, -1, 'diff', 'failed', null, null
    UNION ALL SELECT -3, -1, 'diff', 'failed', null, null
    UNION ALL SELECT -4, -1, 'diff', 'failed', null, null
    UNION ALL SELECT -5, -1, 'diff', 'failed', null, null
    UNION ALL SELECT -6, -1, 'diff', 'done', null, null
    -- Successful scan before the parameter ago, should be shown
    UNION ALL SELECT -7, -2, 'diff', 'done', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 day', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 HOUR'
    -- failed scan within the parameter range
    UNION ALL SELECT -8, -2, 'diff', 'failed', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 day', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '+1 MINUTE'
    -- Successful scan within the parameter range, should not be shown
    UNION ALL SELECT -9, -3, 'diff', 'done', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 day', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '+1 MINUTE'
    -- Last scan before the parameter ago, should be shown
    UNION ALL SELECT -10, -4, 'diff', 'done', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 day', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 MINUTE'
    -- monitor heartbeat before the parameter, should be shown
    UNION ALL SELECT -11, -5, 'monitor_out_of_sync', 'in_progress', NOW() - (SELECT heartbeat_interval::INTERVAL FROM params) + INTERVAL '-1 DAY', NOW() - (SELECT heartbeat_interval::INTERVAL FROM params) + INTERVAL '-1 MINUTE'
     -- monitor heartbeat within the parameter range, should not be shown
    UNION ALL SELECT -12, -6, 'monitor_out_of_sync', 'in_progress', NOW() - (SELECT heartbeat_interval::INTERVAL FROM params) + INTERVAL '-1 DAY', NOW() - (SELECT heartbeat_interval::INTERVAL FROM params) + INTERVAL '+1 MINUTE'
    -- current monitor failed, should be shown
    UNION ALL SELECT -13, -7, 'monitor_out_of_sync', 'failed', NOW() - INTERVAL '2 day', NOW() - INTERVAL '1 HOUR' + INTERVAL '-1 HOUR'
    UNION ALL SELECT -14, -8, 'monitor_out_of_sync', 'in_progress', NOW() - INTERVAL '1 day', NOW() - INTERVAL '1 MINUTE'
    UNION ALL SELECT -15, -9, 'diff', 'done', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '10 day', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 DAY'
    UNION ALL SELECT -16, -9, 'monitor_out_of_sync', 'in_progress', NOW() - INTERVAL '1 day', NOW() - INTERVAL '1 MINUTE'
    -- Successful scan before the parameter ago
    UNION ALL SELECT -17, -10, 'diff', 'done', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 day', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 HOUR'
    -- and failed scan before the parameter range
    UNION ALL SELECT -18, -10, 'diff', 'failed', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 day', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 MINUTE'
    -- failed monitor scan
    UNION ALL SELECT -19, -11, 'monitor_out_of_sync', 'failed', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 day', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 HOUR'
    UNION ALL SELECT -20, -12, 'monitor_out_of_sync', 'failed', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '-1 day', NOW() - (SELECT volume_interval::INTERVAL FROM params) + INTERVAL '+1 HOUR'
    ORDER BY volume_id, heartbeat DESC, id DESC
),
scan_arrays AS
(
    SELECT volume_id, ARRAY_AGG(state_name) AS states, ARRAY_AGG(heartbeat) AS heartbeats
    FROM scans
    GROUP BY volume_id
),
last_scans AS
(
    SELECT volume_id, states[1:(SELECT scans_back FROM params)] AS states, heartbeats[1:(SELECT scans_back FROM params)] AS heartbeat
    FROM scan_arrays
),
monitor_scan_arrays AS
(
    SELECT volume_id, ARRAY_AGG(state_name) AS states, ARRAY_AGG(heartbeat) AS heartbeats
    FROM scans
    WHERE type ILIKE 'monitor%'
    GROUP BY volume_id
),
last_monitor_scans AS
(
    SELECT volume_id, states[1:(SELECT scans_back FROM params)] AS states, heartbeats[1:(SELECT scans_back FROM params)] AS heartbeat
    FROM monitor_scan_arrays
),
volume_scan_heartbeats AS
(
    SELECT volume_id, MAX(heartbeat) AS heartbeat
    FROM scans
    GROUP BY volume_id
),
volume_successful_scan_heartbeats AS
(
    SELECT volume_id, MAX(heartbeat) AS heartbeat
    FROM scans
    WHERE state_name = 'done'
    GROUP BY volume_id
),
volume_monitor_scans_in_progress AS
(
    SELECT volume_id
    FROM scans
    WHERE type ILIKE 'monitor%' AND state_name = 'in_progress'
    GROUP BY volume_id
),
checks AS
(
    SELECT CONCAT('monitor scan failed within last ', (SELECT volume_interval FROM params)) AS problem, volume_id
    FROM last_monitor_scans
    WHERE states[1] = 'failed' AND heartbeat[1] >= NOW() - (SELECT volume_interval::INTERVAL FROM params)

    UNION SELECT CONCAT('monitor heartbeat more than ', (SELECT heartbeat_interval FROM params), ' ago'), volume_id
    FROM scans WHERE type ILIKE 'monitor%' AND state_name = 'in_progress' AND heartbeat < NOW() - (SELECT heartbeat_interval::INTERVAL FROM params)

    UNION SELECT CONCAT((SELECT scans_back FROM params)::TEXT, ' failed scans in row'), volume_id
    FROM last_scans WHERE 'failed' = ALL(states) AND ARRAY_LENGTH(states, 1) >= (SELECT scans_back FROM params)

    UNION SELECT CONCAT('volume scan more than ', (SELECT volume_interval FROM params), ' ago'), COALESCE(vol.id, scan.volume_id)
    FROM sf_volumes.volume vol
    FULL OUTER JOIN volume_scan_heartbeats scan ON scan.volume_id = vol.id
    WHERE scan.volume_id IS NULL OR scan.heartbeat < NOW() - (SELECT volume_interval::INTERVAL FROM params)

    UNION SELECT CONCAT('volume successful scan more than ', (SELECT volume_interval FROM params), ' ago'), COALESCE(vol.id, scan.volume_id)
    FROM sf_volumes.volume vol
    FULL OUTER JOIN volume_successful_scan_heartbeats scan ON scan.volume_id = vol.id
    LEFT JOIN volume_monitor_scans_in_progress monitor ON monitor.volume_id = scan.volume_id
    WHERE scan.volume_id IS NULL OR (monitor.volume_id IS NULL AND scan.heartbeat < NOW() - (SELECT volume_interval::INTERVAL FROM params))

    ORDER BY volume_id, problem
),
volume_problems AS
(
    SELECT COALESCE(vol.name, checks.volume_id::TEXT) AS volume, problem
    FROM checks
    LEFT JOIN sf_volumes.volume vol ON vol.id = checks.volume_id
    -- WHERE COALESCE(vol.name, checks.volume_id::TEXT) ~ '^-[0-9]+'
    ORDER BY volume, problem
)

SELECT volume, string_agg(problem, ', ') AS problems
FROM volume_problems
GROUP BY volume
ORDER BY volume
