/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */

WITH runtime_range AS (
    SELECT MIN(run_time) AS start,
           MAX(run_time) AS end
    FROM sf_reports.last_time_generic_history
    WHERE run_time >= (now() - interval '{{number_of_days_to_look_back}} days')
), last_scan_time_per_volume AS (
    SELECT
        scan.volume_id,
        MAX(scan.heartbeat)::DATE last_scan_time
    FROM sf_scans.scan scan
        INNER JOIN sf_scans.loader_info ON scan.id=loader_info.scan_id
    WHERE (loader_info.stats->>'ADDED')::bigint > 0
        OR (loader_info.stats->>'CHANGED')::bigint > 0
        OR (loader_info.stats->>'REMOVED')::bigint > 0
    GROUP BY scan.volume_id
), group_size_for_chosen_days AS (
    SELECT volume_name,
           group_name,
           run_time,
           SUM(COALESCE(physical_size, size)) AS size
    FROM sf_reports.last_time_generic_history
    INNER JOIN runtime_range ON run_time = runtime_range.start OR run_time = runtime_range.end
    GROUP BY volume_name,
             group_name,
             run_time
), group_size_delta AS (
    SELECT volume_name,
           group_name,
           (lag(run_time) OVER (PARTITION BY group_name, volume_name ORDER BY run_time))::DATE AS start_date,
           run_time::DATE AS end_date,
           size,
           size - lag(size) OVER (PARTITION BY group_name, volume_name ORDER BY run_time) AS size_delta
    FROM group_size_for_chosen_days
), group_percentage_delta AS (
    SELECT group_name,
           volume_name,
           start_date,
           end_date,
           CASE WHEN (size - size_delta) = 0 THEN
                CASE WHEN size_delta = 0 THEN 0
                ELSE 'Infinity'::float
                END
           ELSE
                (size_delta * 100) / (size - size_delta)
           END AS percentage_delta,
           (size - size_delta) AS previous_size,
           size AS current_size,
           size_delta
   FROM group_size_delta
)
SELECT '<a href="DASHBOARD_LINK(group-detail-analytics)?p_groupname=' || group_name || '">' || 'Zoom' || '</a>' AS " ",
       group_name AS "groupname",
       COALESCE(volume.display_name, upd.volume_name) AS "volume",
       start_date AS "start date",
       LEAST(end_date, ls.last_scan_time) AS "end date",
       -- column below is printed in HTML to produce nice looking result in redash and force correct sorting order
       -- at the same time
       '<div order="' || LPAD((DENSE_RANK() OVER(ORDER BY percentage_delta ASC))::TEXT, 10, '0')
           || '" class="' ||
           CASE
               WHEN (percentage_delta > 0) THEN 'bg-success'
               WHEN (percentage_delta < 0) THEN 'bg-danger'
               ELSE ''
           END
           || ' text-center">' || ROUND(percentage_delta) || '%</div>'
           AS "percent delta",
       ROUND(previous_size / (1024 * 1024 * 1024.0), 1) AS "previous physical size (GiB)",
       ROUND(current_size / (1024 * 1024 * 1024.0), 1) AS "current physical size (GiB)",
       ROUND(size_delta / (1024 * 1024 * 1024.0), 1) AS "delta physical size (GiB)"
FROM group_percentage_delta AS upd
    LEFT JOIN sf_volumes.volume ON upd.volume_name = sf_volumes.volume.name
    LEFT JOIN last_scan_time_per_volume AS ls ON ls.volume_id = volume.id
WHERE ABS(percentage_delta) >= {{minimum_percent_change}}
  AND ABS(size_delta) >= {{minimum_delta_size_gb}}::DECIMAL * (1024 * 1024 * 1024.0)
ORDER BY size_delta DESC;
