"""
***********************************************************************************************************

 Starfish Storage Corporation ("Starfish") CONFIDENTIAL
 Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.

 NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
 Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
 intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
 U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
 Dissemination of this information or reproduction of this material is strictly forbidden unless prior
 written permission is obtained from Starfish. Access to the source code contained herein is hereby
 forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
 confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
 Starfish's software.

 ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
 THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
 AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
 FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
 DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
 WHOLE OR IN PART.

 FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
   These notices shall be marked on any reproduction of this data, in whole or in part.
   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
   Software clause at DFARS 52.227-7013.

***********************************************************************************************************
Version 1.1     2023-03-27      Doug
* fix issue when there's dotfile and dotdirs at top level to get proper commonprefix
Version 1.0     2019-10-24      Doug
* Raise exceptions on failure to find snapshot; standardize logging
"""
############################################################################
# Snapshot related class
############################################################################

import logging
import os
import shutil
import sys
import unittest
from glob import glob


class TestSnap(unittest.TestCase):
    """Unit test framework"""

    def setUp(self):
        os.makedirs("/tmp/test1fs/.snapshot/")
        os.mkdir("/tmp/test1fs/.snapshot/test_Hourly_20191022:1010")
        os.mkdir("/tmp/test1fs/.snapshot/test_Hourly_20191023:1010")
        os.mkdir("/tmp/test1fs/.snapshot/test_Hourly_20191021:1010")
        os.mkdir("/tmp/test1fs/.snapshot/test_Hourly_20191021:1009")
        os.mkdir("/tmp/test1fs/.snapshot/test_Hourly_20191023:1010/etc")
        os.mkdir("/tmp/test1fs/etc")
        with open("/tmp/test1fs/.snapshot/test_Hourly_20191023:1010/etc/hosts", "w") as tfile:
            print("testsnap", file=tfile)
        with open("/tmp/test1fs/etc/hosts", "w") as tfile:
            print("testdata", file=tfile)

    def test_snap_fail(self):
        """try a known failure"""
        with self.assertRaises(Exception) as context:
            print("Error on Karma expected:")
            SnapShot("top", "Karma", logging="TestSnap")
            self.assertTrue("bad path" in context.exception)

    def test_snap_instance_detect(self):
        """test bad glob, good dir"""
        with self.assertRaises(Exception) as context:
            print("bad path error expected:")
            SnapShot("top", "/tmp/test1fs/.snapshot/*2week*", logging="TestSnap")
            self.assertTrue("bad path" in context.exception)

    def test_good_glob_and_sort(self):
        """test snapshot detection"""
        snaph = SnapShot("top", "/tmp/test1fs/.snapshot/*Hourly*", logging="TestSnap")
        self.assertTrue(snaph)
        recent_dir = snaph.most_recent_glob()
        self.assertNotEqual(recent_dir, "/tmp/test1fs/.snapshot/test_Hourly_20191023:1009")
        self.assertEqual(recent_dir, "/tmp/test1fs/.snapshot/test_Hourly_20191023:1010")

    def test_snap_prefix_suffix(self):
        """test snapshot detection"""
        snaph = SnapShot("top", "/tmp/test1fs/.snapshot/*Hourly*", debug=0, logging="TestSnap")
        self.assertTrue(snaph)
        self.assertEqual(snaph.get_snap_prefix("/tmp/test1fs/etc/hosts"), "/tmp/test1fs/")
        self.assertEqual(snaph.get_path_suffix("/tmp/test1fs/etc/hosts"), "etc/hosts")
        self.assertEqual(
            snaph.get_snap_path("/tmp/test1fs/etc/hosts"), "/tmp/test1fs/.snapshot/test_Hourly_20191023:1010/etc/hosts"
        )
        self.assertNotEqual(
            snaph.get_snap_path("/tmp/test1fs/etc/hosts"), "/tmp/test1fs/.snapshot/test_Hourly_20191023:1009/etc/hosts"
        )

    def tearDown(self):
        if os.path.exists("/tmp/test1fs"):
            shutil.rmtree("/tmp/test1fs")


class SnapShot:
    """Deal with snapshots and path transpositions"""

    def __init__(self, snaploc, snapname, **kwargs):
        if "debug" in kwargs:
            self._debug_ = kwargs["debug"]
        else:
            self._debug_ = None
        if "logging" in kwargs:
            self._logger_ = logging.getLogger(kwargs["logging"])
        else:
            print(
                "improper logging module name used. Must initialize " "snapshot with a logger name (e.g. program name)",
                file=sys.stderr,
            )
            return
        self.location = snaploc
        self.name = snapname
        self._snappath_ = self.validate_snapshot()
        self._prefix_ = None

    def validate_snapshot(self):
        """validate snapshot arguments. Can really only do this adequately if it is 'top'.
        :returns: a normalized snapshot path of the directory to use for reference"""
        if self.location == "top":
            # it could be a glob, so get the most recent of the glob in dictionary order
            deglob_path = self.most_recent_glob()
            if not os.path.isdir(deglob_path):
                self._logger_.error(
                    "The specified snapshot directory does not appear to exist. "
                    "Please make sure you used the fully qualified path in the filesystem from /. "
                    " Exiting."
                )
                sys.exit(1)
        else:
            self._logger_.error("relative paths not yet supported for matching")
            sys.exit(1)

        return deglob_path

    def most_recent_glob(self):
        """Get the most recent file in a glob list for configuring a snapshot
        :returns: most recent (time/sort) of a list of candidate snapshots from a glob (or None)"""

        # It could be used as an absolute path or as a path relative to $volume:
        # if it's a relative path, prepend the volume name from the environment.
        # Do this once per program invocation and backfix self.name
        if not self.name.startswith("/") and "SF_JOB_SRC_VOL_MNT_POINT" in os.environ:
            volhome = os.environ["SF_JOB_SRC_VOL_MNT_POINT"]
            self.name = volhome + "/" + self.name
            if self._debug_ > 0:
                print("prepending volume path to snapshot path from relative path. Result: " + self.name)

        testglob = glob(self.name)
        if not testglob:
            logging.error("snapshot path %s appears to be invalid. No matches." % (self.name))  # noqa: S001
            raise Exception("bad path")
        bestpath = (sorted(testglob))[-1]
        if self._debug_:
            print("chose snapshot: " + bestpath, file=sys.stderr)
            logging.debug("chose snapshot: " + bestpath, file=sys.stderr)
        return bestpath

    @staticmethod
    def commonpath(path1, path2):
        """
        >>> SnapShot.commonpath("/foo/", "/shmoo/")
        '/'
        >>> SnapShot.commonpath("/foo/", "/foo/")
        '/foo'
        >>> SnapShot.commonpath("/foo/", "/foo")
        '/foo'
        >>> SnapShot.commonpath("/foo/bar/", "/foo/bar/")
        '/foo/bar'
        >>> SnapShot.commonpath("/foo/bar/", "/foo/baz/")
        '/foo'
        >>> SnapShot.commonpath("/mnt/vol/.snap", "/mnt/vol/.conda")
        '/mnt/vol'
        >>> SnapShot.commonpath("/mnt/vol/.snap/", "/mnt/vol/.conda/")
        '/mnt/vol'
        >>> SnapShot.commonpath("/mnt/vol/.snapshot/weekly-duration-123/", "/mnt/vol/.conda/")
        '/mnt/vol'
        """
        ret = os.path.commonpath([path1, path2])
        return ret

    def get_snap_prefix(self, path):
        """if snapshots are turned on, look for the snapshot file. If snapshot analog doesn't exist,
        skip it and log an error
        :returns: snapshot common prefix"""
        if self._debug_ > 1:
            print("comparing prefixes for " + path + " and " + self._snappath_)
        if not self._prefix_:
            self._prefix_ = self.commonpath(path, self._snappath_)
        if self._debug_ > 1:
            print("path common prefix is " + self._prefix_)
        return self._prefix_

    def get_path_suffix(self, path):
        """given a fully qualified path location, return the suffix which is just the part
        relative to the top of the volume.
        : returns: path suffix from top of volume (or snapshot)"""
        if not self._prefix_:
            self.get_snap_prefix(path)
        return path.replace(self._prefix_, "", 1)

    def get_snap_path(self, path):
        """take the snapshot directory prefix and append the path suffix
        : returns : interpolated path or None if it doesn't exist"""
        snappath = os.path.normpath(self._snappath_ + "/" + self.get_path_suffix(path))  # normpath:ignore
        if os.path.exists(snappath):
            return snappath
        else:
            logging.warning("snapshot for %s missing. Skipping.", path)
            return None
