/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */
/* no-temp-table-check */
/* All functions in this file should have a name ending '_vol_{{ vol_id }}', as this file is run
 * for each volume.
 * Defining function with name not related to '{{vol_id}}' may cause subtle 'tuple concurrently updated' error.
 * This is probably related to replacing function that is already running.
 */

CREATE OR REPLACE FUNCTION sf_internal.load_events_dirs_removed_vol_{{ vol_id }}
(
    bHistory BOOLEAN,                                           -- Track history for path and file changes
    oos_accuracy_secs BIGINT
)
    RETURNS sf_internal.load_events_ret_type AS $$
DECLARE
    events_processed BIGINT = 0;    -- Number of events processed (succeeded, ignored or failed)
    events_ignored BIGINT = 0;      -- Number of events ignored due to old sync_time
    oos_accuracy INTERVAL;
    temp record;
    ret record;                 -- return record
BEGIN
    oos_accuracy = oos_accuracy_secs * INTERVAL '1 seconds';

    FOR temp IN select * from load_events_events_with_entries
    LOOP
        events_processed = events_processed + 1;
        IF NOT sf_internal.process_event_vol_{{ vol_id }}(temp.event_row, temp.parent_id, temp.dir_row, NULL, bHistory, oos_accuracy) THEN
            events_ignored = events_ignored + 1;
        END IF;
    END LOOP;

    SELECT events_processed, events_ignored, 0::BIGINT AS events_failed INTO ret;

    RETURN ret;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.load_events_files_removed_vol_{{ vol_id }}
(
    bHistory BOOLEAN,                                       -- Track history for path and file changes
    oos_accuracy_secs BIGINT
)
    RETURNS sf_internal.load_events_ret_type AS $$
DECLARE
    events_processed BIGINT = 0;                            -- Number of events processed (succeeded, ignored or failed)
    events_ignored BIGINT = 0;                              -- Number of events ignored due to old sync_time
    event sf_internal.event_base;                           -- Row from the event table
    current_file sf.file_current;                           -- Row from file_current table
    files sf.file_current[] = ARRAY[]::sf.file_current[];
    temp record;
    ret sf_internal.load_events_ret_type;                   -- return record
BEGIN
    FOR temp IN
        SELECT *
        FROM load_events_events_with_entries
    LOOP
        events_processed := events_processed + 1;
        event = temp.event_row;
        current_file = temp.file_row;
        IF
            current_file.id IS NULL  -- Most likely it is a stale event
            OR ((current_file.sync_time IS NOT NULL) AND (event.sync_time < current_file.sync_time))
        THEN
            events_ignored := events_ignored + 1;
        ELSE
            -- Update parent path rec_aggrs
            INSERT INTO sf.refresh_aggregates_part_{{ partition_id }}(id, volume_id, depth)
            VALUES (current_file.parent_id, {{ vol_id }}, event.depth - 1);

            IF bHistory THEN
                PERFORM sf_internal.insert_file_into_history(current_file, event.sync_time);
            END IF;

            files := sf_internal.gather_remove_file_event_and_flush_if_needed_vol_{{ vol_id }}(
                files, current_file, bHistory
            );
        END IF;
    END LOOP;

    PERFORM sf_internal.flush_gathered_remove_file_events_vol_{{ vol_id }}(files, bHistory);

    SELECT events_processed, events_ignored, 0::BIGINT AS events_failed INTO ret;

    RETURN ret;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.load_events_dirs_added_vol_{{ vol_id }}
(
    bHistory BOOLEAN,               -- Track history for path and file changes
    oos_accuracy_secs BIGINT        -- unused here
)
    RETURNS sf_internal.load_events_ret_type AS $$
DECLARE
    events_processed BIGINT = 0;    -- Number of events processed (succeeded, ignored or failed)
    events_ignored BIGINT = 0;      -- Number of events ignored due to old sync_time
    events_failed BIGINT = 0;       -- Number of failed events
    size_sum NUMERIC = 0;           -- Sum of sizes processed
    processing_stats JSONB;         -- loading stats
    oos_accuracy INTERVAL;
    temp record;                    -- record with load_events_events_with_entries row
    ret sf_internal.load_events_ret_type;
BEGIN
    oos_accuracy = oos_accuracy_secs * INTERVAL '1 seconds';

    -- ADDED DIR EVENTS
    -- they have corresponding dir_row == NULL

        -- this temp table remembers which dir(seq_num) corresponds to which id
        INSERT INTO inserted_ids (
            SELECT (l.event_row).seq_num, nextval('sf.object_id_seq') AS id
            FROM load_events_events_with_entries as l
            WHERE dir_row IS NULL
        );

        WITH insert_data AS (
            -- data to be inserted into custom_fs_attrs
            SELECT (l.event_row).seq_num, nextval('sf.custom_fs_attrs_id_seq') AS custom_fs_attrs_id, (l.event_row).custom_fs_attrs
            FROM load_events_events_with_entries as l
            WHERE dir_row IS NULL AND (l.event_row).custom_fs_attrs IS NOT NULL AND (l.event_row).custom_fs_attrs != '{}'::JSONB
        ),
        insert_result AS (
            -- real inserting
            INSERT INTO sf.custom_fs_attrs_current(id, volume_id, attrs) (
                SELECT custom_fs_attrs_id, {{ vol_id }}, custom_fs_attrs
                FROM insert_data
            )
        )
        -- this temp table remembers which dir(seq_num) corresponds to which custom_fs_attrs row (custom_fs_attrs_id)
        INSERT INTO inserted_custom_fs_attrs_ids
            (SELECT seq_num, custom_fs_attrs_id from insert_data);

    WITH insert_data AS (
        -- inserting data to dir_current
        INSERT INTO sf.dir_current_part_{{ partition_id }} (
            id,
            history_id,
            valid,
            rec_aggrs,
            local_aggrs,
            sync_time,
            parent_id,
            volume_id,
            inode,
            size,
            blocks,
            ctime,
            atime,
            mtime,
            uid,
            gid,
            perms,
            name,
            path,
            depth,
            out_of_sync_time,
            errors,
            custom_fs_attrs_id,
            ancestor_ids
        )
            (SELECT
                inserted_ids.id,
                nextval('sf.history_object_id_seq') AS history_id,
                tstzrange(COALESCE((l.event_row).sync_time, (l.event_row).out_of_sync_time, (l.event_row).tree_out_of_sync_time), 'infinity', '[)') AS valid,
                '{}' AS rec_aggrs,
                COALESCE((l.event_row).local_aggrs, '{}') AS local_aggrs,
                COALESCE((l.event_row).sync_time, (l.event_row).out_of_sync_time, (l.event_row).tree_out_of_sync_time) AS sync_time,
                l.parent_id,
                {{ vol_id }} AS volume_id,
                (l.event_row).inode,
                (l.event_row).size,
                (l.event_row).blocks,
                (l.event_row).ctime,
                (l.event_row).atime,
                (l.event_row).mtime,
                (l.event_row).uid,
                (l.event_row).gid,
                (l.event_row).perms,
                (l.event_row).name,
                (l.event_row).path,
                (l.event_row).depth,
                CASE WHEN ((l.event_row).out_of_sync_time + oos_accuracy < (l.event_row).sync_time) THEN
                    NULL
                ELSE
                    (l.event_row).out_of_sync_time
                END,
                (l.event_row).errors,
                inserted_custom_fs_attrs_ids.custom_fs_attrs_id,
                l.parent_ancestor_ids || inserted_ids.id AS ancestor_ids
            FROM load_events_events_with_entries AS l
            JOIN inserted_ids ON (l.event_row).seq_num = inserted_ids.seq_num
            LEFT JOIN inserted_custom_fs_attrs_ids ON (l.event_row).seq_num = inserted_custom_fs_attrs_ids.seq_num
            WHERE dir_row IS NULL)
        RETURNING id, size
        )
    -- store stats from inserted data
    SELECT count(*), COALESCE(sum(size), 0)::NUMERIC from insert_data INTO events_processed, size_sum;

    INSERT INTO process_event_temp_processing_stats VALUES('ADDED', 'DIR', size_sum, events_processed);

    -- store tree_out_of_sync
    FOR temp IN (
        SELECT *
        FROM load_events_events_with_entries AS l
        JOIN inserted_ids ON (l.event_row).seq_num = inserted_ids.seq_num
        WHERE l.dir_row IS NULL AND (l.event_row).tree_out_of_sync_time IS NOT NULL)
    LOOP
        PERFORM sf_internal.set_tree_out_of_sync_time(temp.id, {{ vol_id }}, (temp.event_row).tree_out_of_sync_time);
    END LOOP;

    -- update rec_aggrs for parent path
    INSERT INTO sf.refresh_aggregates_part_{{ partition_id }}(id, volume_id, depth) (
        SELECT id, {{ vol_id }}, (l.event_row).depth
        FROM load_events_events_with_entries AS l
        JOIN inserted_ids ON (l.event_row).seq_num = inserted_ids.seq_num
        WHERE dir_row IS NULL
    );

    -- CHANGED EVENTS

    FOR temp IN
        SELECT *
        FROM load_events_events_with_entries AS events WHERE (events.dir_row).id IS NOT NULL
    LOOP
        events_processed = events_processed + 1;
        IF NOT sf_internal.process_event_vol_{{ vol_id }}(temp.event_row, temp.parent_id, temp.dir_row, NULL, bHistory, oos_accuracy) THEN
            events_ignored = events_ignored + 1;
        END IF;
    END LOOP;

    SELECT events_processed, events_ignored, 0::BIGINT AS events_failed INTO ret;

    RETURN ret;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.load_events_files_added_vol_{{ vol_id }}
(
    bHistory BOOLEAN,               -- Track history for path and file changes
    oos_accuracy_secs BIGINT        -- unused here
)
    RETURNS sf_internal.load_events_ret_type AS $$
DECLARE
    events_processed BIGINT = 0;    -- Number of events processed (succeeded, ignored or failed)
    events_ignored BIGINT = 0;      -- Number of events ignored due to old sync_time
    events_failed BIGINT = 0;       -- Number of failed events
    size_sum NUMERIC = 0;           -- Sum of sizes processed
    processing_stats JSONB;         -- loading stats
    oos_accuracy INTERVAL;
    temp record;                    -- record with load_events_events_with_entries row
    ret sf_internal.load_events_ret_type;
BEGIN
    oos_accuracy = oos_accuracy_secs * INTERVAL '1 seconds';

    -- ADDED FILE EVENTS
    -- they have corresponding file_row == NULL

        WITH insert_data AS (
            -- data to be inserted into custom_fs_attrs
            SELECT (l.event_row).seq_num, nextval('sf.custom_fs_attrs_id_seq') AS custom_fs_attrs_id, (l.event_row).custom_fs_attrs
            FROM load_events_events_with_entries as l
            WHERE file_row IS NULL AND (l.event_row).custom_fs_attrs IS NOT NULL AND (l.event_row).custom_fs_attrs != '{}'::JSONB
        ),
        insert_result AS (
            -- real inserting
            INSERT INTO sf.custom_fs_attrs_current(id, volume_id, attrs) (
                SELECT custom_fs_attrs_id, {{ vol_id }}, custom_fs_attrs
                FROM insert_data
            )
        )
        -- this temp table remembers which file(seq_num) corresponds to which custom_fs_attrs row (custom_fs_attrs_id)
        INSERT INTO inserted_custom_fs_attrs_ids
            (SELECT seq_num, custom_fs_attrs_id from insert_data);

    WITH insert_data AS (
        -- inserting data to file_current
        INSERT INTO sf.file_current_part_{{ partition_id }} (
                id,
                history_id,
                valid,
                volume_id,
                parent_id,
                inode,
                size,
                blocks,
                nlinks,
                ctime,
                atime,
                mtime,
                uid,
                gid,
                type,
                perms,
                name,
                target,
                sync_time,
                custom_fs_attrs_id
        )
            (SELECT
                nextval('sf.object_id_seq'),
                nextval('sf.history_object_id_seq'),
                tstzrange((l.event_row).sync_time, 'infinity', '[)'),
                {{ vol_id }} AS volume_id,
                l.parent_id AS parent_id,
                (l.event_row).inode,
                (l.event_row).size,
                (l.event_row).blocks,
                (l.event_row).nlinks,
                (l.event_row).ctime,
                (l.event_row).atime,
                (l.event_row).mtime,
                (l.event_row).uid,
                (l.event_row).gid,
                (l.event_row).type,
                (l.event_row).perms,
                (l.event_row).name,
                (l.event_row).target,
                (l.event_row).sync_time,
                inserted_custom_fs_attrs_ids.custom_fs_attrs_id
            FROM load_events_events_with_entries AS l
            LEFT JOIN inserted_custom_fs_attrs_ids ON (l.event_row).seq_num = inserted_custom_fs_attrs_ids.seq_num
            WHERE file_row IS NULL)
        RETURNING id, size
        )
    -- store stats from inserted data
    SELECT count(*), COALESCE(sum(size), 0)::NUMERIC from insert_data INTO events_processed, size_sum;

    INSERT INTO process_event_temp_processing_stats VALUES('ADDED', 'FILE', size_sum, events_processed);

    -- Update rec_aggrs for parent path
    INSERT INTO sf.refresh_aggregates_part_{{ partition_id }}(id, volume_id, depth) (
        SELECT parent_id, {{ vol_id }}, (l.event_row).depth - 1 FROM load_events_events_with_entries AS l WHERE file_row IS NULL
    );

    -- CHANGED EVENTS

    FOR temp IN
        SELECT *
        FROM load_events_events_with_entries AS events WHERE (events.file_row).id IS NOT NULL
    LOOP
        events_processed = events_processed + 1;
        IF NOT sf_internal.process_event_vol_{{ vol_id }}(temp.event_row, temp.parent_id, NULL, temp.file_row, bHistory, oos_accuracy) THEN
            events_ignored = events_ignored + 1;
        END IF;
    END LOOP;

    SELECT events_processed, events_ignored, 0::BIGINT AS events_failed INTO ret;

    RETURN ret;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.process_event_vol_{{ vol_id }}(
    rEvent sf_internal.event_base,      -- Row from the event table
    lParentId BIGINT,                   -- Parent path Id
    rCurrentDir sf.dir_current,         -- Row from dir_current table
    rCurrentFile sf.file_current,       -- Row from file_current table
    bHistory BOOLEAN,                   -- Track history for path and file changes
    oos_accuracy INTERVAL
) RETURNS BOOLEAN AS $$                 -- Returns TRUE if the event is applied, returns FALSE if the event is ignored
DECLARE
    rNewDir sf.dir_current_part_{{ partition_id }};         -- event converted to dir row
    rNewFile sf.file_current_part_{{ partition_id }};       -- event converted to file row
    custom_fs_attrs_id BIGINT;
    validUpperBound timestamptz;
    historically_significant_change BOOLEAN;
    current_row_is_stub BOOLEAN = FALSE;
    event_contains_sync_time BOOLEAN;
    posix_attrs_updated BOOLEAN = FALSE;
    out_of_sync_time_updated BOOLEAN = FALSE;
    tree_out_of_sync_time_updated BOOLEAN = FALSE;
    calculated_sync_time timestamptz;
    files_to_be_deleted sf.file_current[] = ARRAY[]::sf.file_current[];
    dirs_to_be_deleted sf.dir_current[] = ARRAY[]::sf.dir_current[];
BEGIN
    event_contains_sync_time = (rEvent.sync_time IS NOT NULL);
    calculated_sync_time = COALESCE(rEvent.sync_time, rEvent.out_of_sync_time, rEvent.tree_out_of_sync_time);

    -- Process dirs
    IF rEvent.type = 16384 THEN
        IF lParentId IS NULL AND rEvent.path != '' AND rEvent.event_type != 'REMOVED' THEN
            raise exception 'Parent_id should be NULL only for root path.';
        END IF;

        -- Get path info from event
        rNewDir.parent_id = lParentId;
        rNewDir.volume_id = {{ vol_id }};
        rNewDir.inode = rEvent.inode;
        rNewDir.size = rEvent.size;
        rNewDir.blocks = rEvent.blocks;
        rNewDir.ctime = rEvent.ctime;
        rNewDir.atime = rEvent.atime;
        rNewDir.mtime = rEvent.mtime;
        rNewDir.uid = rEvent.uid;
        rNewDir.gid = rEvent.gid;
        rNewDir.perms = rEvent.perms;
        rNewDir.name = rEvent.name;
        rNewDir.path = rEvent.path;
        rNewDir.depth = rEvent.depth;
        rNewDir.local_aggrs = rEvent.local_aggrs;
        rNewDir.out_of_sync_time = rEvent.out_of_sync_time;
        rNewDir.errors = rEvent.errors;

        -- Processing ADDED dirs is no longer here. See: load_events_dirs_added_vol_{{ vol }}

        -- Process REMOVED dirs
        IF rEvent.event_type = 'REMOVED' THEN
            IF rCurrentDir.out_of_sync_time IS NOT NULL AND rCurrentDir.out_of_sync_time + oos_accuracy >= calculated_sync_time THEN
                RETURN FALSE;  -- do not remove dir, as it would erase information that it has to be refreshed (probably it was quickly recreated)
            END IF;

            IF rCurrentDir.id IS NULL THEN
                -- It is normal because POSIX crawler emits REMOVED events for each directory in a removed tree
                -- so Dir may not exists when parent received REMOVED event first and removed whole subtree
                RETURN TRUE;
            END IF;

            INSERT INTO process_event_temp_loadevents_dir_delete
                SELECT * FROM sf.dir_current_part_{{ partition_id }} WHERE volume_id={{ vol_id }}
                      AND ancestor_ids && ARRAY[rCurrentDir.id]::BIGINT[];

            analyze process_event_temp_loadevents_dir_delete (volume_id, id);
            -- Update rec_aggrs for parent
            INSERT INTO sf.refresh_aggregates_part_{{ partition_id }}(id, volume_id, depth) VALUES (rCurrentDir.parent_id, {{ vol_id }}, rCurrentDir.depth - 1);

            -- Create history for files and delete them
            FOR rCurrentFile IN
                SELECT file_current_part_{{ partition_id }}.*
                  FROM process_event_temp_loadevents_dir_delete AS dir_delete
                       INNER JOIN sf.file_current_part_{{ partition_id }}
                            ON file_current_part_{{ partition_id }}.volume_id = {{ vol_id }} AND file_current_part_{{ partition_id }}.parent_id = dir_delete.id
            LOOP
                IF (rCurrentFile.sync_time IS NOT NULL) AND (calculated_sync_time < rCurrentFile.sync_time) THEN
                    INSERT INTO process_event_temp_dir_ids_not_to_delete VALUES (rCurrentFile.parent_id);
                    CONTINUE;
                END IF;

                IF bHistory THEN
                    PERFORM sf_internal.insert_file_into_history(rCurrentFile, calculated_sync_time);
                END IF;

                files_to_be_deleted := sf_internal.gather_remove_file_event_and_flush_if_needed_vol_{{ vol_id }}(
                    files_to_be_deleted, rCurrentFile, bHistory
                );
            END LOOP;

            PERFORM sf_internal.flush_gathered_remove_file_events_vol_{{ vol_id }}(files_to_be_deleted, bHistory);

            -- Create history for dirs and delete them
            FOR rCurrentDir IN
                SELECT *
                  FROM process_event_temp_loadevents_dir_delete
                 ORDER BY depth DESC
            LOOP
                IF (EXISTS (SELECT 1 FROM process_event_temp_dir_ids_not_to_delete WHERE dir_id=rCurrentDir.id LIMIT 1)) OR
                    ((rCurrentDir.sync_time IS NOT NULL) AND (calculated_sync_time < rCurrentDir.sync_time) OR (calculated_sync_time < LOWER(rCurrentDir.valid))) THEN
                    -- this dir has child newer then REMOVED event - but all children with smaller sync_time will
                    -- be removed - we need to set out_of_sync_time flag so that mtime scan may have a chance to fix this
                    IF rCurrentDir.out_of_sync_time IS NULL THEN
                            -- SET out of sync marker. We used to set 0 here and now Marker is out_of_sync_time == sync_time + 1
                            UPDATE sf.dir_current_part_{{ partition_id }} SET out_of_sync_time = rCurrentDir.sync_time + INTERVAL '1 second'
                            WHERE id = rCurrentDir.id AND volume_id = {{ vol_id }};
                    END IF;
                    INSERT INTO sf.refresh_aggregates_part_{{ partition_id }}(id, volume_id, depth) VALUES (rCurrentDir.id, {{ vol_id }}, rCurrentDir.depth);
                    INSERT INTO process_event_temp_dir_ids_not_to_delete VALUES (rCurrentDir.parent_id);
                    CONTINUE;
                END IF;

                IF bHistory THEN
                     PERFORM sf_internal.insert_dir_into_history(rCurrentDir, calculated_sync_time);
                END IF;

                dirs_to_be_deleted := sf_internal.gather_remove_dir_event_and_flush_if_needed_vol_{{ vol_id }}(
                    dirs_to_be_deleted, rCurrentDir, bHistory
                );
            END LOOP;

            PERFORM sf_internal.flush_gathered_remove_dir_events_vol_{{ vol_id }}(dirs_to_be_deleted, bHistory);

            TRUNCATE process_event_temp_loadevents_dir_delete;
            TRUNCATE process_event_temp_dir_ids_not_to_delete;

        -- Process CHANGED dirs
        ELSIF (rEvent.event_type = 'ADDED' OR rEvent.event_type = 'CHANGED') AND rCurrentDir.id IS NOT NULL THEN
            IF (NOT event_contains_sync_time) THEN
                IF sf_internal.carries_file_attr_changes(rEvent) THEN
                    RAISE EXCEPTION 'Only events with sync_time can carry changes in file attribute and permissions; '
                                    'event: %', rEvent;
                END IF;
            END IF;

            current_row_is_stub = (rCurrentDir.size IS NULL);
            IF (NOT event_contains_sync_time)
                OR ((rCurrentDir.sync_time IS NOT NULL)  -- sync_time can be NULL in old migrated data - changing events should not be ignored
                    AND (NOT current_row_is_stub) AND (calculated_sync_time <= rCurrentDir.sync_time)) THEN
                rNewDir = rCurrentDir;
            ELSE
                posix_attrs_updated = TRUE;
                rNewDir.sync_time = calculated_sync_time;

                rNewDir.id = rCurrentDir.id;
                rNewDir.history_id = COALESCE(rCurrentDir.history_id, nextval('sf.history_object_id_seq'));
                rNewDir.rec_aggrs = rCurrentDir.rec_aggrs;
                rNewDir.depth = COALESCE(rNewDir.depth, rCurrentDir.depth);
                rNewDir.inode = COALESCE(rNewDir.inode, rCurrentDir.inode);
                rNewDir.size = COALESCE(rNewDir.size, rCurrentDir.size);
                rNewDir.blocks = COALESCE(rNewDir.blocks, rCurrentDir.blocks);
                rNewDir.ctime = COALESCE(rNewDir.ctime, rCurrentDir.ctime);
                rNewDir.atime = COALESCE(rNewDir.atime, rCurrentDir.atime);
                rNewDir.mtime = COALESCE(rNewDir.mtime, rCurrentDir.mtime);
                rNewDir.uid = COALESCE(rNewDir.uid, rCurrentDir.uid);
                rNewDir.gid = COALESCE(rNewDir.gid, rCurrentDir.gid);
                rNewDir.perms = COALESCE(rNewDir.perms, rCurrentDir.perms);
                rNewDir.path = COALESCE(rNewDir.path, rCurrentDir.path);
                rNewDir.name = COALESCE(rNewDir.name, rCurrentDir.name);
                rNewDir.local_aggrs = COALESCE(rNewDir.local_aggrs, rCurrentDir.local_aggrs);
                rNewDir.errors = rNewDir.errors;
                rNewDir.ancestor_ids = rCurrentDir.ancestor_ids;

                IF NOT sf_internal.is_historically_significant_change(rCurrentDir, rEvent.custom_fs_attrs, rNewDir)
                        OR (NOT current_row_is_stub AND LOWER(rCurrentDir.valid) >= calculated_sync_time)
                        OR (current_row_is_stub AND LOWER(rCurrentDir.valid) < calculated_sync_time) THEN
                    -- Old valid range is retained when:
                    -- - no entry is added to sf.dir_history (insignificant change),
                    -- - or entry is not a stub and processing an event with sync_time newer than current sync_time,
                    --      but older than valid range (possible due to accumulating directory's recursive aggregates
                    --      in history every 24 hours - see commit 4e9c80317a for more details),
                    -- - or entry is a stub and current valid range begins earlier than event's sync_time:
                    --      it's not safe to shorten the valid range, because --point-in-time queries may include
                    --      entry's descendants, but not the entry itself.
                    -- As stubs don't go to history, the last point is also the reason why the valid range is extended
                    -- if the opposite condition is true: LOWER(rCurrentDir.valid) >= calculated_sync_time).
                    rNewDir.valid = rCurrentDir.valid;
                ELSE
                    rNewDir.valid = tstzrange(calculated_sync_time, 'infinity', '[)');
                END IF;
            END IF;
            IF sf_internal.can_update_oos_time(rCurrentDir, rEvent, calculated_sync_time, oos_accuracy) THEN
                out_of_sync_time_updated = TRUE;
                rNewDir.out_of_sync_time = rEvent.out_of_sync_time;
            ELSE
                IF (rCurrentDir.out_of_sync_time IS NOT NULL) AND
                   -- this is the stub created by inserting missing parents.
                   (rCurrentDir.out_of_sync_time + oos_accuracy < calculated_sync_time) THEN
                    -- we can safely clean out_of_sync_time
                    rNewDir.out_of_sync_time = NULL;
                ELSE
                    rNewDir.out_of_sync_time = rCurrentDir.out_of_sync_time;
                END IF;
            END IF;

            IF rEvent.tree_out_of_sync_time IS NOT NULL THEN
                SELECT sf_internal.set_tree_out_of_sync_time(rNewDir.id, {{ vol_id }}, rEvent.tree_out_of_sync_time)
                    INTO tree_out_of_sync_time_updated;
            END IF;

            IF (NOT posix_attrs_updated) AND (NOT out_of_sync_time_updated) THEN
                RETURN tree_out_of_sync_time_updated;
            END IF;

            INSERT INTO process_event_temp_processing_stats VALUES('CHANGED', 'DIR', COALESCE(rNewDir.size, 0) - COALESCE(rCurrentDir.size, 0), 1);

            IF (posix_attrs_updated) THEN
                IF bHistory AND (NOT current_row_is_stub) AND (rNewDir.valid <> rCurrentDir.valid) THEN
                    PERFORM sf_internal.insert_dir_into_history(rCurrentDir, calculated_sync_time);
                    rNewDir.history_id = nextval('sf.history_object_id_seq');
                END IF;
            END IF;

            SELECT sf_internal.update_custom_fs_attrs(rCurrentDir.custom_fs_attrs_id, {{ vol_id }}, rEvent.custom_fs_attrs, bHistory) INTO custom_fs_attrs_id;
            rNewDir.custom_fs_attrs_id = custom_fs_attrs_id;

            UPDATE sf.dir_current_part_{{ partition_id }}
               SET valid = rNewDir.valid,
                   parent_id = rNewDir.parent_id,
                   depth = rNewDir.depth,
                   inode = rNewDir.inode,
                   size = rNewDir.size,
                   blocks = rNewDir.blocks,
                   ctime = rNewDir.ctime,
                   atime = rNewDir.atime,
                   mtime = rNewDir.mtime,
                   uid = rNewDir.uid,
                   gid = rNewDir.gid,
                   perms = rNewDir.perms,
                   path = rNewDir.path,
                   name = rNewDir.name,
                   rec_aggrs = rNewDir.rec_aggrs,
                   local_aggrs = rNewDir.local_aggrs,
                   out_of_sync_time = rNewDir.out_of_sync_time,
                   history_id = rNewDir.history_id,
                   sync_time = rNewDir.sync_time,
                   errors = rNewDir.errors,
                   custom_fs_attrs_id = rNewDir.custom_fs_attrs_id
             WHERE id = rNewDir.id AND volume_id = {{ vol_id }};

            INSERT INTO sf.refresh_aggregates_part_{{ partition_id }}(id, volume_id, depth) VALUES (rNewDir.id, {{ vol_id }}, rNewDir.depth);

            -- Move old job results to history
            WITH old_results AS
                (DELETE FROM sf.job_result_current_part_{{ partition_id }}
                 WHERE fs_entry_id = rNewDir.id AND volume_id = {{ vol_id }}
                    AND ((ctime IS NOT NULL AND ctime != rNewDir.ctime)
                      OR (mtime IS NOT NULL AND mtime != rNewDir.mtime))
                 RETURNING id, fs_entry_id, inode, mtime, ctime, run_time, name_id, result, volume_id, job_id)
            INSERT INTO sf.job_result_history_part_{{ partition_id }}(id, fs_entry_id, inode, mtime, ctime, run_time, name_id, result, volume_id, job_id)
                SELECT * FROM old_results;
        ELSE
            -- The event type was not valid
            raise exception 'invalid event "%"', COALESCE(rEvent.event_type, '<NULL>');
        END IF;
        -- DO NOT PUT ANY CODE HERE: BRANCHES ABOVE CAN RETURN FROM THIS FUNCTION!!!
    -- Process files
    ELSE
        -- Get file info from event
        rNewFile.parent_id = lParentId;
        rNewFile.volume_id = {{ vol_id }};
        rNewFile.inode = rEvent.inode;
        rNewFile.size = rEvent.size;
        rNewFile.blocks = rEvent.blocks;
        rNewFile.nlinks = rEvent.nlinks;
        rNewFile.ctime = rEvent.ctime;
        rNewFile.atime = rEvent.atime;
        rNewFile.mtime = rEvent.mtime;
        rNewFile.uid = rEvent.uid;
        rNewFile.gid = rEvent.gid;
        rNewFile.type = rEvent.type;
        rNewFile.perms = rEvent.perms;
        rNewFile.name = rEvent.name;
        rNewFile.target = rEvent.target;
        rNewFile.sync_time = calculated_sync_time;

        -- Processing ADDED files is no longer here. See: load_events_files_added_vol_{{ vol }}
        -- Processing REMOVED files is no longer here. See: load_events_files_removed_vol_{{ vol }}

        -- Process CHANGED files
        IF (rEvent.event_type = 'CHANGED' OR rEvent.event_type = 'ADDED') AND rCurrentFile.id IS NOT NULL THEN

            IF (rCurrentFile.sync_time IS NOT NULL) AND (calculated_sync_time < rCurrentFile.sync_time) THEN
                RETURN FALSE;
            END IF;

            -- Pull in data from the current row
            rNewFile.id = rCurrentFile.id;
            rNewFile.history_id = COALESCE(rCurrentFile.history_id, nextval('sf.history_object_id_seq'));
            rNewFile.valid = tstzrange(calculated_sync_time, 'infinity', '[)');
            rNewFile.sync_time = calculated_sync_time;
            rNewFile.inode = COALESCE(rNewFile.inode, rCurrentFile.inode);
            rNewFile.size = COALESCE(rNewFile.size, rCurrentFile.size);
            rNewFile.blocks = COALESCE(rNewFile.blocks, rCurrentFile.blocks);
            rNewFile.nlinks = COALESCE(rNewFile.nlinks, rCurrentFile.nlinks);
            rNewFile.ctime = COALESCE(rNewFile.ctime, rCurrentFile.ctime);
            rNewFile.atime = COALESCE(rNewFile.atime, rCurrentFile.atime);
            rNewFile.mtime = COALESCE(rNewFile.mtime, rCurrentFile.mtime);
            rNewFile.uid = COALESCE(rNewFile.uid, rCurrentFile.uid);
            rNewFile.gid = COALESCE(rNewFile.gid, rCurrentFile.gid);
            rNewFile.type = COALESCE(rNewFile.type, rCurrentFile.type);
            rNewFile.perms = COALESCE(rNewFile.perms, rCurrentFile.perms);
            rNewFile.name = COALESCE(rNewFile.name, rCurrentFile.name);
            rNewFile.target = COALESCE(rNewFile.target, rCurrentFile.target);

            historically_significant_change = sf_internal.is_historically_significant_change(rCurrentFile, rEvent.custom_fs_attrs, rNewFile);
            IF historically_significant_change THEN
                IF bHistory THEN
                    PERFORM sf_internal.insert_file_into_history(rCurrentFile, calculated_sync_time);
                END IF;
                rNewFile.history_id = nextval('sf.history_object_id_seq');
            ELSE
                -- only atime is different so we did not put row to history - valid should stay unchanged
                rNewFile.valid = rCurrentFile.valid;
            END IF;

            -- do not report changed stat if nothing changed (or only atime)
            IF (historically_significant_change OR rEvent.custom_fs_attrs IS NOT NULL) THEN
                INSERT INTO process_event_temp_processing_stats VALUES('CHANGED', 'FILE', rNewFile.size - rCurrentFile.size, 1);
            END IF;

            SELECT sf_internal.update_custom_fs_attrs(rCurrentFile.custom_fs_attrs_id, {{ vol_id }}, rEvent.custom_fs_attrs, bHistory) INTO custom_fs_attrs_id;
            rNewFile.custom_fs_attrs_id = custom_fs_attrs_id;

            UPDATE sf.file_current_part_{{ partition_id }}
               SET valid = rNewFile.valid,
                   parent_id = rNewFile.parent_id,
                   inode = rNewFile.inode,
                   size = rNewFile.size,
                   blocks = rNewFile.blocks,
                   ctime = rNewFile.ctime,
                   atime = rNewFile.atime,
                   mtime = rNewFile.mtime,
                   nlinks = rNewFile.nlinks,
                   uid = rNewFile.uid,
                   gid = rNewFile.gid,
                   type = rNewFile.type,
                   perms = rNewFile.perms,
                   name = rNewFile.name,
                   target = rNewFile.target,
                   history_id = rNewFile.history_id,
                   sync_time = rNewFile.sync_time,
                   custom_fs_attrs_id = rNewFile.custom_fs_attrs_id
             WHERE id = rNewFile.id AND volume_id = {{ vol_id }};

            INSERT INTO sf.refresh_aggregates_part_{{ partition_id }}(id, volume_id, depth) VALUES (rNewFile.parent_id, {{ vol_id }}, rEvent.depth - 1);

            -- Move old job results to history
            WITH old_results AS
                (DELETE FROM sf.job_result_current_part_{{ partition_id }}
                 WHERE fs_entry_id = rNewFile.id
                    AND volume_id = {{ vol_id }}
                    AND ((ctime IS NOT NULL AND ctime != rNewFile.ctime)
                      OR (mtime IS NOT NULL AND mtime != rNewFile.mtime))
                 RETURNING id, fs_entry_id, inode, mtime, ctime, run_time, name_id, result, volume_id, job_id)
            INSERT INTO sf.job_result_history_part_{{ partition_id }}(id, fs_entry_id, inode, mtime, ctime, run_time, name_id, result, volume_id, job_id)
                SELECT * FROM old_results;

        ELSE
            -- The event type was not valid
            raise exception 'invalid event "%"', COALESCE(rEvent.event_type, '<NULL>');
        END IF;
        -- DO NOT PUT ANY CODE HERE: BRANCHES ABOVE CAN RETURN FROM THIS FUNCTION!!!
    END IF;

    RETURN TRUE;
END;
$$ LANGUAGE plpgsql VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.gather_remove_dir_event_and_flush_if_needed_vol_{{ vol_id }}(
    dirs sf.dir_current[],
    dir sf.dir_current,
    history BOOLEAN
) RETURNS sf.dir_current[] AS $$
BEGIN
    dirs := dirs || dir;

    IF array_length(dirs, 1) >= 1000 THEN
        PERFORM sf_internal.flush_gathered_remove_dir_events_vol_{{ vol_id }}(dirs, history);
        dirs := ARRAY[]::record[];
    END IF;

    return dirs;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.gather_remove_file_event_and_flush_if_needed_vol_{{ vol_id }}(
    files sf.file_current[],
    file sf.file_current,
    history BOOLEAN
) RETURNS sf.file_current[] AS $$
BEGIN
    files := files || file;

    IF array_length(files, 1) >= 1000 THEN
        PERFORM sf_internal.flush_gathered_remove_file_events_vol_{{ vol_id }}(files, history);
        files := ARRAY[]::record[];
    END IF;

    return files;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.flush_gathered_remove_dir_events_vol_{{ vol_id }}(
    dirs sf.dir_current[],
    history BOOLEAN
) RETURNS VOID AS $$
DECLARE
    entry_ids_to_be_deleted BIGINT[];
    custom_fs_attrs_to_be_deleted BIGINT[];
BEGIN
    SELECT array_agg(d.id), array_agg(d.custom_fs_attrs_id)
    FROM unnest(dirs) AS d
    INTO entry_ids_to_be_deleted, custom_fs_attrs_to_be_deleted;

    INSERT INTO process_event_temp_processing_stats
    SELECT 'REMOVED', 'DIR', COALESCE(SUM(d.size), 0), count(*)
    FROM unnest(dirs) AS d;

    DELETE FROM sf.dir_current_part_{{ partition_id }}
    WHERE id = ANY(entry_ids_to_be_deleted) AND volume_id = {{ vol_id }};

    PERFORM sf_internal.remove_from_extra_tables_vol_{{ vol_id }}(entry_ids_to_be_deleted, custom_fs_attrs_to_be_deleted, history);
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.flush_gathered_remove_file_events_vol_{{ vol_id }}(
    files sf.file_current[],
    history BOOLEAN
) RETURNS VOID AS $$
DECLARE
    entry_ids_to_be_deleted BIGINT[];
    custom_fs_attrs_to_be_deleted BIGINT[];
BEGIN
    SELECT array_agg(f.id), array_agg(f.custom_fs_attrs_id)
    FROM unnest(files) AS f
    INTO entry_ids_to_be_deleted, custom_fs_attrs_to_be_deleted;

    INSERT INTO process_event_temp_processing_stats
    SELECT 'REMOVED', 'FILE', sum(f.size), count(*)
    FROM unnest(files) AS f;

    DELETE FROM sf.file_current_part_{{ partition_id }}
    WHERE id = ANY(entry_ids_to_be_deleted) AND volume_id = {{ vol_id }};

    PERFORM sf_internal.remove_from_extra_tables_vol_{{ vol_id }}(entry_ids_to_be_deleted, custom_fs_attrs_to_be_deleted, history);
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;


CREATE OR REPLACE FUNCTION sf_internal.remove_from_extra_tables_vol_{{ vol_id }}(
    fsEntryIds BIGINT[],
    customFsAttrsIds BIGINT[],
    bHistory BOOLEAN
) RETURNS void AS $$
DECLARE
    customFsAttrsId BIGINT;
BEGIN
    DELETE FROM sf_cron.cron WHERE sf_cron.cron.fs_entry_id = ANY(fsEntryIds);

    FOREACH customFsAttrsId IN ARRAY COALESCE(array_remove(customFsAttrsIds, NULL), ARRAY[]::BIGINT[])
    LOOP
        PERFORM sf_internal.update_custom_fs_attrs(customFsAttrsId, {{ vol_id }}, '{}'::JSONB, bHistory);
    END LOOP;

    -- Move job results to history
    WITH old_job_results AS
    (DELETE FROM sf.job_result_current_part_{{ partition_id }} WHERE fs_entry_id = ANY(fsEntryIds) AND volume_id = {{ vol_id }}
        RETURNING id, fs_entry_id, inode, mtime, ctime, run_time, name_id, result, volume_id, job_id)
    INSERT INTO sf.job_result_history_part_{{ partition_id }}(id, fs_entry_id, inode, mtime, ctime, run_time, name_id, result, volume_id, job_id)
        SELECT * FROM old_job_results WHERE bHistory;

    -- Move tags to history
    WITH old_tags AS
    (DELETE FROM sf.tag_value_current WHERE fs_entry_id = ANY(fsEntryIds) AND volume_id = {{ vol_id }} RETURNING volume_id, fs_entry_id, name_id, value)
    INSERT INTO sf.tag_value_history(volume_id, fs_entry_id, name_id, value) SELECT * FROM old_tags WHERE bHistory;
END;
$$ LANGUAGE plpgsql VOLATILE PARALLEL UNSAFE;
