/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */
-- use partition function to produce continuous averages and values
with averages AS (
  select "volume occupied space" as y,
       avg("volume occupied space") over() as y_bar,
       extract(epoch from run_time) as x,
       avg(extract(epoch from run_time)) over () as x_bar
  FROM sf_reports.stats_history
  WHERE run_time > now() - INTERVAL '{{number_of_days_to_look_back}} days'
  AND "volume name" = '{{volume}}'
  ORDER BY x
),
-- compute linear regression fit, save maximum x and y values
-- handle case when days might be 0 or other infinite div/0 error
slope AS (
  SELECT
  CASE
    WHEN sum((x - x_bar) * (x - x_bar)) != 0
      THEN sum((x - x_bar) * (y - y_bar)) / sum((x - x_bar) * (x - x_bar))
    ELSE 0
  END as slope,
  max(x_bar) as max_xbar,
  max(y_bar) as max_ybar
  FROM averages
),
-- save most recent y value, occupied space, and current capacity to use for drawing slope for future prediction
most_recent as (
  SELECT run_time as base_time,
  "volume occupied space"  as base_point,
  "total capacity"
  FROM sf_reports.stats_current
  WHERE "volume name" = '{{volume}}'
),
-- compute the y intercept of the slope
intercept as (
  select slope,
       max_ybar - max_xbar * slope as intercept
  FROM slope
)
-- now draw the current data points, the curve fit based upon time of the x value, and the capacity of the Volume
-- extrapolate the prediction into the future based upon some future date in time
SELECT TO_TIMESTAMP(x) as run_time,
       (averages.y/(1024.0*1024*1024*1024))::BIGINT AS size_TiB,
       ((intercept.slope*x + intercept.intercept)/(1024.0*1024*1024*1024))::BIGINT as estimated_size_TiB,
       "total capacity" / (1024.0*1024*1024*1024) AS "Volume Capacity TiB"
FROM averages, intercept, most_recent
union
select run_time,
       most_recent.base_point / (1024.0*1024*1024*1024) as size_TiB,
       predict_point(run_time, intercept.slope, intercept.intercept, most_recent.base_time, most_recent.base_point) AS estimated_size_TiB,
       most_recent."total capacity" /(1024.0*1024*1024*1024) as "Volume Capacity TiB"
  FROM generate_series(CURRENT_DATE, CURRENT_DATE + INTERVAL '{{number_of_days_to_look_ahead}} days', INTERVAL '1 day') run_time,
       most_recent,
       intercept

ORDER BY run_time
