/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */

WITH volume AS (
    SELECT id
    FROM sf_volumes.volume
    WHERE name = '{{volume}}'
),
max_file_size AS (
    SELECT MAX(size) AS size
    FROM sf.file_current file
    INNER JOIN sf_volumes.volume vol ON vol.id = file.volume_id
    AND ('{{volume}}' = '*' OR volume_id = (SELECT id FROM volume))
),
calculate_size_unit AS (
    SELECT
        size,
        CASE
            WHEN size IS NULL OR size = 0 THEN 1
            WHEN size < {{number_of_ranges}} THEN size + 1
            ELSE size / {{number_of_ranges}}
        END AS gs_jump,
        CASE WHEN size <= 1 THEN 0
        ELSE FLOOR(LOG(1024, size)) END AS size_power
    FROM max_file_size
),
get_unit_names AS (
    SELECT
        size,
        gs_jump,
        size_power,
        CASE
            WHEN size_power = 0 THEN ''
            WHEN size_power = 1 THEN ' KiB'
            WHEN size_power = 2 THEN ' MiB'
            WHEN size_power = 3 THEN ' GiB'
        END AS unit
    FROM calculate_size_unit
),
ranges AS (
    SELECT
        GENERATE_SERIES(0, size, gs_jump) threshold
    FROM get_unit_names
),
ranges2 AS (
    SELECT
        ROW_NUMBER() OVER() AS range_num,
        threshold AS point1,
        LEAD(threshold) OVER() AS point2
    FROM ranges
    LIMIT {{number_of_ranges}}
),
ranges3 AS (
    SELECT
        ranges2.range_num,
        ranges2.point1,
        CASE WHEN ranges2.range_num = (SELECT MAX(range_num) FROM ranges2) THEN (SELECT size FROM max_file_size)
        ELSE ranges2.point2 - 1 END AS point2,
        CASE WHEN ranges2.range_num = (SELECT MAX(range_num) FROM ranges2) THEN (SELECT size FROM max_file_size)
        ELSE ranges2.point2 END AS point2_to_display,
        unit.size_power AS size_power,
        unit.unit AS unit
    FROM ranges2
    CROSS JOIN get_unit_names unit
)

SELECT
    CONCAT('range ', r.range_num) AS range,
    CONCAT(ROUND(r.point1 / power(1024.0, r.size_power), 2)::TEXT, r.unit) AS range_start,
    CONCAT(ROUND(r.point2_to_display / power(1024.0, r.size_power), 2)::TEXT, r.unit) AS range_end,
    CASE WHEN SUM(f.size) IS NULL THEN 0 ELSE COUNT(*) END AS count,
    CONCAT(ROUND(COALESCE(SUM(f.size)::NUMERIC, 0) /  power(1024.0, r.size_power), 2)::TEXT, r.unit) AS size
FROM ranges3 r
LEFT JOIN sf.file_current f ON (f.size BETWEEN r.point1 AND r.point2)
          AND ('{{volume}}' = '*' OR f.volume_id = (SELECT id FROM sf_volumes.volume WHERE name = '{{volume}}'))
GROUP BY r.range_num, r.point1, r.point2, r.point2_to_display, r.size_power, r.unit
ORDER BY range_num
