/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */
WITH ext_sizes AS (
    -- do not JOIN WITH volumes here, aggregating OVER volume_id is faster than JOIN OVER volume_name
    -- which is longer (text vs int) AND requires a JOIN
    SELECT f.volume_id,
           sf.get_extension(f.name) AS ext,
           COUNT(*) AS count,
           SUM(f.size) AS logical_size,
           SUM(f.blocks * 512 / CASE WHEN f.nlinks > 0 THEN f.nlinks ELSE 1 END) AS physical_size
    FROM sf.file_current f
    GROUP BY volume_id, ext
), ext_sizes_for_all_volumes AS (
-- windows function SUM(size) OVER (PARTITION BY ext)
-- https://www.postgresql.org/docs/current/static/tutorial-window.html
    SELECT volume_id,
           ext,
           logical_size AS logical_size,
           SUM(logical_size) OVER (PARTITION BY ext) AS logical_size_all_volumes_ext,
           physical_size AS physical_size,
           SUM(physical_size) OVER (PARTITION BY ext) AS physical_size_all_volumes_ext,
           count AS count_volume_ext,
           SUM(count) OVER (PARTITION BY ext) AS count_all_volumes_ext
    FROM ext_sizes
), ext_sizes_for_all_volumes_with_row_numbers AS (
    SELECT
        *,
        ROW_NUMBER() OVER (PARTITION BY volume_id ORDER BY logical_size_all_volumes_ext DESC) as seq_num
    FROM ext_sizes_for_all_volumes
)
SELECT v.name AS volume_name,
       ext,
       logical_size_all_volumes_ext AS size_all_volumes_ext,  -- this name is backward compatibility
       logical_size AS size_volume_ext,  -- this name is backward compatibility
       physical_size_all_volumes_ext,
       physical_size,
       count_all_volumes_ext,
       count_volume_ext,
       CASE
            WHEN COALESCE(v.total_capacity, 0) > 0 THEN ROUND(physical_size * 100 / v.total_capacity, 1)
            ELSE NULL
       END AS percent_of_whole_volume
FROM ext_sizes_for_all_volumes_with_row_numbers e
    JOIN sf_volumes.volume v ON e.volume_id=v.id
WHERE seq_num <= 100  -- up to 100 extensions for each volume
ORDER BY size_all_volumes_ext DESC, volume_name, ext
