#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

readonly _REDIS_SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
LOG_FILE="$(basename "${BASH_SOURCE[0]}" '.sh').log"

# shellcheck source=scripts/installation/_common_install.sh
source "${_REDIS_SCRIPT_DIR}/_common_install.sh"

log_and_exit() {
    log "$@"
    exit 1
}

set_redis_opt() {
    local file_path temp_file key val
    temp_file="$(mktemp .sf.redis.augeas.XXXXX)"
    file_path="$(readlink --canonicalize-missing "$1")"
    shift

    cat > "${temp_file}" <<EOF
set /augeas/load/Redis/lens Redis.lns
set /augeas/load/Redis/incl ${file_path}
load
EOF
    {
        while [[ $# -gt 0 ]]; do
            key="$1"
            shift
            val="$1"
            shift
            echo "set /files${file_path}/${key}" "${val}"
        done
        echo save
        echo print /augeas//error
    } >> "${temp_file}"

    augtool --include="${_REDIS_SCRIPT_DIR}" --noautoload < "${temp_file}"
    rm -f -- "${temp_file}"
}

prepare_redis() {
    local listen_address="$1"
    local port="$2"
    local password="$3"
    local redis_dir="$4"
    local username="$5"
    local install_pkgs="$6"
    local config_file="${redis_dir}/redis.conf"
    local logfile="${redis_dir}/redis.log"
    local pidfile="${redis_dir}/redis.pid"

    if [[ "${install_pkgs}" = "true" ]]; then
        pkg_install sf-redis
    fi
    mkdir --parents "${redis_dir}"
    chown --recursive "${username}" "${redis_dir}"
    cp /opt/starfish/redis/conf/redis.conf.example "${config_file}"
    set_redis_opt "${config_file}" \
        always-show-logo no \
        bind "${listen_address}" \
        daemonize no \
        dir "${redis_dir}" \
        logfile "${logfile}" \
        pidfile "${pidfile}" \
        port "${port}" \
        protected-mode yes \
        requirepass "${password}"
}

check_non_empty_param() {
    local name="$1"
    local val="$2"

    if [[ -z "${val}" ]]; then
        log "--${name} has to be specified"
        exit 2
    fi
}

main() {
    fail_if_internal_script_run_directly

    local listen_address="127.0.0.1"
    local port="" password="" redis_dir="" username=""
    local install_pkgs="true"
    local args=( "$@" )

    while [[ $# -gt 0 ]]; do
        case $1 in
        --log-file)
            shift
            LOG_FILE="$1"
            ;;
        --listen-address)
            shift
            listen_address="$1"
            ;;
        --port)
            shift
            port="$1"
            ;;
        --password)
            shift
            password="$1"
            ;;
        --redis-dir)
            shift
            redis_dir="$1"
            ;;
        --username)
            shift
            username="$1"
            ;;
        --no-install-pkgs)
            install_pkgs="false"
            ;;
        *)
            break
            ;;
        esac
        shift
    done
    export LOG_FILE
    log "Script called with arguments: $0 " ${args[@]+"${args[@]}"}

    if (( $# != 0 )); then
        log "Unknown parameters specified: $*"
        exit 2
    fi

    check_non_empty_param listen_address "${listen_address}"
    check_non_empty_param port "${port}"
    check_non_empty_param password "${password}"
    check_non_empty_param redis-dir "${redis_dir}"
    check_non_empty_param username "${username}"

    fail_if_not_root

    prepare_redis "${listen_address}" "${port}" "${password}" "${redis_dir}" "${username}" "${install_pkgs}"
}

sourced() {
    [[ "${BASH_SOURCE[0]}" != "${0}" ]]
}

if ! sourced; then
    main "$@"
fi
