#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

SCRIPTS_INSTALLATION_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
readonly SCRIPTS_INSTALLATION_DIR

# shellcheck source=scripts/installation/_utils.sh
source "${SCRIPTS_INSTALLATION_DIR}/_utils.sh"
# shellcheck source=scripts/installation/_ini_file.sh
source "${SCRIPTS_INSTALLATION_DIR}/_ini_file.sh"
# shellcheck source=scripts/installation/_secret_key.sh
source "${SCRIPTS_INSTALLATION_DIR}/_secret_key.sh"
# shellcheck source=scripts/installation/_install_params.sh
source "${SCRIPTS_INSTALLATION_DIR}/_install_params.sh"

locate_keyfile() {
    local cfg_keyfile="${STARFISH_ETC_DIR}/license"

    if [[ -e "${cfg_keyfile}" ]]; then
        log "License already installed: ${cfg_keyfile}"
        echo "${cfg_keyfile}"
        return
    fi

    local keyfile_pattern="*.keyfile"
    local keyfile_dir num_of_keyfiles license_file

    keyfile_dir="$(readlink --canonicalize-missing "$1")"
    license_file="$(find "${keyfile_dir}" -name "${keyfile_pattern}")"

    if [ -z "${license_file}" ]; then
        log <<_EOF
License keyfile (${keyfile_pattern}) not found in ${keyfile_dir}
If you put the license file in this directory, rename it to a filename ending with .keyfile
_EOF
        exit 1
    fi

    num_of_keyfiles="$(echo "${license_file}" | wc -l)"
    if [ "${num_of_keyfiles}" -gt 1 ]; then
        log <<_EOF
Multiple keyfiles found in ${keyfile_dir}:
${license_file}
_EOF
        exit 1
    fi
    log "License keyfile found: ${license_file}"
    echo "${license_file}"
}

fail_if_keyfile_not_found() {
    locate_keyfile "${SCRIPTS_INSTALLATION_DIR}/.." > /dev/null
}

fail_if_selinux_enforcing() {
    if command_exists getenforce; then
        # On Ubuntu selinux is not installed by default, and the check is not 100% bullet-proof,
        # because getenforce is in a separate package selinux-utils which may not be installed
        local selinux_mode
        selinux_mode=$(getenforce)
        if [ "${selinux_mode}" = "Enforcing" ]; then
            log "You must set SELINUX to Permissive or Disabled mode"
            exit 1
        fi
   fi
}

fail_if_no_service_config_file() {
    local service_conf_file
    service_conf_file=$(get_service_conf_file)

    file_exists "${service_conf_file}" || {
        log "Configuration file ${service_conf_file} does not exist. Run install-Starfish.sh first"
        exit 1
    }
}

pre_verifications_starfish() {
    fail_if_not_root
    fail_if_selinux_enforcing
    fail_if_unknown_distribution
    fail_if_keyfile_not_found
}

input_not_empty_value() {
    local message="$1"
    local loop=0
    # a primitive infinite loop checker
    local max_loop_count=1000
    local value=""

    while [[ -z "${value}" && "${loop}" -lt "${max_loop_count}" ]]; do
        loop="$((loop+1))"
        read -r -p "${message}" value
    done
    if [[ "${loop}" -ge "${max_loop_count}" ]]; then
        >&2 echo "Looped ${max_loop_count} in input_not_empty_value, something's wrong"
        exit 1
    fi
    echo "${value}"
}

input_with_default_value() {
    local message="$1"
    local default_value="$2"
    local value=""

    read -r -p "${message} [default=${default_value}]: " value
    if [[ -z "${value}" ]]; then
        value="${default_value}"
    fi
    echo "${value}"
}

prompt_to_remove_path() {
    local path="$1"
    local remove="no"

    remove="$(input_not_empty_value "${path} already exists. Do you want to remove it? Type yes or no:")"
    if [[ "${remove}" == "yes" ]] || [[ "${remove}" == "y" ]]; then
        log "Removing ${path}"
        rm -rf -- "${path}"
    else
        log "${path} already exists. Run the script again with a non-existing directory and it will be automatically created."
        exit 1
    fi
}

try_run_with_fallback_on_ancestors() {
    # If path doesn't exist, it's sometimes useful to fall back on its ancestors.
    # For instance, "df /opt/not-existing-dir" fails, but "df /opt" gives expected result.
    local func="$1"
    local path="$2"

    local parent result
    while true; do
        if result="$("${func}" "${path}")"; then
            echo "${result}"
            return 0
        else
            parent="$(dirname "${path}")"
            if [[ "${parent}" == "${path}" ]]; then
                return 1
            fi
            path="${parent}"
        fi
    done
}

_df_get_free_disk_space() {
    local path="$1"

    # Columns of df --portability --block-size=1: Filesystem, 1-blocks, Used, Available, Capacity, Mounted on
    df --portability --block-size=1 "${path}" 2> /dev/null | awk '{ print $4 }' | tail -1
}


get_free_disk_space() {
    local directory="$1"
    local available=""

    if ! available="$(try_run_with_fallback_on_ancestors _df_get_free_disk_space "${directory}")"; then
        log "Can't get free disk space for directory: ${directory}"
        return 1
    fi
    log "Free disk space for ${directory}: ${available} bytes"
    echo "${available}"
}

fail_if_not_enough_space_for_db() {
    local db_path="$1"
    local threshold="$2"
    local free_space
    free_space=$(get_free_disk_space "${db_path}")
    if [ "${free_space}" -lt "${threshold}" ]; then
        log "Not enough free space for Starfish database in ${db_path}. There are ${free_space} bytes free, but at least ${threshold} is needed."
        exit 1
    fi
}

prompt_for_db_storage_type() {
    local cmdline_value="$1"
    local default_value="$2"
    local use_defaults="$3"

    if ! empty_string "${cmdline_value}"; then
        echo "${cmdline_value}"
    else
        if [ "${use_defaults}" == "yes" ]; then
            echo "${default_value}"
        else
            local message
            read -r -d '' message <<_EOF
To optimize database performance Starfish needs to know the storage type on which database is placed.
    1. HDD
    2. RAID (HDD)
    3. SSD
    4. NVMe
Please enter storage type
_EOF
        while true; do
            local storage_type
            storage_type=$(input_with_default_value "${message}" "${default_value}" \
                | tr '[:upper:]' '[:lower:]')
            case "${storage_type}" in
                1|hdd)  echo "hdd" ;;
                2|raid|"raid (hdd)") echo "raid" ;;
                3|ssd)  echo "ssd" ;;
                4|nvme) echo "nvme" ;;
                *)
                    log "Invalid value: ${storage_type}, please enter a valid number."
                    continue;
            esac;
            break
        done
        fi;
    fi;
}

is_positive_integer() {
    local value="$1"
    [[ "${value}" =~ ^[\0-9]+$ ]] && (( value > 0))
}

prompt_for_positive_integer() {
    local message="$1"
    local cmdline_value="$2"
    local default_value="$3"
    local use_default="$4"

    if ! empty_string "${cmdline_value}"; then
        echo "${cmdline_value}"
    else
        if [ "${use_default}" == "yes" ]; then
            echo "${default_value}"
        else
            while true; do
                local value
                value=$(input_with_default_value "${message}" "${default_value}")
                if is_positive_integer "${value}"; then
                    echo "${value}"
                    break
                else
                    log "Invalid value: ${value}, please provide positive integer."
                    continue
                fi
            done
        fi
    fi
}

prompt_for_param() {
    local message="$1"
    local cmdline_value="$2"
    local default_value="$3"
    local use_default="$4"

    if ! empty_string "${cmdline_value}"; then
        echo "${cmdline_value}"
    else
        if [ "${use_default}" == "yes" ]; then
            echo "${default_value}"
        else
            input_with_default_value "${message}" "${default_value}"
        fi
    fi
}

prompt_for_yes_or_no() {
    local message="$1"
    local cmdline_value="$2"
    local default_value="$3"
    local use_default="$4"
    local third_option="${5:-""}"

    if ! empty_string "${cmdline_value}"; then
        echo "${cmdline_value}"
    else
        if [ "${use_default}" == "yes" ]; then
            echo "${default_value}"
        else
            local value=""
            while true;
            do
                local prompt="type yes or no"
                if ! empty_string "${third_option}"; then
                    prompt="type yes, no or ${third_option}"
                fi
                read -r -p "${message}; ${prompt} [default=${default_value}]: " value
                if [ -z "${value}" ]; then
                    value="${default_value}"
                elif echo "${value}" | grep --ignore-case --quiet "^y"; then
                    value="yes"
                elif echo "${value}" | grep --ignore-case --quiet "^n"; then
                    value="no"
                elif ! empty_string "${third_option}" && echo "${value}" | grep --ignore-case --quiet "^${third_option:0:1}"; then
                    # *** WARNING *** We assume here that the third option does not start with 'y' or 'n'
                    # let's postpone handling it until someone wants to use such, which is very unlikely
                    value="${third_option}"
                fi
                if ! empty_string "${value}"; then
                    break
                fi
            done
            echo "${value}"
        fi
    fi
}

choose_default_value_if_no_cli_value() {
    local cmdline_value="$1"
    local default_value="$2"

    if ! empty_string "${cmdline_value}"; then
        echo "${cmdline_value}"
    else
        echo "${default_value}"
    fi
}

fail_if_internal_script_run_directly() {
    if [ -z "${ALLOW_FOR_INTERNAL_SCRIPT+x}" ]; then
        log "$0 shouldn't be run directly. Use install-* scripts instead."
        exit 2
    fi
}

remove_trailing_empty_lines_from_file() {
    local file_path="$1"
    sed --in-place --expression :a --expression '/^\n*$/{$d;N;};/\n$/ba' "${file_path}"
}

get_starfish_cron_file() {
    echo "/etc/cron.d/starfish"
}

set_convenient_umask() {
    umask 0022
}

is_address_resolvable() {
    local address=$1
    local ip_address

    ip_address=$(getent ahosts "${address}" || true)
    test ! -z "${ip_address}"
}

is_proxy_set() {
    [[ $(env | grep --count --extended-regexp --ignore-case "(all|http[s]?)_proxy") -gt 0 ]]
}

is_correct_format_email_address() {
    local email=$1
    email=$(echo "${email}" | grep "@" || true)
    test ! -z "${email}"
}

is_starfish_running() {
    # to make upgrade tests pass either with sfclient or sf
    local sf_client
    sf_client="$(command -v sfclient 2>&1 || command -v sf 2>&1)"
    ${sf_client} check-config &>> "${LOG_FILE}"
}

systemd_restart_starfish() {
    systemctl restart "$(get_systemd_sf_target_name)"
}

supervisord_restart_starfish() {
    supervisorctl stop "$(get_supervisord_sf_group_name)*"
    supervisorctl start "$(get_supervisord_sf_group_name)*"
}

start_starfish() {
    log "Starting Starfish services"

    # restart instead of start as it happens in tests (for not-so-clear reasons) that sf:config starts with
    # "default" config, listening on localhost. Need to restart it so that it reads the new config
    run_func_for_service_manager "$(get_systemd_sf_target_name)" systemd_restart_starfish sf: supervisord_restart_starfish

    service sf-agent start
    log "Waiting for Starfish services to be running (this can take more than a minute)"
    wait_for is_starfish_running 90 || {
        log "Starfish is not running. See ${LOG_FILE} for more details."
        exit 1
    }
}

is_starfish_supervisor_group_stopped() {
    local running_count

    if ! command -v supervisorctl; then
        return 0
    fi
    running_count=$(supervisorctl status | grep --count "$(get_supervisord_sf_group_name).*RUNNING")
    if [[ "${running_count}" -eq 0 ]]; then
        return 0
    fi
    return 1
}

is_starfish_systemd_target_stopped() {
    systemd_all_unit_members_stopped "$(get_systemd_sf_target_name)"
}

is_starfish_agent_stopped() {
    ! service sf-agent status
}

is_starfish_stopped() {
    is_starfish_systemd_target_stopped && is_starfish_supervisor_group_stopped && is_starfish_agent_stopped
}

systemd_stop_starfish() {
    systemctl stop "$(get_systemd_sf_target_name)"
}

supervisord_stop_starfish() {
    supervisorctl stop "$(get_supervisord_sf_group_name)*"
}

stop_starfish() {
    service sf-agent stop

    run_func_for_service_manager "$(get_systemd_sf_target_name)" systemd_stop_starfish sf: supervisord_stop_starfish
}

ubuntu_add_signing_key() {
    local key_url="$1"
    local keyring="$2"
    local fingerprint="$3"
    local gpg_out

    run_with_sudo_if_not_root apt-get install -y curl gnupg2 ca-certificates lsb-release ubuntu-keyring

    curl_silent_to_stdout "${key_url}" | gpg --dearmor | run_with_sudo_if_not_root tee "${keyring}" > /dev/null

    gpg_out=$(gpg --quiet --import --import-options show-only "${keyring}")

    if [[ $(echo "${gpg_out}" | grep --count "${fingerprint}") -ne 1 ]]; then
        >&2 echo "Wrong fingerprint of signing key in ${keyring}, should be ${fingerprint}!"
        >&2 echo "${gpg_out}"
        exit 1
    fi
}

ubuntu_add_apt_source() {
    local filename="$1"
    local keyring="$2"
    local url="$3"
    local component="$4"
    local distro

    distro="${5:-$(map_to_ubuntu_distro)}"

    echo "deb [signed-by=${keyring}] ${url} ${distro} ${component}" | run_with_sudo_if_not_root tee "${filename}" > /dev/null
}
