#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

# shellcheck disable=SC2145

set -euo pipefail

usage() {
    echo >&2 "Send SF usage reports via e-mail"
    echo >&2 "Usage: $0 [--dry-run] warning_email [warning_email2]... [--errors-only error_email [error_email2]...]"
    echo >&2 "  --dry-run: do not send anything, only print what would be sent and where."
    echo >&2 "Note: any environment variables passed to this script will override those defined in"
    echo >&2 "$(get_sysconfig_file), e.g."
    echo >&2 "OBFUSCATE_CONFIDENTIAL_INFO_ENABLED=true $0"
    echo >&2 "will run the script with confidential obfuscation enabled regardless of the value in the sysconfig file."
}

fatal() {
    echo "$1" >&2
    exit 1
}

SFHOME=${SFHOME:-/opt/starfish}
SF=${SFHOME}/bin/client

# shellcheck source=scripts/installation/_utils.sh
source "${SFHOME}/data/installation/_utils.sh"

REPORT_TMP_DIR="$(mktemp --directory --tmpdir="${SFHOME}/tmp" -t tmp_sf_report_stats.XXXXXXXXXX)"

REPORT_FILE="${REPORT_TMP_DIR}/report"
REPORT_FILE_ERROR="${REPORT_TMP_DIR}/report_error"
REPORT_FILE_WARN="${REPORT_TMP_DIR}/report_warn"

REPORT_JSON_SUMMARY_FILENAME="sf_report.json"
REPORT_JSON_SUMMARY_TMPFILE="${REPORT_TMP_DIR}/${REPORT_JSON_SUMMARY_FILENAME}"

REPORT_DATE="$(date +'%Y/%m/%d %H:%M:%S')"
REPORT_FILE_DATE="$(date +'%Y%m%d-%H%M%S')"
REPORT_JSON_TAR_FILENAME="sf_report-${REPORT_FILE_DATE}.tar"
REPORT_JSON_TAR_TMPFILE="${REPORT_TMP_DIR}/${REPORT_JSON_TAR_FILENAME}"

REPORT_DRY_OUTPUT_FILE="${REPORT_TMP_DIR}/dry_run.out"

REPORT_TARGET_DIR="${SFHOME}/log/report_stats"

CENTOS_STARFISH_REPO_FILE=/etc/yum.repos.d/starfish.repo
UBUNTU_STARFISH_REPO_FILE=/etc/apt/sources.list.d/starfish.list

EXIT_CODE=0

finish() {
    if [[ "${KEEP_TMP}" != true ]]; then
        rm -rf -- "${REPORT_TMP_DIR}"
    else
        echo "Temp directory kept on demand: ${REPORT_TMP_DIR}"
    fi
}
trap finish EXIT

centos_get_starfish_repo_url() {
    if [[ -f "${CENTOS_STARFISH_REPO_FILE}" ]]; then
        grep "baseurl" "${CENTOS_STARFISH_REPO_FILE}" | cut -d'=' -f2
    fi
}

ubuntu_get_starfish_repo_url() {
    if [[ -f "${UBUNTU_STARFISH_REPO_FILE}" ]]; then
        cut -d' ' -f2 <"${UBUNTU_STARFISH_REPO_FILE}"
    fi
}

get_starfish_repo_url() {
    run_func_for_distro centos_get_starfish_repo_url ubuntu_get_starfish_repo_url
}

starfish_repo_url() {
    local repo_url
    repo_url="$(get_starfish_repo_url)"
    echo "${repo_url:-"(repository not found)"}"
}

_cat_file_if_exists() {
    local path="$1"
    if [[ -r "${path}" ]]; then
        echo "------ ${path} ------"
        cat "${path}"
    fi
}

starfish_cron_info() {
    echo "------ /etc/cron.d/starfish ------"
    cat /etc/cron.d/starfish
    echo "------ crontab of root ------"
    if [[ "$CONFIDENTIAL_OPT" != --confidential ]]; then
        crontab -u root -l || true # non-zero exit code if empty
    else
        echo "Contents of 'crontab -u root' are not displayed in confidential mode"
    fi
    echo "------ crontab of postgres ------"
    crontab -u postgres -l || true
    _cat_file_if_exists /etc/sysconfig/starfish # RedHat/CentOS
    _cat_file_if_exists /etc/default/starfish   # Ubuntu
}

retain_only_us_ascii_printable_chars_if_mailx() {
    local mail_path mail_basename
    mail_path=$(command -v "mail" 2>/dev/null || echo "/mail-not-found")
    mail_basename=$(basename "$(readlink --canonicalize-missing "${mail_path}")")

    if [[ "${mail_basename}" == mailx ]]; then
        # Remove carriage returns and all non-printable characters so that mailx doesn't set MIME type to
        # application/octet-stream which is interpreted as attachment by mail clients (https://access.redhat.com/solutions/1136493)
        tr --delete '\015' | tr --complement '[:graph:][:blank:]\n' '[?*]'
    else
        cat
    fi
}

maybe_ignore_lines() {
    if [[ -n "${DIAGNOSTIC_REPORTS_IGNORED_ERRORS_PATTERN}" ]]; then
        grep --extended-regexp --invert-match "${DIAGNOSTIC_REPORTS_IGNORED_ERRORS_PATTERN}"
    else
        cat
    fi
}

report_error_if_not_ignored() {
    local cmd="$1"
    local exit_code="$2"
    local file="$3"

    declare -a ignored_exit_codes
    # shellcheck disable=SC2034
    local limited_license_error="37"
    ignored_exit_codes=("${limited_license_error}")

    # shellcheck disable=SC2076
    # if current exit_code not in ignored_exit_codes array - report an error
    if [[ ! " ${ignored_exit_codes[*]} " =~ " ${exit_code} " ]]; then
        echo "ERROR: ${cmd} returned ${exit_code}" >>"${file}"
    fi
}

append_report_to_file() {
    local file="$1"
    shift
    local cmd=("$@")

    {
        echo "------------------------------------------------------------------"
        echo "${cmd[@]}"
        echo "------------------------------------------------------------------"
    } >>"${file}"
    "${cmd[@]}" |& retain_only_us_ascii_printable_chars_if_mailx >>"${file}" 2>&1 ||
        report_error_if_not_ignored "${cmd[*]}" "$?" "${file}"
}

init_json_summary_report() {
    echo "{" \
        "\"timestamp\": \"${REPORT_DATE}\", " \
        "\"reports\": [" >>"${REPORT_JSON_SUMMARY_TMPFILE}"
}

init_attachment() {
    tar -cf "${REPORT_JSON_TAR_TMPFILE}" --files-from /dev/null
    init_json_summary_report
}

finalize_json_summary_report() {
    truncate --size=-2 "${REPORT_JSON_SUMMARY_TMPFILE}"
    echo "]," \
        "\"username\": \"$(get_username)\" " \
        "}" >>"${REPORT_JSON_SUMMARY_TMPFILE}"

    local exit_code=0
    if command -v python3 &>/dev/null; then
        PYTHON_BIN=python3
    else
        PYTHON_BIN=python
    fi

    "${PYTHON_BIN}" -c "import json; p='${REPORT_JSON_SUMMARY_TMPFILE}'; \
        f=open(p); j=json.load(f); f.close(); \
        f=open(p, \"w\"); json.dump(j, f, indent=2); f.close();" || exit_code=$?
    if [[ "${exit_code}" -ne 0 ]]; then
        echo "Warning: ${REPORT_JSON_SUMMARY_FILENAME} is malformed"
    fi
    tar -rvf "${REPORT_JSON_TAR_TMPFILE}" -C "${REPORT_TMP_DIR}" "${REPORT_JSON_SUMMARY_FILENAME}" >/dev/null
}

finalize_attachment() {
    finalize_json_summary_report
    gzip "${REPORT_JSON_TAR_TMPFILE}"
    REPORT_JSON_ATTACHMENT_FILE="${REPORT_TMP_DIR}/sf_report-$(get_username)-${REPORT_FILE_DATE}.tar.gz"
    mv "${REPORT_JSON_TAR_TMPFILE}.gz" "${REPORT_JSON_ATTACHMENT_FILE}"
}

add_report_to_attachment() {
    local report_key="$1"
    shift
    local local_report_file="${report_key}.json"
    local local_error_file="${report_key}.err"
    local cmd=("$@")
    local exit_code=0
    "${cmd[@]}" >"${REPORT_TMP_DIR}/${local_report_file}" 2>"${REPORT_TMP_DIR}/${local_error_file}" || exit_code=$?
    if [[ "${exit_code}" -eq 0 ]]; then
        tar -rvf "${REPORT_JSON_TAR_TMPFILE}" -C "${REPORT_TMP_DIR}" "${local_report_file}" >/dev/null
        rm "${REPORT_TMP_DIR}/${local_error_file}"
        echo "{\"report\": \"${report_key}\", \"cmd\": \"${cmd[@]}\", \"exit_code\": ${exit_code}, " \
            "\"contents_file\": \"${local_report_file}\"}," \
            >>"${REPORT_JSON_SUMMARY_TMPFILE}"
        echo "${REPORT_TMP_DIR}/${local_report_file}"
    else
        tar -rvf "${REPORT_JSON_TAR_TMPFILE}" -C "${REPORT_TMP_DIR}" "${local_error_file}" >/dev/null
        rm "${REPORT_TMP_DIR}/${local_report_file}"
        echo "{\"report\": \"${report_key}\", \"cmd\": \"${cmd[@]}\", \"exit_code\": ${exit_code}, " \
            "\"error_file\": \"${local_error_file}\"}," \
            >>"${REPORT_JSON_SUMMARY_TMPFILE}"
    fi
}

report_stat() {
    append_report_to_file "${REPORT_FILE}" "$@"
}

report_stat_json() {
    add_report_to_attachment "$@" --json
    shift
    report_stat "$@"
}

report_stat_json_details() {
    add_report_to_attachment "$@" --json
    shift
    report_stat "$@" --details
}

report_stat_json_reused() {
    local local_report_file
    local_report_file="$(add_report_to_attachment "$@" --json)"
    shift
    if [[ -n "${local_report_file}" ]]; then
        report_stat "$@" "--input" "${local_report_file}"
    else
        report_stat "$@"
    fi
}

obfuscate_confidential_redash_status_logs() {
    # obfuscate hostname confidential data in REPORT_FILE redash logs
    sed -i 's/ [^ ]* postmaster/ postmaster/g' "${REPORT_FILE}" 2>/dev/null
    sed -i 's/ [^ ]* systemd/ systemd/g' "${REPORT_FILE}" 2>/dev/null
}

redash_status() {
    local confidential_mode=$1

    add_report_to_attachment redash "${SFHOME}/bin/redash" manage status >/dev/null
    report_stat "${SFHOME}/bin/redash_status"
    if [[ "${confidential_mode}" == --confidential ]]; then
        obfuscate_confidential_redash_status_logs
    fi
}

collect_logs() {
    append_report_to_file "${REPORT_FILE_ERROR}" "${SF}" collect-logs --days 1 --error --maxlen 1M --print
    append_report_to_file "${REPORT_FILE_WARN}" "${SF}" collect-logs --days 1 --warn --maxlen 1M --print
}

remove_duplicate_emails_and_sort() {
    local unique_warn_emails unique_error_emails

    # remove duplicates from that same array
    unique_warn_emails="$(echo "${WARN_EMAILS[@]+"${WARN_EMAILS[@]}"}" | tr ' ' '\n' | sort --unique | tr '\n' ' ')"
    unique_error_emails="$(echo "${ERROR_EMAILS[@]+"${ERROR_EMAILS[@]}"}" | tr ' ' '\n' | sort --unique | tr '\n' ' ')"

    # make an array out of emails again
    IFS=' ' read -ra WARN_EMAILS <<<"${unique_warn_emails:-""}"
    IFS=' ' read -ra ERROR_EMAILS <<<"${unique_error_emails:-""}"

    # if email is present in both arrays - include it in more detailed emails
    declare -a temp_error_array
    local is_duplicated=""

    for i in "${!ERROR_EMAILS[@]}"; do
        is_duplicated=false
        for j in "${!WARN_EMAILS[@]}"; do
            if [[ ${ERROR_EMAILS[i]} == "${WARN_EMAILS[j]}" ]]; then
                is_duplicated=true
            fi
        done
        if [[ ${is_duplicated} == false ]]; then
            temp_error_array+=("${ERROR_EMAILS[i]}")
        fi
    done
    ERROR_EMAILS=("${temp_error_array[@]+${temp_error_array[@]}}")
}

remove_email_to_starfish_corporation() {
    declare -a temp_warning_array

    # if we just 'unset' one email address - an empty element in an array would persist
    # ${arr[@]+"${arr[@]}"} construct is used as a workaround to passing empty arrays
    # https://stackoverflow.com/questions/7577052/bash-empty-array-expansion-with-set-u
    for email in "${WARN_EMAILS[@]+"${WARN_EMAILS[@]}"}"; do
        if [[ ${email} != "${STARFISH_STATUS_MAIL}" ]]; then
            temp_warning_array+=("${email}")
        fi
    done
    WARN_EMAILS=("${temp_warning_array[@]+${temp_warning_array[@]}}")
}

collect_email_addresses() {

    declare -a warn_emails_sysconf
    declare -a error_emails_sysconf

    IFS=' ' read -ra warn_emails_sysconf <<<"${DIAGNOSTIC_REPORTS_EXTRA_EMAILS}"
    IFS=' ' read -ra error_emails_sysconf <<<"${DIAGNOSTIC_REPORTS_EXTRA_EMAILS_ERRORS_ONLY}"

    WARN_EMAILS=("${warn_emails_sysconf[@]+"${warn_emails_sysconf[@]}"}" "${WARN_EMAILS[@]+"${WARN_EMAILS[@]}"}")
    ERROR_EMAILS=("${error_emails_sysconf[@]+"${error_emails_sysconf[@]}"}" "${ERROR_EMAILS[@]+"${ERROR_EMAILS[@]}"}")

    remove_duplicate_emails_and_sort

    # to ensure compatibility with previous versions
    remove_email_to_starfish_corporation

    FROM_EMAIL=${DIAGNOSTIC_REPORTS_FROM_EMAIL}
}

report_contains_errors() {
    grep --extended-regexp --invert-match '(DIAGNOSTIC_REPORTS_EXTRA_EMAILS_ERRORS_ONLY|DIAGNOSTIC_REPORTS_IGNORED_ERRORS_PATTERN|#.*\(ERROR\))' "${REPORT_FILE}" |
        maybe_ignore_lines |
        grep --extended-regexp '(ERROR|FATAL|^ERR |^CRI | ERR | CRI |Traceback)' >/dev/null
}

get_username() {
    local exit_code=0
    username="$(jq -er '.license.out_data.license.username' "${REPORT_TMP_DIR}/check_config.json")" || exit_code=$?
    if [[ "${exit_code}" -eq 0 ]]; then
        echo "${username}"
    else
        echo ""
    fi
}

send_emails() {
    local host username subject mail_args

    if [[ "${OBFUSCATE_CONFIDENTIAL_INFO_ENABLED}" == true ]]; then
        host=confidential
    else
        host="$(get_hostname)"
    fi

    subject="SF usage report #$(get_username) @${host} ${REPORT_DATE}"

    if report_contains_errors; then
        subject="(ERROR) $subject"
        EXIT_CODE=99
    fi

    mail_args=(-s "${subject}")
    if [[ -n "${FROM_EMAIL}" ]]; then
        mail_args+=(-r "${FROM_EMAIL}")
    fi

    local print_email=false
    # ${arr[@]+"${arr[@]}"} construct is used as a workaround to passing empty arrays
    # https://stackoverflow.com/questions/7577052/bash-empty-array-expansion-with-set-u
    if [[ "${#ERROR_EMAILS[@]}" -gt 0 ]]; then
        echo "Error report would be sent to:" "${ERROR_EMAILS[@]+"${ERROR_EMAILS[@]}"}" >>"${REPORT_DRY_OUTPUT_FILE}"
        print_email=true
        if [[ "${DRY_RUN}" != true ]]; then
            echo "Report mail being sent to ${ERROR_EMAILS[@]}"
            mail "${mail_args[@]}" "${ERROR_EMAILS[@]}" <"${REPORT_FILE_ERROR}"
        fi
    fi
    if [[ "${#WARN_EMAILS[@]}" -gt 0 ]]; then
        echo "Warning report would be sent to:" "${WARN_EMAILS[@]+"${WARN_EMAILS[@]}"}" >>"${REPORT_DRY_OUTPUT_FILE}"
        print_email=true
        if [[ "${DRY_RUN}" != true ]]; then
            echo "Report mail being sent to ${WARN_EMAILS[@]}"
            mail "${mail_args[@]}" "${WARN_EMAILS[@]}" <"${REPORT_FILE_WARN}"
        fi
    fi
    if [[ "${SENDING_DIAGNOSTIC_REPORTS_TO_STARFISH_ENABLED}" == true ]]; then
        echo "Report would be sent to Starfish: ${STARFISH_STATUS_MAIL}" >>"${REPORT_DRY_OUTPUT_FILE}"
        print_email=true
    fi
    if [[ ${print_email} == true ]]; then
        # shellcheck disable=SC2129
        echo "Email subject would be: ${subject}" >>"${REPORT_DRY_OUTPUT_FILE}"
        # cat only REPORT_FILE_WARN, as it should be superset of REPORT_FILE_ERROR
        echo "Report contents would be:" >>"${REPORT_DRY_OUTPUT_FILE}"
        cat "${REPORT_FILE_WARN}" >>"${REPORT_DRY_OUTPUT_FILE}"
    else
        echo "No report would be sent because neither warning nor errors-only reports emails were specified," \
            "and sending to Starfish is not switched on in sysconfig." >>"${REPORT_DRY_OUTPUT_FILE}"
    fi

    if [[ "${SENDING_DIAGNOSTIC_REPORTS_TO_STARFISH_ENABLED}" == true ]]; then
        if [[ "${SENDING_DIAGNOSTIC_JSON_TO_STARFISH_ENABLED}" == true ]]; then
            echo "JSON report would be attached to the email sent to Starfish." >>"${REPORT_DRY_OUTPUT_FILE}"
            if [[ "${KEEP_TMP}" == true ]]; then
                echo "JSON report contents can be found in ${REPORT_JSON_ATTACHMENT_FILE}" >>"${REPORT_DRY_OUTPUT_FILE}"
            else
                echo "JSON report contents will not be available, as the temp folder is removed at the end." \
                    >>"${REPORT_DRY_OUTPUT_FILE}"
            fi
            attach_file_option="$(get_attachment_option)"
            if [[ "${attach_file_option}" ]]; then
                mail_args+=("${attach_file_option}" "${REPORT_JSON_ATTACHMENT_FILE}")
            else
                echo "No JSON report would be attached: unable to detect mail option to attach file." \
                    >>"${REPORT_DRY_OUTPUT_FILE}"
            fi
        else
            echo "No JSON report would be attached as it is switched off in sysconfig." >>"${REPORT_DRY_OUTPUT_FILE}"
        fi

        if [[ "${DRY_RUN}" != true ]]; then
            echo "Report mail being sent to ${STARFISH_STATUS_MAIL}"
            mail "${mail_args[@]}" "${STARFISH_STATUS_MAIL}" <"${REPORT_FILE_WARN}"
        fi
    fi

    if [[ "${DRY_RUN}" == true ]]; then
        cat "${REPORT_DRY_OUTPUT_FILE}"
    fi
}

parse_cmdline_args() {

    while [[ $# -gt 0 && $1 == -* ]]; do
        key=${1:-""}
        case ${key} in
        -h | --help)
            usage
            exit 0
            ;;
        --dry-run)
            DRY_RUN=true
            shift
            ;;
        --no-redash)
            WITH_REDASH=false
            shift
            ;;
        --errors-only)
            break
            ;;
        *)
            usage
            fatal "Unknown option: $1"
            exit 2
            ;;
        esac
    done

    while [[ $# -gt 0 && $1 != -* ]]; do
        WARN_EMAILS+=("$1")
        shift
    done

    while [[ $# -gt 0 ]]; do
        case $1 in
        "-e" | "--errors-only")
            shift
            while [[ $# -gt 0 && $1 != -* ]]; do
                ERROR_EMAILS+=("$1")
                shift
            done
            ;;
        *)
            usage
            fatal "Unknown option: $1"
            break
            ;;
        esac
    done
}

# Look at env var definitions in the sysconfig file and take only those that are not defined in the env yet.
source_yet_undefined_vars_from_starfish_sysconfig() {
    local sysconfig_file sysconfig_lines var_name

    sysconfig_file="$(get_sysconfig_file)"
    if [[ -r "${sysconfig_file}" ]]; then
        # shellcheck source=starfish/config/sysconfig-starfish
        mapfile -t sysconfig_lines <"${sysconfig_file}"
        for line in "${sysconfig_lines[@]}"; do
            # Consider only lines that are valid env var setting lines.
            if [[ "${line}" =~ ^[[:space:]]*[[:alnum:]_]+=.+$ ]]; then
                var_name=$(echo "${line}" | cut -d= -f1)
                # If env var named ${var_name} is not set, set it (by eval, i.e. "sourcing" the line from the file).
                if [[ -z "${!var_name+x}" ]]; then
                    eval "${line}"
                fi
            fi
        done
    fi
}

source_system_or_custom_config() {
    if [[ -z "${CONFIG_PATH}" ]]; then
        source_yet_undefined_vars_from_starfish_sysconfig
    else
        local exit_code=0
        # shellcheck source=/dev/null
        source "${CONFIG_PATH}" || exit_code=$?
        if [[ "${exit_code}" -ne 0 ]]; then
            echo "Problem sourcing config file; exit code: ${exit_code}; contents: "
            cat "${CONFIG_PATH}"
            echo "------------------------------------------------------------------"
        fi
    fi

    SENDING_DIAGNOSTIC_REPORTS_ENABLED=${SENDING_DIAGNOSTIC_REPORTS_ENABLED:-false}
    SENDING_DIAGNOSTIC_REPORTS_TO_STARFISH_ENABLED=${SENDING_DIAGNOSTIC_REPORTS_TO_STARFISH_ENABLED:-false}
    SENDING_DIAGNOSTIC_JSON_TO_STARFISH_ENABLED=${SENDING_DIAGNOSTIC_JSON_TO_STARFISH_ENABLED:-false}
    OBFUSCATE_CONFIDENTIAL_INFO_ENABLED=${OBFUSCATE_CONFIDENTIAL_INFO_ENABLED:-false}
    if [[ "${OBFUSCATE_CONFIDENTIAL_INFO_ENABLED}" == true ]]; then
        CONFIDENTIAL_OPT="--confidential"
    else
        CONFIDENTIAL_OPT=""
    fi
    DIAGNOSTIC_REPORTS_EXTRA_EMAILS=${DIAGNOSTIC_REPORTS_EXTRA_EMAILS:-""}
    DIAGNOSTIC_REPORTS_EXTRA_EMAILS_ERRORS_ONLY=${DIAGNOSTIC_REPORTS_EXTRA_EMAILS_ERRORS_ONLY:-""}
    DIAGNOSTIC_REPORTS_FROM_EMAIL=${DIAGNOSTIC_REPORTS_FROM_EMAIL:-""}
    DIAGNOSTIC_REPORTS_IGNORED_ERRORS_PATTERN=${DIAGNOSTIC_REPORTS_IGNORED_ERRORS_PATTERN:-""}
}

report_pgagent() {
    local exit_code
    "${SF}" pgagent-status "$@" || {
        exit_code=$?
        if [[ "${exit_code}" -ne 3 ]]; then
            echo >&2 "ERROR: pgagent-status returned ${exit_code}"
        fi
    }
}

main() {

    STARFISH_STATUS_MAIL=${STARFISH_STATUS_MAIL:-sf-status@starfishstorage.com}

    # do not remove tmp dir as the script ends
    KEEP_TMP=${KEEP_TMP:-false}
    # use provided config file instead of the one stored in sysconf
    CONFIG_PATH=${CONFIG_PATH:-""}

    DRY_RUN=false
    WITH_REDASH=true

    WARN_EMAILS=()
    ERROR_EMAILS=()
    FROM_EMAIL=""

    parse_cmdline_args "$@"

    source_system_or_custom_config

    if [[ "${SENDING_DIAGNOSTIC_REPORTS_ENABLED}" != true && "${DRY_RUN}" != true ]]; then
        cat <<_EOF
Exiting, because sending diagnostic reports is disabled.
Enable it by editing $(get_sysconfig_file) or run this script with --dry-run option.
_EOF
        # This script is run by cron even if sending reports is not configured;
        # zero exit code to prevent cron from logging failures
        exit 0
    fi

    collect_email_addresses

    init_attachment

    # ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"} trick in calls below prevents from passing "" param instead of nothing
    # yet it satisfies SC2086 missing double quotes check
    # https://serverfault.com/questions/867932/quote-only-if-variable-is-non-empty
    report_stat starfish_repo_url
    report_stat_json_reused check_config "${SF}" check-config ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat "${SF}" service-status
    report_stat "${SF}" volume list ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"} \
        --format "+files:{number_of_files} dirs:{number_of_dirs} logical_size:{sum_of_logical_sizes_no_nlinks} logical_size_with_hardlinks:{sum_of_logical_sizes_div_nlinks} physical_size:{sum_of_physical_sizes_no_nlinks} physical_size_with_hardlinks:{sum_of_physical_sizes_div_nlinks}" \
        --sort-by name
    report_stat_json_details volume "${SF}" volume list ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"} --with-mount-opts
    report_stat_json cron "${SF}" cron list ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat_json_details scan_pending "${SF}" scan pending ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat_json scan "${SF}" scan list --limit 50 ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat_json_details job_pending "${SF}" job pending ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat_json job "${SF}" job list --limit 100 ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat_json_details zone "${SF}" zone list ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    if [[ "$CONFIDENTIAL_OPT" != --confidential ]]; then # Skip collecting for confidential mode
        report_stat "${SF}" tag list --with-namespace --with-private --limit 500
    fi
    report_stat_json_details archive_target "${SF}" archive-target list ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat_json archive "${SF}" archive list --limit 50 ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat_json restore "${SF}" restore list --limit 50 ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat_json reports report_pgagent ${CONFIDENTIAL_OPT:+"$CONFIDENTIAL_OPT"}
    report_stat starfish_cron_info
    if [[ "${WITH_REDASH}" == true ]]; then
        redash_status "${CONFIDENTIAL_OPT}"
    fi
    finalize_attachment

    # Copy original report to two files before collect_logs.
    # Files are later sent to different recipients with different contents.
    cp "${REPORT_FILE}" "${REPORT_FILE_ERROR}"
    cp "${REPORT_FILE}" "${REPORT_FILE_WARN}"
    if [[ "${OBFUSCATE_CONFIDENTIAL_INFO_ENABLED}" == true ]]; then
        echo "Logs are not collected, as they may contain confidential information."
    elif [[ $EUID -ne 0 ]]; then
        echo "Logs are not collected. Only root can do that."
    else
        echo "Logs are being collected."
        collect_logs
    fi
    send_emails

    if [[ "${DRY_RUN}" != true ]]; then
        mkdir -p "${REPORT_TARGET_DIR}"
        cp "${REPORT_JSON_ATTACHMENT_FILE}" "${REPORT_TARGET_DIR}/"
    fi

    exit "${EXIT_CODE}"
}

main "$@"
