# ***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
# ***********************************************************************************************************
"""
    Rename rsync_wrapper job names to sfcopy
"""
import os
from collections import namedtuple
from logging import getLogger
from typing import List

from alembic import op
from sqlalchemy import text
from sqlalchemy.orm import Session

# revision identifiers, used by Alembic.
revision = "0113"
down_revision = "0112"
branch_labels = None
depends_on = None

logger = getLogger(__name__)

FAILURE_INSTRUCTION = """
The migration failed because job names could not be updated automatically.
Since rsync_wrapper tool has became deprecated this migration renames
rsync_wrapper job names to sfcopy job names and a conflict has been encountered.
Below you can find the complete list of conflicts.

Recovery
--------

First of all, check the list of conflicts and try to understand what is
the reason for each conflict. Conflict may look like this:

Couldn't rename job 'rsync_wrapper-src_path-dst_vol-dst_path' (name id: A)
to 'sfcopy-src_path-dst_vol-dst_path' (name id: B), because there is already a job with the latter name

That probably means that:
  - there is job that was automatically named to rsync_wrapper-src_path-dst_vol-dst_path (name id: A)
  - there is job that has been manually named sfcopy-src_path-dst_vol-dst_path (name id: B)
Thus the best solution here would be:
  - to rename job B to something other than sfcopy-src_path-dst_vol-dst_path
    This can be achieved by executing the following SQL query:
        UPDATE sf.job_name SET name = 'new-job-name' WHERE name = 'sfcopy-src_path-dst_vol-dst_path';
    Use appropriate names in the query, this is only an example.
  - if there are any scripts that start job and use that name they should be altered accordingly

In case of issues, please contact Starfish Tech Support team.
"""

JobName = namedtuple("JobName", ["id", "old_name", "new_name"])
Conflict = namedtuple("Conflict", ["renamed_id", "renamed_name", "conflicting_id", "conflicting_name"])


class JobNamesConflictException(Exception):
    def __init__(self, conflicts: List[Conflict]):
        self.conflicts = conflicts


def upgrade():
    session = Session(op.get_bind())
    file_name = os.path.basename(__file__)
    # This migration may fail if there are names conflicting with new job names
    # (rsync_wrapper is replaced by sfcopy in sf.job_name table).
    # Inserting '0113' to sf.skip_migration table allows to skip executing SQL script from this migration.
    if should_migration_be_skipped(session, revision):
        logger.warning(f"Migration {file_name} is being skipped")
        return

    execute_migration(session)


def execute_migration(session):
    job_names_to_be_updated = get_job_names_for_update(session)
    logger.info(f"Job names to be updated ({len(job_names_to_be_updated)}):")
    for job_name in job_names_to_be_updated:
        logger.info(f"Job name id: {job_name.id} name: {job_name.old_name} -> {job_name.new_name}")

    conflicts = get_conflicting_job_names(session, job_names_to_be_updated)
    if len(conflicts) > 0:
        logger.error(FAILURE_INSTRUCTION)
        logger.error(f"The following {len(conflicts)} job names are in conflict:")
        for conflict in conflicts:
            logger.error(
                f"Couldn't rename job  '{conflict.renamed_name}' (name id: {conflict.renamed_id}) "
                f"to '{conflict.conflicting_name}' (name id: {conflict.conflicting_id}), "
                "because there is already a job with the latter name"
            )
        raise JobNamesConflictException(conflicts)

    for job_name in job_names_to_be_updated:
        session.execute(
            text("UPDATE sf.job_name SET name = :new_name WHERE id = :id").bindparams(
                id=job_name.id, new_name=job_name.new_name
            )
        )


# unfortunately, it is not possible to import here other starfish modules
def should_migration_be_skipped(session, version: str) -> bool:
    if not does_table_exist(session, "sf", "skip_migration"):
        return False
    row = session.execute(
        "SELECT count(*) > 0 FROM sf.skip_migration WHERE version_num=:version", {"version": version}
    ).fetchone()
    return row[0]


def does_table_exist(session, schema_name, table_name):
    table_exists = session.execute(
        text(
            """SELECT
                EXISTS(
                    SELECT 1
                    FROM pg_tables
                    WHERE schemaname = :schema_name AND tablename = :table_name
                );
        """
        ).bindparams(schema_name=schema_name, table_name=table_name)
    ).first()[0]
    return table_exists


def get_job_names_for_update(session) -> List[JobName]:
    rows = session.execute(
        text(
            """
        SELECT
            id,
            name,
            regexp_replace(name, '^(restore_)?rsync_wrapper', '\\1sfcopy')
        FROM sf.job_name
        WHERE
            -- do not update irrelevant rows
            (
                sf.job_name.name LIKE 'rsync_wrapper%'
                OR sf.job_name.name LIKE 'restore_rsync_wrapper%'
            )
            -- update only those names that have job incarnations without explicitly set --job-name
            AND NOT EXISTS (
                SELECT 1
                FROM (
                    SELECT bool_or(inc_cmd.job_name_option) AS incarnation_has_explicit_job_name
                    FROM (
                        SELECT
                        i.incarnation_id,
                        i.job_id,
                        -- It is possible to run sf job start --job-n something-something ...
                        -- (all: --job-n, --job-na, --job-nam, --job-name options are accepted)
                        -- The array may contain single entry like '--job-name=my_job_name' as well as
                        -- 2 entries: '--job-name', 'my_job_name'
                        jsonb_array_elements_text(
                            CASE
                            WHEN jsonb_typeof(i.cmd_line) = 'array' THEN i.cmd_line
                            ELSE '[]'::jsonb
                            END
                        ) LIKE '--job-n%' AS job_name_option
                        FROM sf_dispatcher.incarnation i
                        JOIN sf_dispatcher.job j ON i.job_id = j.id
                            WHERE j.name_id = sf.job_name.id
                        ) AS inc_cmd
                    GROUP BY inc_cmd.incarnation_id, inc_cmd.job_id
                ) AS inc_explicit_job_name
                WHERE inc_explicit_job_name.incarnation_has_explicit_job_name
            );
    """
        )
    )
    return [JobName(*row) for row in rows]


def get_conflicting_job_names(session, job_names: List[JobName]) -> List[Conflict]:
    result = []
    for job_name in job_names:
        conflicting = session.execute(
            text(
                """SELECT id, name
                FROM sf.job_name
                WHERE
                    id != :id
                    AND name = :name"""
            ).bindparams(id=job_name.id, name=job_name.new_name)
        )
        for row in conflicting:
            result.append(Conflict(job_name.id, job_name.old_name, row[0], row[1]))
    return result
