/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */
/* no-temp-table-check */

CREATE OR REPLACE FUNCTION sf_internal.load_events_fallback_{{ dirs_or_files }}_{{ added_or_removed }}_vol_{{ vol_id }}
(
    pipeline_id BIGINT,                                         -- Pipeline id
    bHistory BOOLEAN,                                           -- Track history for path and file changes
    oos_accuracy_secs bigint
)
    RETURNS sf_internal.load_events_ret_type AS $$
DECLARE
    strErrorMessage TEXT;           -- Hold error message for logging
    events_processed BIGINT = 0;    -- Number of events processed (succeeded, ignored or failed)
    events_ignored BIGINT = 0;      -- Number of events ignored due to old sync_time
    events_failed BIGINT = 0;       -- Number of failed events
    row_id tid;
    event_with_entry record;
    event_result sf_internal.load_events_ret_type;
    ret sf_internal.load_events_ret_type;       -- return record
BEGIN

    CREATE TEMP TABLE load_events_events_with_entries_fallback AS SELECT * FROM load_events_events_with_entries;
    TRUNCATE load_events_events_with_entries;

    -- Processing each event in BEGIN .. EXCEPTION .. END block is inefficient, because:
    -- * entering and exiting the block is "significantly more expensive" (looks like ~10% on my desktop)
    --   as described in http://www.postgresql.org/docs/current/static/plpgsql-control-structures.html#PLPGSQL-ERROR-TRAPPING
    -- * as a new transaction is created for each event, a full vacuum is performed every 2 billion events:
    --   http://www.postgresql.org/docs/current/static/routine-vacuuming.html#VACUUM-FOR-WRAPAROUND
    FOR row_id IN SELECT ctid FROM load_events_events_with_entries_fallback
    LOOP
        INSERT INTO load_events_events_with_entries (SELECT * FROM load_events_events_with_entries_fallback WHERE ctid = row_id);

        BEGIN
            event_result := (SELECT (sf_internal.load_events_{{ dirs_or_files }}_{{ added_or_removed }}_vol_{{ vol_id }}(bHistory, oos_accuracy_secs)));
            events_processed = events_processed + event_result.events_processed;
            events_ignored = events_ignored + event_result.events_ignored;
            events_failed = events_failed + event_result.events_failed;

            -- This forces postgres to check foreign_key_violation now
            set constraints all immediate;
            set constraints all deferred;
        exception WHEN others THEN
            get stacked diagnostics strErrorMessage = MESSAGE_TEXT;
            event_with_entry := (SELECT e FROM load_events_events_with_entries AS e);
            raise warning 'exception occurred: % for event %', strErrorMessage, row_to_json(event_with_entry.event_row);
            INSERT INTO sf_internal.event_error SELECT (event_with_entry.event_row).*, strErrorMessage, pipeline_id;
            events_failed = events_failed + 1;
            events_processed = events_processed + 1;
        END;

        DELETE FROM load_events_events_with_entries;
        -- We have to truncate it, otherwise performance will go down -
        -- each next event will be processed slower, because of more dead tuples in the table.
        -- OTOH truncate itself is quite slow.
        -- Truncating after each 1000 rows is the Golden mean for best performance.
        IF events_processed % 1000 = 0 THEN
            TRUNCATE load_events_events_with_entries;
        END IF;
    END LOOP;

    SELECT events_processed, events_ignored, events_failed INTO ret;

    DROP TABLE load_events_events_with_entries_fallback;

    RETURN ret;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER VOLATILE PARALLEL UNSAFE;
