#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

_PG_COMMON_SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
readonly _PG_COMMON_SCRIPT_DIR

# shellcheck source=scripts/installation/_utils.sh
source "${_PG_COMMON_SCRIPT_DIR}/_utils.sh"
# if adding anything sourced here, note that it needs to be copied in copy_ctl_scripts

PSQL="psql --variable ON_ERROR_STOP=1 -P pager=off"
STARFISH_PG_VERSION="13"
STARFISH_PG_CLUSTER_NAME="main"
# shellcheck disable=SC2034  # variable used in other scripts, which source this one
STARFISH_PG_CONFIG_TEMPLATE="starfish"
# shellcheck disable=SC2034  # variable used in other scripts, which source this one
STARFISH_PG_DB_NAME="starfish"
# shellcheck disable=SC2034  # variable used in other scripts, which source this one
STARFISH_PG_PORT=5432
# shellcheck disable=SC2034  # variable used in other scripts, which source this one
REPORTS_USER_IN_SF_CLUSTER="reports_user"
readonly PSQL STARFISH_PG_VERSION STARFISH_PG_CLUSTER_NAME STARFISH_PG_CONFIG_TEMPLATE STARFISH_PG_DB_NAME \
    STARFISH_PG_PORT REPORTS_USER_IN_SF_CLUSTER
export STARFISH_PG_CONFIG_TEMPLATE STARFISH_PG_DB_NAME STARFISH_PG_PORT REPORTS_USER_IN_SF_CLUSTER

get_default_db_storage_type() {
    echo "hdd"
}

get_default_pgbackup_path() {
    echo "${STARFISH_PG_DIR}/backups"
}

centos_get_default_pgdata() {
    local pg_version="${1:-${STARFISH_PG_VERSION}}"
    local pg_cluster_name="${2:-${STARFISH_PG_CLUSTER_NAME}}"
    local pg_dir_name

    if [[ "${pg_cluster_name}" = "main" ]]; then
        pg_dir_name=data
    else
        pg_dir_name="${pg_cluster_name}"
    fi
    echo "/var/lib/pgsql/${pg_version}/${pg_dir_name}"
}

ubuntu_get_default_pgdata() {
    local pg_version="${1:-${STARFISH_PG_VERSION}}"
    local pg_cluster_name="${2:-${STARFISH_PG_CLUSTER_NAME}}"

    echo "/var/lib/postgresql/${pg_version}/${pg_cluster_name}"
}

get_default_pgdata() {
    # default path used in PostgreSQL packages
    run_func_for_distro centos_get_default_pgdata ubuntu_get_default_pgdata "$@"
}

dereference_db_path() {
    local db_path="$1"

    # pgbackrest doesn't support symlinks and checks if data_directory is exactly the same as
    # directory in pgbackrest config
    # https://github.com/pgbackrest/pgbackrest/issues/671
    readlink --verbose --canonicalize-existing "${db_path}"
}

centos_get_pgconfig_dir() {
    local pg_data="$1"

    echo "${pg_data}"
}

ubuntu_get_pgconfig_dir() {
    local pg_data="$1"
    local pg_version="$2"
    local pg_cluster_name="$3"

    echo "/etc/postgresql/${pg_version}/${pg_cluster_name}"
}

get_pgconfig_dir() {
    run_func_for_distro centos_get_pgconfig_dir ubuntu_get_pgconfig_dir "$@"
}

get_pg_local_conf_file() {
    local pg_dir=$1

    echo "${pg_dir}/local.conf"
}

centos_get_pgbin_dir() {
    local pg_version="$1"

    echo "/usr/pgsql-${pg_version}/bin/"
}

ubuntu_get_pgbin_dir() {
    local pg_version="$1"

    echo "/usr/lib/postgresql/${pg_version}/bin/"
}

get_pgbin_dir() {
    run_func_for_distro centos_get_pgbin_dir ubuntu_get_pgbin_dir "$@"
}

centos_get_pg_service_name() {
    local pg_cluster_name="$1"
    local pg_version="$2"

    if [[ "${pg_cluster_name}" = "${STARFISH_PG_CLUSTER_NAME}" ]]; then
        echo "postgresql-${pg_version}"
    else
        echo "postgresql-${pg_version}-${pg_cluster_name}"
    fi
}

ubuntu_get_pg_service_name() {
    local pg_cluster_name="$1"
    local pg_version="$2"

    echo "postgresql@${pg_version}-${pg_cluster_name}"
}

get_pg_service_name() {
    run_func_for_distro centos_get_pg_service_name ubuntu_get_pg_service_name "$@"
}

get_pg_service_description() {
    local pg_cluster_name="$1"

    if [[ "${pg_cluster_name}" = "${STARFISH_PG_CLUSTER_NAME}" ]]; then
        echo "Starfish"
    else
        echo "${pg_cluster_name}"
    fi
}

run_as_postgres() {
    # do not use sudo if not necessary, see STAR-7049
    run_with_sudo_if_not_root su --login postgres --command "$@"
}

psql_as_postgres() {
    local port="$1"
    local db_name="$2"
    local query="${3:-""}"

    if [[ -n "${query}" ]]; then
        run_as_postgres "${PSQL} --port=${port} --dbname=${db_name} --quiet --tuples-only --no-align -c \"${query}\""
    else
        run_as_postgres "${PSQL} --port=${port} --dbname=${db_name} --quiet --tuples-only --no-align --file=-"
    fi
}

postgresql_database_exists() {
    local port="$1"
    local db_name="$2"

    psql_as_postgres "${port}" "${db_name}" "select 1;" > /dev/null 2>&1
}

drop_postgresql_database() {
    local port="$1"
    local db_name="$2"

    local query
    query="DROP DATABASE IF EXISTS ${db_name};"
    run_as_postgres "${PSQL} --port=${port} --quiet --tuples-only --no-align -c \"${query}\""
}

create_database_from_template() {
    local port="$1"
    local db_name="$2"
    local template_name="$3"

    local query
    query="CREATE DATABASE ${db_name} WITH TEMPLATE ${template_name} OWNER starfish;"

    run_as_postgres "${PSQL} --port=${port} --quiet --tuples-only --no-align -c \"${query}\""
}

centos_get_postgres_service_name() {
    local pg_version="$1"
    local pg_cluster_name="$2"

    if [[ "${pg_cluster_name}" = "main" ]]; then
        echo "postgresql-${pg_version}"
    else
        echo "postgresql-${pg_version}-${pg_cluster_name}"
    fi
}

centos_postgres_action() {
    local action="$1"
    local pg_version="$2"
    local pg_cluster_name="$3"
    # shellcheck disable=SC2034
    local pg_port="$4"

    service "$(centos_get_postgres_service_name "${pg_version}" "${pg_cluster_name}")" "${action}"
}

ubuntu_postgres_action() {
    local action="$1"
    local pg_version="$2"
    local pg_cluster_name="$3"
    # shellcheck disable=SC2034
    local pg_port="$4"

    pg_ctlcluster "${pg_version}" "${pg_cluster_name}" "${action}"
}

postgres_action() {
    run_func_for_distro centos_postgres_action ubuntu_postgres_action "$@"
}

reload_postgresql_service() {
    postgres_action reload "$@"
}

restart_postgresql_service() {
    # shellcheck disable=SC2034
    local pg_version="$1"
    # shellcheck disable=SC2034
    local pg_cluster_name="$2"
    local pg_port="$3"

    postgres_action restart "$@"
    wait_for "is_postgresql_running ${pg_port}" 60 || {
      log "PostgreSQL ${pg_version} on port ${pg_port} failed to restart. See ${LOG_FILE} for details."
      exit 1
    }
}

stop_postgresql_service() {
    postgres_action stop "$@" || true
}

start_postgresql_service() {
    # shellcheck disable=SC2034
    local pg_version="$1"
    # shellcheck disable=SC2034
    local pg_cluster_name="$2"
    local pg_port="$3"

    postgres_action start "$@"
    wait_for "is_postgresql_running ${pg_port}" 60 || {
      log "PostgreSQL ${pg_version} on port ${pg_port} failed to start. See ${LOG_FILE} for details."
      exit 1
    }
}

is_postgresql_running() {
    local port="$1"

    # run any postgresql command to check that the server is running
    psql_as_postgres "${port}" postgres "select 1;" > /dev/null
}

is_postgresql_stopped() {
    local port="$1"

    # When postgres is not running, is_postgresql_running prints an error message to stderr.
    # In general, it's useful for diagnosing a problem. Here we expect postgres to be not running,
    # so printing the error message is not desired.
    ! is_postgresql_running "${port}" 2>/dev/null
}


ensure_postgresql_user_exists() {
    local port="$1"
    local username="$2"
    local password="$3"
    local superuser="${4:-false}"

    if [[ "${superuser}" != "superuser" ]]; then
        superuser=""
    fi

    psql_as_postgres "${port}" postgres <<EOF
DO
\$body\$
BEGIN
    IF NOT EXISTS (SELECT * FROM pg_catalog.pg_user WHERE usename = '${username}') THEN
        CREATE USER ${username} WITH ${superuser} PASSWORD '${password}';
    ELSE
        ALTER USER ${username} WITH PASSWORD '${password}';
    END IF;
END
\$body\$;
EOF
}

remove_postgresql_cluster() {
    run_func_for_distro centos_remove_postgresql_cluster ubuntu_remove_postgresql_cluster "$@"
}

centos_remove_postgresql_cluster() {
    local pg_version="$1"
    local pg_cluster_name="$2"
    local db_path="$3"

    log Removing PG cluster "${pg_version}" "${pg_cluster_name}" "${db_path}"
    local service_name
    service_name="$(centos_get_postgres_service_name "${pg_version}" "${pg_cluster_name}")"
    systemctl disable "${service_name}" || true
    rm -f "/etc/systemd/system/${service_name}.service"
    systemctl daemon-reload
    rm -rf -- "${db_path}"
}

ubuntu_remove_postgresql_cluster() {
    local pg_version="$1"
    local pg_cluster_name="$2"

    log Removing PG cluster "${pg_version}" "${pg_cluster_name}"
    pg_dropcluster --stop "${pg_version}" "${pg_cluster_name}" 2> /dev/null || true
    systemctl daemon-reload
}

find_pg_command() {
    local command="$1"
    local ver="$2"
    local path

    path="$(command -v "/usr/pgsql-${ver}/bin/${command}" 2>&1 || \
        command -v "/usr/lib/postgresql/${ver}/bin/${command}" 2>&1 || \
        command -v "${command}" 2>&1 || \
        true)"

    if [[ "${path}" == "" ]]; then
        >&2 echo "could not find ${command}"
        exit 1
    fi

    echo "${path}"
}

find_pg_96_command() {
    local command="$1"
    local path version maj min
    path="$(find_pg_command "${command}" 9.6)"

    version="$("${path}" --version | cut -f 2 -d \) | tr -d " ")"
    maj="$(echo "${version}" | cut -f 1 -d .)"
    min="$(echo "${version}" | cut -f 2 -d .)"

    if [[ "${maj}" -lt 9 || "${min}" -lt 6 ]]; then
        >&2 echo "${path} needs to be at least 9.6, found ${version}"
        exit 1
    fi

    echo "${path}"
}

find_pg_13_command() {
    local command="$1"
    local path version maj
    path="$(find_pg_command "${command}" 13)"

    # get the version from psql command
    # psql --version returns "psql (PostgreSQL) 13.7"
    # "${path}" --version | cut -f 3 -d " " returns "13.7"
    version=$("${path}" --version | cut -f 3 -d " ")
    # cut major version, i.e. the first field with . as separators
    maj=$(echo "${version}" | cut -f 1 -d .)

    if [[ "${maj}" -lt 13 ]]; then
        >&2 echo "${path} needs to be at least 13, found ${version}"
        exit 1
    fi

    echo "${path}"
}


find_psql_command() {
    local pg_version="$1"
    path="$(find_pg_command "psql" "${pg_version}")"
    echo "${path}"
}
