#!/bin/bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

_MIGRATE_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
readonly _MIGRATE_DIR

# shellcheck source=scripts/installation/_common_install.sh
source "${_MIGRATE_DIR}/_common_install.sh"

PREFIX="${RPM_INSTALL_PREFIX:-}"
if [[ "${PREFIX}" = "/" ]]; then
    PREFIX=""
fi
SFHOME="${PREFIX}${SFHOME:-/opt/starfish}"
STARFISH_NGINX_CONF_DIR=$(readlink --verbose --canonicalize-missing "${SFHOME}/nginx/etc/conf.d")
NGINX_REDASH_FILE="${STARFISH_NGINX_CONF_DIR}/starfish/redash.conf"

REDASH_1_BACKUP_FILE_NAME="redash-1.0.3-backup.tar.gz"
# system_files is a new directory for redash files outside sfhome/redash/1.0.3
BACKUP_REDASH_ENTRIES=(
    celery
    .env
    gunicorn.conf
    pg
    redis
    src
    venv
    system_files
)
REDASH_EMPTY_DIRS_TO_BE_RETAINED=(
    src
    venv
)
SYSTEM_ENTRIES_TO_BE_REMOVED=(
    /etc/systemd/system/multi-user.target.wants/postgresql-9.6-redash.service
    /etc/systemd/system/postgresql-9.6-redash.service
     "${NGINX_REDASH_FILE}"
    /etc/logrotate.d/sf-redash
)
BACKUP_SYSTEM_ENTRIES=(
     "${SYSTEM_ENTRIES_TO_BE_REMOVED[@]}"
    /etc/supervisord.d/sf-redash.ini
)

backup_redash_1() {
    local backup_file="$1"

    local redash_1_dir existing_backup_redash_entries=()

    redash_1_dir="$(get_redash_dir 1)"
    system_files_dir="${redash_1_dir}/system_files"

    # ( CMD ) is subshell, so that "cd .." doesn't effect this shell
    (
        for file in "${BACKUP_SYSTEM_ENTRIES[@]}"; do
            if file_exists "${file}"; then
                destination="${system_files_dir}/$(dirname "${file}")"
                log Moving "${file}" to "${destination}"
                mkdir --parents "${destination}"
                cp --preserve=all "${file}" "${destination}"
            fi
        done
        cd "${redash_1_dir}"

        for entry in "${BACKUP_REDASH_ENTRIES[@]}"; do
            if path_exists "${entry}"; then
                existing_backup_redash_entries+=("${entry}")
            fi
        done

        log Backing up Redash 1 to "${backup_file}"
        tar -czf "${backup_file}" "${existing_backup_redash_entries[@]}"

        # Config files and files that were created after installing redash
        # are not automatically removed by yum/apt thus trying to remove all files here.
        # Please not that on CentOS this code is run before removing the package.
        for entry in "${existing_backup_redash_entries[@]}"; do
            log Removing "${entry}"
            rm -rf "${entry}"
        done
        # Pre-remove script of old Redash 1 packages searches venv and src directories
        # for .pyc files and empty dirs. If the directories are not there, 'find' command
        # used by the search fails along with the whole script.
        for dir in "${REDASH_EMPTY_DIRS_TO_BE_RETAINED[@]}"; do
            mkdir --parents "${dir}"
        done
    )
}

dump_db_and_stop_redash_pg() {
    local dump_file
    if is_postgresql_running "$(get_redash_pg_port 1)"; then
        dump_file="$(get_redash_dir 1)/$(date +'%Y%m%d_%H%M%S')_redash_1.sql.gz"
        log Creating Redash PG dump file "${dump_file}"
        dump_redash_pg 1 "${dump_file}"
        log Stopping Redash PG cluster
        stop_redash_postgres 1
    fi
}

cleanup_redash_pg() {
    local pg_version pg_cluster_name db_path
    pg_version="$(get_redash_pg_version 1)"
    pg_cluster_name="$(get_redash_pg_cluster_name 1)"
    db_path="$(get_redash_pg_dir 1)"

    remove_postgresql_cluster "${pg_version}" "${pg_cluster_name}" "${db_path}"
}

cleanup_system_files() {
    local file
    for file in "${SYSTEM_ENTRIES_TO_BE_REMOVED[@]}"; do
        log "Removing ${file}"
        rm -f "${file}"
    done
}

remove_redash_1_package() {
    local distro pkg_name
    distro="$(get_distribution)"
    if [[ "${distro}" == "ubuntu" ]]; then
        pkg_name="$(get_redash_pkg_name 1)"
        log Removing redash 1 package "(${pkg_name})"
        apt-get remove --assume-yes "${pkg_name}"
    fi
}

main() {
    local env1_file backup_file noop_on_ubuntu=false distro
    fail_if_not_root

    env1_file="$(get_redash_env_filename 1)"
    if ! file_exists "${env1_file}"; then
        echo "${env1_file} does not exist, no need to backup Redash 1"
        exit 0
    fi

    LOG_FILE="${SFHOME}/redash/_backup_redash_1.log"
    while [[ $# -gt 0 ]]; do
        case $1 in
        --log-file)
            shift
            LOG_FILE="$1"
            ;;
        --noop-on-ubuntu)
            shift
            noop_on_ubuntu=true
            ;;
        *)
            break
            ;;
        esac
        shift
    done
    export LOG_FILE

    # On Ubuntu, this script may be run in 2 cases:
    # 1. from post-install for sf-redash package -> it does nothing in such a case
    # 2. manually by support -> does the same as on CentOS + removes the redash 1 package
    #  which on CentOS is done automatically
    distro="$(get_distribution)"
    if [[ "${noop_on_ubuntu}" == true && "${distro}" == ubuntu ]]; then
        # noop
        log The OS is Ubuntu and invoked with "--noop-on-ubuntu". Exiting.
        exit 0
    fi

    backup_file="$(get_redash_dir 1)/${REDASH_1_BACKUP_FILE_NAME}"

    stop_redash 1 || true
    dump_db_and_stop_redash_pg
    backup_redash_1 "${backup_file}"
    cleanup_redash_pg
    # Remove files from system at the end because some of them may be needed meanwhile.
    # For example systemd unit files for postgres.
    cleanup_system_files
    remove_redash_1_package
}

main "$@"
