#!/bin/bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

SFHOME="${SFHOME:-/opt/starfish}"
SF="${SFHOME}/bin/client"
# Few notes about SF_LOCK_PREFIX path
# /opt/starfish/run is drwxr-xr-x and could leave the lock files for a long time
# on the other hand /opt/starfish/tmp is drwxrwxrwt
# To have the temporariness of lock files + ability to use sync and verify for other users
# we should use the /opt/starfish/tmp. In that way we assure there are no old lock files stored.
# broader /opt/starfish/tmp permissions are also a possibility to use this script not only by root users.

SF_LOCK_PREFIX="${SFHOME}/tmp/sync-and-verify"
CMP_FILE_TREE_PROG="${SFHOME}/bin/cmp_file_trees"
PROG="$0"
START_TIME_TS="$(date +"%s")"
DATE_FORMAT="%Y-%m-%d %H:%M:%S"

GZIP_ATTACHMENT_OVER_SIZE="${GZIP_ATTACHMENT_OVER_SIZE:-$((10*1024*1024-40960))}"
MAX_ATTACHMENT_SIZE="${MAX_ATTACHMENT_SIZE:-$((10*1024*1024-40960))}"

NUM_RETRIES_TO_SCAN_DEST_VOLUME="${NUM_RETRIES_TO_SCAN_DEST_VOLUME:-3}"
RETRY_SCAN_INTERVAL=60

TMPDIR=${SFHOME}/tmp

# shellcheck source=scripts/installation/_utils.sh
source "${SFHOME}/data/installation/_utils.sh"

usage() {
    cat <<EOF

${PROG} [options] <source> <destination>

This script copies new and modified files from a source to a destination and reports on deleted files on the source.
By default diff scan is run automatically when this script is run.

    -h, --help     - print this help and exit
    --copy-args    - use extra copy arguments
    --resync       - force invocation of re-copy on all files, even if already done
                     (useful if some files have been modified/removed from destination)
    --skip-source-scan  - Disable scanning of source volume (e.g. when other, overlapping scans
                     are expected to be running at the same time)
    --scan-type,-t - change scan type ('diff' by default). Faster 'mtime' scan type may leave files
                     modified on source not synchronized with destination
    --estimate     - option to run all jobs with calculating estimations (potential performance drop)
    --email        - destination email(s) address (ex. --email "a@a.pl b@b.com")
                     if more then one recipient then quotes are needed around the emails
    --log          - save email contents to specified file (it must be a path to not existing file
                    in existing directory). It may contain datatime parts (see 'man date'), for example:
                    --log "/opt/starfish/log/sync-and-verify-sourcevol-%Y%m%d-%H%M%S.log"
                    NOTE: When running from cron, escape the % chars using the following
                     --log "/opt/starfish/log/sync-and-verify-sourcevol-\\%Y\\%m\\%d-\\%H\\%M\\%S.log"
    --no-remove-dest - do not remove files from destination even if they don't exist on source
    --stop-overlapping-scan   - interrupt overlapping scan on destination volume instead of waiting for it
    source         - volume and path of files to be synchronized
    destination    - volume and path where files should be synchronized to
EOF
}

fatal() {
    echo "$@"
    exit 1
}

# global variables
BROKEN_ENTRIES=""

USE_LOCK="1"
REMOVE_DEST="1"
COPY_ARGS=""
JOB_ARGS=()
EMAILS=()
LOG_EMAIL_CONTENT=""
KEEP_TEMP_DIR="0"
VERBOSE="0"
ESTIMATIONS_OPTION="--no-estimations"
SCAN_TYPE="diff"
WAIT_INTERVAL=10
STOP_SCAN=""
SKIP_SOURCE_SCAN=0
MANIFEST_OPT="--generate-manifest"

check_parameters_value() {
    local param="$1"

    [ $# -gt 1 ] || fatal "Missing value for parameter ${param}"
}

while [[ $# -gt 0 ]]; do
    case $1 in
    "-h"|"--help")
        usage
        exit 0
        ;;
    "--copy-args")
        shift
        COPY_ARGS="$1"
        ;;

    "--resync")
        JOB_ARGS+=("--from-scratch")
        ;;
    "--allow-empty-dst-dir")
        ;;
    "--email"|"--emails")
        check_parameters_value "$@"
        shift
        # Word splitting intentional - disable shellcheck rule
        # shellcheck disable=SC2206
        EMAILS=($1)
        ;;
    "--keep-temp-dir")
        KEEP_TEMP_DIR=1
        ;;
    "--verbose")
        VERBOSE=1
        ;;
    "--estimate")
        ESTIMATIONS_OPTION=""
        ;;
    "--skip-source-scan")
        SKIP_SOURCE_SCAN=1
        ;;
    "--scan-type"|"-t")
        check_parameters_value "$@"
        shift
        SCAN_TYPE="$1"
        ;;
    "--disable-lock")
        USE_LOCK=""
        ;;
    "--wait-interval")
        check_parameters_value "$@"
        shift
        WAIT_INTERVAL="$1"
        ;;
    "--log")
        shift
        LOG_EMAIL_CONTENT=$(date +"$1")
        ;;
    "--no-remove-dest")
        REMOVE_DEST="0"
        ;;
    "--stop-overlapping-scan")
        STOP_SCAN="1"
        ;;
    "--no-manifest")
        MANIFEST_OPT="--no-manifest"
        ;;
    *)
        if [[ "$1" =~ ^--.* ]]; then
            usage
            fatal "Unexpected parameter: $1"
        fi

        if [ -z "${SOURCE:-}" ]; then
            SOURCE="$1"
        elif [ -z "${DESTINATION:-}" ]; then
            DESTINATION="$1"
        else
            fatal "Unexpected parameter: $1"
        fi
        ;;
    esac;
    shift
done

[ "${VERBOSE}" = 0 ] || set -x


clear_temp_files() {
    if [ "${KEEP_TEMP_DIR}" = "0" ] && [ -n "${TEMP_DIR}" ];
    then
        rm -fr -- "${TEMP_DIR}"
    else
        echo "TEMP_DIR=${TEMP_DIR}"
    fi
}

create_email_description() {
    local subject=$1
    local file_with_result=$2
    local attachment=$3

    echo "SUBJECT: ${subject}"
    echo "REPORT: ---"
    cat "${file_with_result}"
    if [ -s "${attachment}" ];
    then
        echo "--- ATTACHMENT: ${attachment}"
        cat "${attachment}"
    fi
    echo "---"
}

_is_attachment_size_acceptable() {
    local attachment_file="$1"
    local attachment_size
    attachment_size=$(stat --format %s "${attachment_file}")

    [ $((attachment_size)) -lt $((MAX_ATTACHMENT_SIZE)) ]
}

_prepare_attachment() {
    local attachment_file
    attachment_file="$1"
    local attachment_size
    attachment_size=$(stat --format %s "${attachment_file}")
    local err=""

    if [ $((attachment_size)) -gt $((GZIP_ATTACHMENT_OVER_SIZE)) ]; then
        # NOTE: earlier versions of gzip does not support --keep option
        if err=$(gzip --stdout < "${attachment_file}" > "${attachment_file}.gz" 2>&1); then
            local attachment_gz="${attachment_file}.gz"
            if _is_attachment_size_acceptable "${attachment_gz}"; then
                echo -n "${attachment_gz}"
                return 0
            fi
        else
            echo "Can't gzip attachment: ${err}" >&2
        fi
    fi
    # not gzipping but maybe it's acceptable to attach details as plain text
    if _is_attachment_size_acceptable "${attachment_file}"; then
        echo -n "${attachment_file}"
        return 0
    fi;
    return 1
}

prepare_attachment_parameters() {
    local attachment_option="$1"
    local attachment_file="$2"

    local valid_attachment_file=""

    if [ "$attachment_option" ]; then
        if [ "${attachment_file}" ]; then
            if valid_attachment_file=$(_prepare_attachment "${attachment_file}"); then
                ATTACHMENT_PARAMETERS+=("${attachment_option}" "${valid_attachment_file}")
                return 0
            fi

            attachment_file_no_warnings="${attachment_file}_no_warn"
            grep -v '^WARN ' "${attachment_file}" > "${attachment_file_no_warnings}"

            if valid_attachment_file=$(_prepare_attachment "${attachment_file_no_warnings}"); then
                ATTACHMENT_PARAMETERS+=("${attachment_option}" "${valid_attachment_file}")
                {
                    echo -n "Attachment contains only errors because number of warnings exceeds maximum allowed size, "
                    echo "review warnings via log file on Starfish Server"
                } >&2
                return 0
            fi
            echo "Attachment exceeds maximum allowed size, review warnings/errors via log file on Starfish Server" >&2
        fi;
    fi;
}

send_report() {
    local subject="$1"
    local file_with_result="$2"
    local attachment="${3:-}"
    local attach_file_option=""

    if [ "${#EMAILS[@]}" -ne 0 ]; then
        if [ -s "${attachment}" ]; then
            attach_file_option="$(get_attachment_option)"
            if [ -z "${attach_file_option}" ]; then
                # can't detect attach option - ignore detailed messages
                {
                    echo -n -e "\\nError/warning details were not able to be attached, "
                    echo "unable to detect mail option to attach file. See log file for error/warning details"
                } >> "${file_with_result}"
            fi
            ATTACHMENT_PARAMETERS=()
            prepare_attachment_parameters "${attach_file_option}" "${attachment}" 2>> "${file_with_result}"
        fi
        set +u  # empty arrays in bash require use of this "set" directive - otherwise it fails with "unbound variable"
        mail -s "${subject}" "${ATTACHMENT_PARAMETERS[@]}" "${EMAILS[@]}" < "${file_with_result}"
        set -u
    else
        create_email_description "${subject}" "${file_with_result}" "${attachment}"
    fi;
    if [ "${LOG_EMAIL_CONTENT}" ]; then
        {
            [ "${#EMAILS[@]}" -ne 0 ] && echo "TO: ${EMAILS[*]}"
            create_email_description "${subject}" "${file_with_result}" "${attachment}"
        } > "${LOG_EMAIL_CONTENT}"
    fi
}

remove_lock_path_if_not_locked(){
    # shellcheck disable=SC2015
    flock --nonblock 9 9>> "${LOCK_FILE_PATH}" && rm -f -- "${LOCK_FILE_PATH}" || true
}

on_exit() {
    trap - DEBUG
    remove_lock_path_if_not_locked

    local err_result="${TEMP_DIR}/report-err.txt"
    {
        echo "Fatal error during: ${last_one_action}" | tr _ ' '
        if [ -s "${LAST_CMD_OUTPUT}" ]; then
            echo "---[BEGIN]"
            cat "${LAST_CMD_OUTPUT}"
            echo "---[END]"
        fi;
    } > "${err_result}"

    send_report "[FATAL] Sync Report ${SOURCE} to ${DESTINATION}" "${err_result}"
    echo "TEMP_DIR=${TEMP_DIR}"
    echo "Error during sync-and-verify, error report in" "${err_result}"
}

_log_next_step() {
    local datetime
    local variable_name_with_semaphore_file wait_before_next_step_on_file
    datetime="$(date +"${DATE_FORMAT}")"

    echo " === ${datetime} ${BASH_COMMAND} ===" | tr _ ' ' | tee -a "${REPORT_DETAILS_FILE}"
    rm -f "${LAST_CMD_OUTPUT}"
    last_one_action="$BASH_COMMAND"

    if [ "${USE_SEMAPHORE_FILES:-}" ]; then
        # semaphore files are ONLY for testing and should not be used in production
        # it's mainly for halting sync-and-verify in some particular step
        variable_name_with_semaphore_file=semaphore_${last_one_action}
        wait_before_next_step_on_file=${!variable_name_with_semaphore_file:-}
        if [ "${wait_before_next_step_on_file}" ]; then
            echo "$$" > "${wait_before_next_step_on_file}"
            echo "wait until file '${wait_before_next_step_on_file}' disappear"
            while test -e "${wait_before_next_step_on_file}"; do
                sleep 1
            done
        fi
    fi
}

setup() {

    local filename
    filename=$(echo "${SOURCE}-${DESTINATION}" | md5sum - | cut -f1 -d ' ')
    LOCK_FILE_PATH="${SF_LOCK_PREFIX}-${filename}.lock"
    readonly LOCK_FILE_PATH

    DST_VOLUME="$(echo "${DESTINATION}" | cut -d: -f1)"
    readonly DST_VOLUME
    SRC_PATH="$(echo "${SOURCE}" | cut -d: -f2-)"
    readonly SRC_PATH
    DST_PATH="$(echo "${DESTINATION}" | cut -d: -f2-)"
    readonly DST_PATH

    [[ "${SRC_PATH}" = "${DST_PATH}" ]] || \
        fatal "For now only synchronization with the same path is supported! (${SRC_PATH} != ${DST_PATH})"

    TEMP_DIR="$(mktemp --directory -t sfsync-and-verify-XXXXXXX)"
    readonly TEMP_DIR

    LAST_CMD_OUTPUT="${TEMP_DIR}/last_cmd.out"
    readonly LAST_CMD_OUTPUT
    REPORT_DETAILS_FILE="${TEMP_DIR}/report-details.txt"
    readonly REPORT_DETAILS_FILE
    REPORT_FILE="${TEMP_DIR}/report.txt"
    readonly REPORT_FILE
    CMP_DETAILS="${TEMP_DIR}/compare-results.txt"
    readonly CMP_DETAILS
    CMP_STATS="${TEMP_DIR}/compare-stats.txt"
    readonly CMP_STATS

    if [[ "${LOG_EMAIL_CONTENT}" ]]; then
        [[ -e "${LOG_EMAIL_CONTENT}" ]] && fatal "File ${LOG_EMAIL_CONTENT} already exists!"
        touch "${LOG_EMAIL_CONTENT}" || fatal "Can not create ${LOG_EMAIL_CONTENT} file"
    fi

    REMOVED_OUTPUT_FILES="${TEMP_DIR}/removed_files.tmp"
    readonly REMOVED_OUTPUT_FILES

    trap on_exit EXIT
    trap _log_next_step DEBUG
}

_get_job_id_from_last_cmd_output() {
    sed --quiet "s/Job \\(.*\\) started/\\1/p" "${LAST_CMD_OUTPUT}"
}

_get_scan_id_from_last_cmd_output() {
    sed --quiet 's/Scan id: \(.*\)/\1/p' "${LAST_CMD_OUTPUT}"
}

_get_scan_fields() {
    local scan_id="$1"
    shift

    ${SF} scan show "${scan_id}" --format "$@"
}

_report_scan_results() {
    local volume_and_path="$1"
    local scan_id
    scan_id="$(_get_scan_id_from_last_cmd_output)"
    "${SF}" scan show "${scan_id}" | grep --extended "( dirs:| files:)" >> "${REPORT_DETAILS_FILE}"
    echo >> "${REPORT_DETAILS_FILE}"

    BROKEN_ENTRIES="$(_get_scan_fields "${scan_id}" broken_entry_count)"
    if [ "${BROKEN_ENTRIES}" -gt 0 ]; then
        local broken_entries
        broken_entries="$(_get_scan_fields "${scan_id}" broken_entries_hum)"
        {
            echo "ERR While scanning ${volume_and_path} found ${BROKEN_ENTRIES} entries that failed to scan"
            echo "ERR ${broken_entries}"
            echo
        } >> "${REPORT_DETAILS_FILE}"
    fi
}

scan_volume() {
    local volume_and_path="$1"
    local scan_type="$2"
    local return_code

    "${SF}" scan start "${volume_and_path}" --wait --wait-interval "${WAIT_INTERVAL}" --type "${scan_type}" &> "${LAST_CMD_OUTPUT}"
    # NOTE: even "set -e" the return code may be non-zero if that function was run as conditional
    #       for example:  "if scan_volume; then ..." or "scan_volume || error=$?"
    return_code=$?
    if [[ "${return_code}" == "0" ]]
    then
        echo -e "\\nScan stats of ${volume_and_path} (type ${scan_type}):" >> "${REPORT_DETAILS_FILE}"
        _report_scan_results "${volume_and_path}"
    fi
    return "${return_code}"
}

scan_source_volume() {
    # skip source scan if user directs
    if [[ "${SKIP_SOURCE_SCAN}" = "0" ]];
    then
        scan_volume  "${SOURCE}" "${SCAN_TYPE}"
    fi;
}

_handle_overlapping_scan() {
    local running_scan_id=$1

    if [[ "${STOP_SCAN}" ]]
    then
        echo "Stopping all scans running on volume ${DST_VOLUME}"
        "${SF}" scan stop --volume "${DST_VOLUME}" || true
    else
        echo "Waiting for scan ${running_scan_id}"
        "${SF}" scan wait --quiet --wait-interval "${WAIT_INTERVAL}" "${running_scan_id}" || true
    fi
}

scan_destination_volume() {
    local pending_retries="${NUM_RETRIES_TO_SCAN_DEST_VOLUME}"
    local running_scan_id
    local return_code=1

    while [[ "${pending_retries}" -gt 0 ]] && [[ "${return_code}" != 0 ]];
    do
        if scan_volume "${DESTINATION}" "${SCAN_TYPE}";
        then
            return_code=0
        else
            return_code=$?
            echo "Scan destination volume failed with code ${return_code}, pending retries: ${pending_retries}"
            case "${return_code}" in
                "10") # ExitCodes.OVERLAPPING_SCAN = 10
                    running_scan_id=$(cut -f2 -d"'" "${LAST_CMD_OUTPUT}")
                    _handle_overlapping_scan "${running_scan_id}"
                    ;;
                *)
                    echo "Unexpected error, wait ${RETRY_SCAN_INTERVAL} seconds before next try"
                    sleep "${RETRY_SCAN_INTERVAL}"
                    ;;
            esac;
            pending_retries="$((pending_retries - 1))"
        fi;
    done
    # NOTE: we don't need return code here, it's enough to verify if scan succeeded or not
    # Returning error code for some reason cause invoking TRAP DEBUG handler for this line
    # see STAR-7020 (it was already manifested within STAR-6647).
    [[ "${return_code}" = 0 ]]
}




synchronize() {
    set +u  # "${JOB_ARGS[@]}" requires use of this "set" directive - otherwise it fails with "unbound variable"
    # shellcheck disable=SC2086
    # no quotes are added because otherwise extra space is introduced
    "${SF}" job start ${ESTIMATIONS_OPTION} --no-prescan --wait --wait-interval "${WAIT_INTERVAL}" "${JOB_ARGS[@]}" \
        "copy ${COPY_ARGS}" "${SOURCE}" "${DESTINATION}" ${MANIFEST_OPT} &> "${LAST_CMD_OUTPUT}"
    set -u

    local job_id
    job_id="$(_get_job_id_from_last_cmd_output)"

    echo -e "\\nVolumes synchronization details:" >> "${REPORT_DETAILS_FILE}"
    _get_job_details "${job_id}" "copied" >> "${REPORT_DETAILS_FILE}"
}

_duration_from_ts () {
    local started_ts="$1"
    local ended_ts="$2"
    local seconds="$((ended_ts - started_ts))"
    local days="$((seconds / 86400))"

    seconds="$((seconds - days * 86400))"
    local dur_format="%H:%M:%S"
    if [ "${days}" -gt 0 ]; then
        [ "${days}" = 1 ] && dur_format="1 day %H:%M:%S" \
                          || dur_format="${days} days %H:%M:%S"
    fi
    date -u -d "0 ${seconds} seconds" +"${dur_format}"
}

_get_job_details() {
    local job_id="$1"
    local description="$2"
    local format="1:'   duration:' duration_hum 2:'@   failed entries:' :{fs_entries_failed:n} \
3:'@   newly ${description} entries:' :{fs_entries_done:n} 4:'@   newly ${description} data:' fs_bytes_done_hum 5:@"

    ${SF} job show "${job_id}" --format "${format}" | tr '@' '\n'
}

hash_volumes() {
    local src_job_id dst_job_id
    # shellcheck disable=SC2086
    # no quotes are added because otherwise extra space is introduced
    "${SF}" job start ${ESTIMATIONS_OPTION} --no-prescan hasher "${DESTINATION}" &> "${LAST_CMD_OUTPUT}"
    dst_job_id="$(_get_job_id_from_last_cmd_output)"

    # shellcheck disable=SC2086
    # no quotes are added because otherwise extra space is introduced
    "${SF}" job start ${ESTIMATIONS_OPTION} --no-prescan --wait --wait-interval "${WAIT_INTERVAL}" hasher "${SOURCE}" &> "${LAST_CMD_OUTPUT}"
    src_job_id="$(_get_job_id_from_last_cmd_output)"

    echo -e "\\nHashing source details:" >> "${REPORT_DETAILS_FILE}"
    _get_job_details "${src_job_id}" "hashed" >> "${REPORT_DETAILS_FILE}"

    # remove "source" to avoid confusion with sf-sync-and-verify source
    "${SF}" job wait "${dst_job_id}" 2>&1 | sed 's/^src volume:/volume:/' | sed 's/^src path:/root path:/' &> "${LAST_CMD_OUTPUT}"
    echo "Hashing destination details:" >> "${REPORT_DETAILS_FILE}"
    _get_job_details "${dst_job_id}" "hashed" >> "${REPORT_DETAILS_FILE}"

    rm "${LAST_CMD_OUTPUT}"
}

compare_subdirs() {
    local start_time end_time duration end_time_ts
    local store_removed_paths_in_file0=""
    start_time="$(date +"%s")"

    if [[ "${REMOVE_DEST}" = "1" ]]; then
        store_removed_paths_in_file0="${REMOVED_OUTPUT_FILES}"
    fi;

    "${CMP_FILE_TREE_PROG}" --escape-paths --type f --max-mtime "${START_TIME_TS}" \
        --store-removed-paths-in-file0 "${store_removed_paths_in_file0}" \
        "${SOURCE}" "${DESTINATION}" "${CMP_DETAILS}" 2>&1 | tee "${CMP_STATS}" > "${LAST_CMD_OUTPUT}"

    end_time_ts="$(date +"%s")"
    duration="$(_duration_from_ts "${start_time}" "${end_time_ts}")"
    echo "Duration of comparing all entries in db: ${duration}" >> "${REPORT_DETAILS_FILE}"
}

remove_destination_files_removed_from_source() {
    if [ -s "${REMOVED_OUTPUT_FILES}" ];
    then
        "${SF}" job start remove "${DST_VOLUME}:" --from-file0 "${REMOVED_OUTPUT_FILES}" --allow-overlapping-job \
            --no-post-verification --no-prescan --wait --wait-interval "${WAIT_INTERVAL}" &> "${LAST_CMD_OUTPUT}"

        local job_id
        job_id="$(_get_job_id_from_last_cmd_output)"
        local format="1:'@   removed files from destination:' :{fs_entries_done:n} 2:'   failed to remove files:' :{fs_entries_failed:n}@"
        ${SF} job show "${job_id}" --format "${format}" | tr '@' '\n' >> "${REPORT_DETAILS_FILE}" 2> "${LAST_CMD_OUTPUT}"
    fi;
}

create_report() {
    local start_time end_time duration end_time_ts total_duration
    end_time_ts="$(date +"%s")"
    total_duration="$(_duration_from_ts "${START_TIME_TS}" "${end_time_ts}")"
    start_time="$(date -d "@${START_TIME_TS}" +"${DATE_FORMAT}")"
    end_time="$(date -d "@${end_time_ts}" +"${DATE_FORMAT}")"
    {
        echo "Source: ${SOURCE}"
        echo "Destination: ${DESTINATION}"
        echo "Start time: ${start_time}"
        echo "End time: ${end_time}"
        echo "Duration: ${total_duration}"
        echo
        [[ "${BROKEN_ENTRIES}" -gt 0 ]] && echo "Entries that failed to scan: ${BROKEN_ENTRIES}"
        sed --quiet 's/.*newly copied \(.*\)/Newly copied \1/p' "${REPORT_DETAILS_FILE}"
        grep -v '^SUBJECT:' "${CMP_STATS}"
        echo -e "\\n\\n===============\\n=== Details ===\\n===============\\n"
        cat "${REPORT_DETAILS_FILE}"
    } > "${REPORT_FILE}"
}

report_result() {
    local stats_info
    stats_info="$(sed --quiet 's/SUBJECT: \(.*\)/\1/p' "${CMP_STATS}")"
    if [[ "${BROKEN_ENTRIES}" -gt 0 ]]; then
        stats_info="failed to scan entries: ${BROKEN_ENTRIES}, ${stats_info}"
    fi

    send_report "Sync Report ${SOURCE} to ${DESTINATION} ${stats_info}" \
                "${REPORT_FILE}" "${CMP_DETAILS}"
}

cleanup() {
    trap - EXIT
    trap - DEBUG

    remove_lock_path_if_not_locked
    clear_temp_files
}

validate_location() {
    local description=$1
    local location=$2

    local out
    out=$("${SF}" query "${location}" -type=d -maxdepth=0 --no-header)
    if [ -z "$out" ]; then
        fatal "Missing ${description} directory: ${location}
Please check if it's correct directory and if needed run scan on volume to ensure it's in database."

    fi;
}

verify_input_parameters() {
    [[ -n "${SOURCE:-}" ]] || fatal "Missing argument SOURCE"
    [[ -n "${DESTINATION:-}" ]] || fatal "Missing argument DESTINATION"

    validate_location "source" "${SOURCE}"

    # verify that ancestor of destination directory exists
    local parent
    parent=$(dirname "${DESTINATION}")
    if test "${parent}" = ".";
    then
        # we want to copy data directly to volume root so we verify only that volume root exists
        parent="${DESTINATION%%:*}:"
    fi
    validate_location "destination parent" "${parent}"
}

_prepare_message_after_lock_failed() {
    local mesg=$1

    echo ""
    echo "${mesg}"
    echo ""
    local previous_pid
    previous_pid=$(cat "${LOCK_FILE_PATH}")
    echo "Checking pid ${previous_pid}"
    ps "${previous_pid}" || {
        # it may happened that main process has disappeared (killed with SIGKILL) but some child
        # process is still running and is keeping flock...
        echo ""
        echo "Main process seems to be dead... trying to use lsof for processes that are keeping lock:"
        # when running from cron on centos lsof could be not present on PATH... let fix it:
        export PATH="${PATH}:/usr/sbin"
        lsof -- "${LOCK_FILE_PATH}" || {
            echo ""
            echo "lsof has failed, please check if it's properly installed and then run $(lsof -- "${LOCK_FILE_PATH}")"
        }
    }
}

get_lock_if_needed() {
    if [[ "${USE_LOCK}" ]]; then
        exec 9>> "${LOCK_FILE_PATH}"
        flock --nonblock 9 || {
            local subject="Found previous run of sfsync and verify for ${SOURCE} to ${DESTINATION}"
            local err_result="${TEMP_DIR}/report-lock-err.txt"

            _prepare_message_after_lock_failed "${subject}" &> "${err_result}"
            send_report "[FATAL] ${subject}" "${err_result}"
            cat "${err_result}"
            cleanup
            fatal
        }
        echo -n "$$" > "${LOCK_FILE_PATH}"
    fi;
}

verify_input_parameters
setup
get_lock_if_needed
scan_source_volume
synchronize
scan_destination_volume
hash_volumes
compare_subdirs
remove_destination_files_removed_from_source
create_report
report_result
cleanup
