/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */

WITH agents AS (
    SELECT
        default_agent_address AS agent,
        COUNT(*) AS volumes
    FROM sf_volumes.volume vol
    GROUP BY default_agent_address
), first_agent_activity AS (
    SELECT
        agent_address AS agent,
        MIN(creation_time::TIMESTAMP) AS start_time
    FROM sf_scans.scan
    GROUP BY agent_address
), agent_activity_time AS (
    SELECT agent,
    CASE WHEN '{{days_back}}' = '0' OR start_time > (NOW() - INTERVAL '{{days_back}} day') THEN
        EXTRACT(EPOCH FROM (NOW() - start_time))::INT
    ELSE
        EXTRACT(EPOCH FROM INTERVAL '{{days_back}} day')::INT
    END AS activity_time
    FROM first_agent_activity
), scan_agent_statistics AS (
    SELECT
        agent_address AS agent,
        COUNT(*) AS scans,
        MIN(creation_time::TIMESTAMP) AS first_response_time,
        MAX(heartbeat::TIMESTAMP) AS last_response_time
    FROM sf_scans.scan scan
    WHERE '{{days_back}}' = '0' OR creation_time >= (NOW() - INTERVAL '{{days_back}} day')
    GROUP BY agent_address
),
job_agent_info AS (
    SELECT
        vol.default_agent_address AS agent,
        job.id AS job_id,
        inc.started_at::TIMESTAMP AS start_time,
        job.heartbeat::TIMESTAMP AS end_time
    FROM sf_dispatcher.job job
    JOIN sf_dispatcher.incarnation inc ON inc.job_id=job.id
    JOIN sf_volumes.volume vol ON vol.id = job.volume_id
    WHERE '{{days_back}}' = '0' OR inc.started_at::TIMESTAMP >= (NOW() - INTERVAL '{{days_back}} day')
),
job_agent_statistics AS (
    SELECT
        agent,
        COUNT(DISTINCT job_id) AS jobs,
        MIN(start_time) AS first_response_time,
        MAX(end_time) AS last_response_time
    FROM job_agent_info
    GROUP BY agent
)

SELECT
    agents.agent AS agent,
    CONCAT((COALESCE(aat.activity_time, 0) / (24 * 60 * 60))::TEXT, ' day(s)') AS activity_time,
    COALESCE(scans.scans, 0) AS scans,
    COALESCE(ROUND((scans.scans / aat.activity_time::NUMERIC) * (7 * 24 * 60 * 60), 2), 0.00) AS "scans per week",
    COALESCE(jobs.jobs, 0) AS jobs,
    COALESCE(ROUND((jobs.jobs / aat.activity_time::NUMERIC) * (7 * 24 * 60 * 60), 2), 0.00) AS "jobs per week"
FROM agents
LEFT JOIN scan_agent_statistics scans ON scans.agent = agents.agent
LEFT JOIN job_agent_statistics jobs ON jobs.agent = agents.agent
LEFT JOIN agent_activity_time aat ON aat.agent = agents.agent
ORDER BY agent ASC
