/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */

WITH tables AS (
    SELECT tables.schemaname,
           tables.tablename,
           REGEXP_REPLACE(tables.tablename, '_part_[0-9]+', '') AS "parent_table",
           PG_TABLE_SIZE(CONCAT(tables.schemaname, '.', tables.tablename))::BIGINT AS data_size,
           PG_TOTAL_RELATION_SIZE(CONCAT(tables.schemaname, '.', tables.tablename))::BIGINT AS total_size,
           stats.n_live_tup AS live_tuples,
           stats.n_dead_tup AS dead_tuples
    FROM pg_tables tables
    LEFT JOIN pg_stat_all_tables stats ON stats.schemaname = tables.schemaname AND stats.relname = tables.tablename
    LEFT JOIN pg_class ON pg_class.oid = stats.relid
    WHERE tables.schemaname LIKE 'sf%' AND tables.schemaname != 'sf_internal' -- user reports_user does not have access to sf_internal
),
grouped_tables AS (
    SELECT schemaname,
           parent_table,
           SUM(data_size)::BIGINT AS data_size,
           SUM(total_size)::BIGINT AS total_size,
           COUNT(*) -1 AS partitions,
           SUM(live_tuples) AS live_tuples,
           SUM(dead_tuples) AS dead_tuples
    FROM tables
    GROUP BY parent_table, schemaname
)
SELECT schemaname AS "schema",
       SUM(total_size) AS "total size",
       SUM(data_size) AS "data size",
       COUNT(*) AS "number of tables",
       SUM(partitions)::INTEGER AS "number of table partitions",
       SUM(live_tuples)::BIGINT AS "live tuples",
       SUM(dead_tuples)::BIGINT AS "dead tuples",
       CASE WHEN SUM(live_tuples) > 0 THEN ROUND((SUM(dead_tuples) / SUM(live_tuples)::NUMERIC), 2)
       ELSE NULL END AS "ratio of dead/live tuples"
FROM grouped_tables
GROUP BY schemaname
ORDER BY schemaname
