#!/opt/starfish/examples/venv/bin/python3
"""
***********************************************************************************************************

 Starfish Storage Corporation ("Starfish") CONFIDENTIAL
 Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.

 NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
 Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
 intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
 U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
 Dissemination of this information or reproduction of this material is strictly forbidden unless prior
 written permission is obtained from Starfish. Access to the source code contained herein is hereby
 forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
 confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
 Starfish's software.

 ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
 THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
 AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
 FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
 DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
 WHOLE OR IN PART.

 FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
   These notices shall be marked on any reproduction of this data, in whole or in part.
   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
   Software clause at DFARS 52.227-7013.

***********************************************************************************************************
version 1.1 - doug 2020-01-13
* add json structure descent and usage example
"""
import json
import os
import subprocess
import sys
from datetime import datetime
from pprint import pprint
from urllib.parse import quote

import urllib3

try:
    import argparse
    import configparser
    from argparse import RawTextHelpFormatter
except Exception:
    print(
        "In order to use this, you must have configparser, argparse, "
        "installed (available via pip among "
        "other means.)",
        file=sys.stderr,
    )


def str_to_json(jse, fmt):
    """convert a format string into json substruct and bring it out of the job json, if possible.
    format json structs look like jobs.hash.result.md5
    :input: the json results for this record, and a format string
    :returns: a string for output or None"""

    jslist = fmt.split(".")
    if len(jslist) > 1:
        for i in range(len(jslist)):
            if args.debug > 1:
                print("checking arg " + str(i) + " in " + str(jslist))
            if jslist[i] in jse:
                jse = jse[jslist[i]]
                if args.debug > 1:
                    print("sub is " + str(jse))
            else:
                if args.debug > 1:
                    print(jslist[i] + " is not in " + str(jse))
                return None
        return str(jse)

    return None


# write out the manifest file to open file handle
def create_manifest(inargs):
    """main creation of manifest"""

    # read Starfish configs url to generate sfid
    config = configparser.ConfigParser(strict=False)

    # read config to get configured IP for API services
    if not os.access("/opt/starfish/etc/99-local.ini", os.R_OK):
        print("can't read credentials credentials from 99-local.ini. Try as root")
        sys.exit(1)

    config.read("/opt/starfish/etc/99-local.ini")
    sfurl = '<A HREF="' + config.get("global", "starfish_url_prefix") + "/#/details?sfid="
    sftrail = '">details</a>'

    if inargs.jobid:
        try:
            mf = subprocess.check_output(["sf", "manifest", "create", "-a", str(args.jobid)])
        except Exception:
            sys.exit(1)
    else:
        try:
            mf = subprocess.check_output(["sf", "manifest", "create", "-l", str(args.lljob)])
        except Exception:
            sys.exit(1)

    js = json.loads(mf)
    if len(js) == 0:
        # no rows - nothing archived in this job
        return

    proplist = inargs.format
    # print header
    if len(proplist) > 1:
        print(",".join(proplist))
    else:
        print(proplist[0])

    # loop over archive manifest info
    for jse in js:
        # continue on directories
        if jse["type"] == 16384:
            continue
        listitems = []

        for p in proplist:
            if p == "url":
                listitems.append(sfurl + quote(jse["sfid"]) + sftrail)
                continue
            if p == "run_time":
                listitems.append(
                    datetime.fromtimestamp(jse["jobs"]["upload-aws-archive-test"]["run_time"]).strftime(
                        "%Y/%m/%d %X %z"
                    )
                )
                continue
            if p == "dest":
                jobname = list(jse["jobs"].keys())[0]
                if "dest_name" not in jse["jobs"][jobname]["result"]:
                    listitems.append("not upload job")
                else:
                    listitems.append(
                        jse["jobs"][jobname]["result"]["dest_name"] + "/" + jse["jobs"][jobname]["result"]["path"]
                    )
                continue
            if inargs.debug > 1:
                pprint(jse)
            if inargs.debug:
                print("checking %s in JSON" % (p))  # noqa: S001
            # could the format string be a starfish reference to a json structure?
            res = str_to_json(jse, p)
            if res:
                listitems.append(str(res))
            elif p in jse:
                listitems.append(str(jse[p]))
            else:
                listitems.append("NoFieldMatch")

        print(",".join(listitems))


##########################################################################
# Main
##########################################################################
if __name__ == "__main__":
    # disable warnings
    urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

    ###############################
    # Parse command line arguments
    examples_text = (
        "Example: \n"
        "(for a hash job)\n"
        "./manifest.py --lljob 1714 --format vol_path uid size jobs.hash.result.md5\n"
        "\n"
        "To see a list of all valid built-in format specifiers, run sf job list -n 1 --json\n"
        "In addition to run_time, url, and dest (see --format above).\n"
        "you can also use sf query --job <jobid> to get different fields available associated\n"
        "with individual file and directory records."
    )

    parser = argparse.ArgumentParser(
        description="generated formatted manifest file", epilog=examples_text, formatter_class=RawTextHelpFormatter
    )

    parser.add_argument("--jobid", "-a", required=False, help="Archive job id (exclusive of lljob)")
    parser.add_argument("--lljob", "-l", required=False, help="low level job id (exclusive of archive job)")
    parser.add_argument("--debug", required=False, type=int, default=0, help="Turn on some debugging")
    parser.add_argument(
        "--format",
        nargs="+",
        required=True,
        help="a list of space separated keywords (no quoting) \n"
        "to format manifest output, chosen from fields in\n"
        "\x1B[1msf manifest create\x1B[0m. 'run_time', 'url',\n"
        "and 'dest' are special properties referring to\n"
        "when the archive job ran, a url to get the details\n"
        "of the item in question (hyperlink), and the\n"
        "archive destination",
    )
    args = parser.parse_args()

    if not args.format:
        print("must supply a list of items for --format", file=sys.stderr)
        sys.exit(1)

    if args.jobid and args.lljob:
        print("must specify only one of --jobid or --lljob")
        sys.exit(1)

    create_manifest(args)
    sys.exit(0)
