/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */

CREATE OR REPLACE FUNCTION sf_auth.custom_zone_role_capability_has_to_be_from_the_same_zone_and_delegable() RETURNS trigger AS
$$DECLARE
    role_zone_id            BIGINT;
    capability_zone_id      BIGINT;
    capability_delegable    BOOL;
BEGIN
    SELECT zone_id, is_delegable
      INTO capability_zone_id, capability_delegable
      FROM sf_auth.zone_v2_capability
     WHERE id = NEW.zone_capability_id;

    SELECT zone_id
      INTO role_zone_id
      FROM sf_auth.custom_zone_role
     WHERE id = NEW.custom_zone_role_id;

    IF role_zone_id <> capability_zone_id
    THEN
        RAISE 'zone capability (id: %) and custom zone role (id: %) are in different zones: % and %',
                NEW.zone_capability_id,
                NEW.custom_zone_role_id, capability_zone_id, role_zone_id
            USING ERRCODE = 'check_violation';
    END IF;

    IF NOT capability_delegable
    THEN
      RAISE 'zone capability (id: %) cannot be added to custom zone role because it is not delegable, zone id: %',
            NEW.zone_capability_id, capability_zone_id
        USING ERRCODE = 'check_violation';
    END IF;

    RETURN NEW;
END;$$ LANGUAGE plpgsql STABLE PARALLEL SAFE;

CREATE CONSTRAINT TRIGGER check_custom_zone_role_capability_has_to_be_from_the_same_zone_and_delegable
   AFTER INSERT OR UPDATE ON sf_auth.custom_zone_role_capability
   DEFERRABLE INITIALLY DEFERRED
   FOR EACH ROW
   EXECUTE PROCEDURE sf_auth.custom_zone_role_capability_has_to_be_from_the_same_zone_and_delegable();


CREATE OR REPLACE FUNCTION sf_auth.custom_zone_role_capability_resource_has_to_be_available_in_zone() RETURNS trigger AS
$$DECLARE
    role_zone_id            BIGINT;
BEGIN
    IF NOT (SELECT is_available_in_all_zones
              FROM sf_auth.resource
             WHERE unique_id = NEW.resource_unique_id)
    THEN
        SELECT zone_id
          INTO role_zone_id
          FROM sf_auth.custom_zone_role AS role
          JOIN sf_auth.custom_zone_role_capability AS role_capability ON role.id = role_capability.custom_zone_role_id
         WHERE role_capability.id = NEW.custom_zone_role_capability_id;

        IF NOT EXISTS (SELECT 1
                         FROM sf_auth.resource_in_zone_v2
                        WHERE resource_unique_id = NEW.resource_unique_id
                          AND zone_id = role_zone_id)
        THEN
          RAISE 'resource is not available in zone, zone id: %, resource unique id: %',
                role_zone_id, NEW.resource_unique_id
            USING ERRCODE = 'check_violation';
        END IF;
    END IF;

    RETURN NEW;
END;$$ LANGUAGE plpgsql STABLE PARALLEL SAFE;

CREATE CONSTRAINT TRIGGER check_custom_zone_role_capability_resource_has_to_be_available_in_zone
   AFTER INSERT OR UPDATE ON sf_auth.custom_zone_role_capability_resource
   DEFERRABLE INITIALLY DEFERRED
   FOR EACH ROW
   EXECUTE PROCEDURE sf_auth.custom_zone_role_capability_resource_has_to_be_available_in_zone();


CREATE OR REPLACE FUNCTION sf_auth.resource_in_zone_v2_cannot_be_deleted_if_in_custom_zone_role() RETURNS trigger AS
$$DECLARE
    role_zone_id            BIGINT;
BEGIN
    IF EXISTS (SELECT 1
                 FROM sf_auth.custom_zone_role_capability_resource cap_res
                 JOIN sf_auth.custom_zone_role_capability cap ON cap.id = cap_res.custom_zone_role_capability_id
                 JOIN sf_auth.custom_zone_role role ON role.id = cap.custom_zone_role_id
                WHERE cap_res.resource_unique_id = OLD.resource_unique_id
                  AND role.zone_id = OLD.zone_id)
    THEN
      RAISE 'resource is used in custom zone roles, zone id: %, resource unique id: %',
            OLD.zone_id, OLD.resource_unique_id
        USING ERRCODE = 'check_violation';
    END IF;

    RETURN OLD;
END;$$ LANGUAGE plpgsql STABLE PARALLEL SAFE;


CREATE CONSTRAINT TRIGGER check_resource_in_zone_v2_cannot_be_deleted_if_in_custom_zone_role
   AFTER DELETE ON sf_auth.resource_in_zone_v2
   DEFERRABLE INITIALLY DEFERRED
   FOR EACH ROW
   EXECUTE PROCEDURE sf_auth.resource_in_zone_v2_cannot_be_deleted_if_in_custom_zone_role();
