/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */
-- Why custom SQL instead of extracting SQL from `sf query --latest-copied-version` and wrapping it
-- with a GROUP BY?
-- Because of performance. Implementation inside sf query has to be generic and produce a good results
-- for big and small subsets. For example, one user can run a query on the whole volume and the second
-- one on a small subtree. In the case of recoverable files, we know that we will calculate for the
-- whole volume so we can prepare -> create temp table with copy_job_results for a given
-- volume which is very helpful. We can further get better result by grouping data as fast as possible
-- (for example group files by parent_id) which makes next joins faster. Generic solution from sf query
-- creates one big query which joins sf.file with sf.dir and then lateral join with sf.job_result which
-- can be super slow for volumes with small amount of copy_job results but huge amount of other job results.
-- Tests on dogfood showed that custom SQL is ~7 times faster.

CREATE TEMP TABLE copy_job_res ON COMMIT DROP AS
    SELECT DISTINCT ON (fs_entry_id) volume_id, fs_entry_id, id  as job_result_id, job_id, mtime, ctime
    FROM sf.job_result
    WHERE result->>'copy_type' IN ('vol','storage') and volume_id IN :volume_ids
    ORDER BY fs_entry_id, job_result_id DESC;
ANALYZE copy_job_res;

CREATE INDEX copy_job_res__vol_id_entry_id ON copy_job_res (volume_id, fs_entry_id);

CREATE TEMP TABLE files_recoverable ON COMMIT DROP AS
    SELECT DISTINCT ON (file.parent_id, file.name)
        file.id,
        file.name,
        file.parent_id,
        file.size,
        file.blocks,
        file.nlinks,
        file.volume_id,
        file.valid
    FROM sf.file as file
        JOIN copy_job_res as job ON  job.fs_entry_id = file.id
    WHERE
        file.volume_id = job.volume_id
        AND file.volume_id IN :volume_ids
        AND (file.ctime = job.ctime OR job.ctime IS NULL)
        AND (file.mtime = job.mtime OR job.mtime IS NULL)
    ORDER BY file.parent_id, file.name, LOWER(file.valid) DESC;
ANALYZE files_recoverable;

-- In code above we look only at files and ignore size and count of directories.
-- This is inconsistent with recursive_aggregates which do take dirs into account.
-- Including the size and the count of directories would give better user experience
-- for clients that run archive jobs. But there is an important tradeoff related
-- to how --latest-copied-version works.
-- Imagine a fresh system where no archive was run yet. What should be shown in a recoverable
-- aggregate for a random directory? As a user would expect that there should be 0 recoverable files.
-- In order to be 100% the same as --latest-copied-version we should show the aggregates size
-- of all directories in that subtree, which can be misleading. At the same time from the technical
-- point of view that would mean that we would have to keep recoverable aggregates for each
-- directory path that ever existed. That would mean that on each system we keep a copy
-- of the directory path in the second table even where there was no archive ever run on that system.
-- This would also be much more compute-intensive to calculate. This approach which looks only at files
-- is not perfect but should be good enough.

CREATE TEMP TABLE files_per_path ON COMMIT DROP AS
    SELECT
        volume_id,
        path,
        SUM(size) AS size,
        SUM(blocks / COALESCE(nlinks, 1)) AS blocks_div_nlinks,
        COUNT(*) AS files_count
    FROM (
        SELECT DISTINCT ON (f.volume_id, d.path, f.name)
            f.volume_id,
            d.path,
            f.size,
            f.blocks,
            f.nlinks
        FROM files_recoverable as f
        JOIN sf.dir as d on f.parent_id = d.id and f.volume_id = d.volume_id AND tstzrange_contains_upper(d.valid, upper(f.valid))
        where d.volume_id IN :volume_ids
        ORDER BY f.volume_id, d.path, f.name, LOWER(f.valid) DESC
    ) t
    GROUP BY volume_id, path;

ANALYZE files_per_path;

DELETE FROM sf.recoverable_aggrs WHERE volume_id IN :volume_ids;

INSERT INTO sf.recoverable_aggrs
    SELECT
        volume_id,
        path,
        SUM(size) AS size,
        SUM(blocks_div_nlinks) AS blocks_div_nlinks,
        SUM(files_count) AS files_count,
        SUM(local_files_count) AS local_files_count,
        SUM(local_size) AS local_size
    FROM (
        SELECT
            volume_id,
            sf.posix_path_ancestors(path, include_self:=TRUE) AS path,
            0 local_size,
            0 AS local_files_count,
            size,
            blocks_div_nlinks,
            files_count
        FROM files_per_path
        UNION ALL
        SELECT
            volume_id,
            path,
            size AS local_size,
            files_count AS local_files_count,
            0 AS size,
            0 AS blocks_div_nlinks,
            0 AS files_count
        FROM files_per_path
    ) t
    GROUP BY volume_id, path;
