#!/opt/starfish/examples/venv/bin/python3
"""
***********************************************************************************************************

 Starfish Storage Corporation ("Starfish") CONFIDENTIAL
 Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.

 NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
 Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
 intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
 U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
 Dissemination of this information or reproduction of this material is strictly forbidden unless prior
 written permission is obtained from Starfish. Access to the source code contained herein is hereby
 forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
 confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
 Starfish's software.

 ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
 THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
 AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
 FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
 DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
 WHOLE OR IN PART.

 FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
   These notices shall be marked on any reproduction of this data, in whole or in part.
   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
   Software clause at DFARS 52.227-7013.

***********************************************************************************************************
"""

import argparse
import configparser
import contextlib
import copy
import csv
import dataclasses
import datetime
import logging
import os
import signal
import sys
import time
from argparse import BooleanOptionalAction
from concurrent import futures
from concurrent.futures import FIRST_COMPLETED, ThreadPoolExecutor
from functools import wraps
from pathlib import Path
from textwrap import dedent
from typing import Optional
from urllib.parse import parse_qs, urlencode, urlparse, urlunparse

import psycopg2

REINDEX_DB_EXEC_NAME = os.path.basename(__file__)
logger = logging.getLogger(REINDEX_DB_EXEC_NAME)

stop_requested = False

FILE_LOG_FORMAT = (
    "%(levelname).3s %(asctime)s %(name)s %(process)d '%(threadName)s': %(message)s [%(filename)s:%(lineno)d]"
)

TMUX_SCREEN_RECOMMENDATION = """\
Reindexing must be done in a screen or tmux session.
It may run for a long time and a ssh client usually times out, killing the process.
See https://linux.die.net/man/1/screen or `man screen` for details about using screen
and https://linux.die.net/man/1/tmux or `man tmux` for details about tmux.
"""

HELP_EPILOG = f"""
EXAMPLES

    It is recommended to run this command when Starfish is stopped.
    In that case, use options to reindex all text indexes, both starfish and postgres internal indexes:

        > {REINDEX_DB_EXEC_NAME} --starfish-text-indexes --pg-text-indexes

    If Starfish cannot be stopped or for any reason the command must be run
    when Starfish is running then it is recommended to reindex only
    starfish indexes in concurrently mode:

        > {REINDEX_DB_EXEC_NAME} --starfish-text-indexes --concurrently

    Then, if there is a window when starfish is down, postgres indexes may be rebuilt:

        > {REINDEX_DB_EXEC_NAME} --pg-text-indexes

   Running the command with --starfish-text-indexes option will take significantly more time
   than running it with --pg-text-indexes.

"""

BRIGHT = "\033[1m"
RED = "\033[31m"
RESET_ALL = "\033[0m"

DEFAULT_SFHOME = "/opt/starfish"

# Both schemas may contain ephemeral tables which may be removed
# between getting the name of index from DB and starting actual reindexing.
# And there should be no need to reindex indexes from these schemas.
EXCLUDED_SCHEMAS = ["sf_internal", "fs_entry_queues"]


def _cli_print(text, out_file):
    if out_file is not None:
        out_file.write(text)
        out_file.flush()


def log(msg, to_file_only=False):
    logger.info(msg)
    if to_file_only:
        return
    _cli_print(f"{msg}\n", out_file=sys.stdout)


def log_error(msg):
    logger.error(msg)
    _cli_print(f"{BRIGHT}{RED}{msg}{RESET_ALL}\n", out_file=sys.stderr)


def cli_print_progress(prefix: str, done: int, total: int, suffix=""):
    total = max(done, total)
    msg = prefix + f"{done}/{total}" + suffix
    log(msg)


def get_confirmation_for(msg):
    accepted_answers = ("y", "yes")
    logger.info(f"prompting for confirmation ({msg})")
    _cli_print(msg, out_file=sys.stdout)
    answer = input("")
    logger.info(f"answer read: {answer}")
    if answer.lower() in accepted_answers:
        return True
    return False


def confirm_running_in_terminal_multiplexer(recommendation_text):
    is_tmux = os.getenv("TMUX") is not None
    screen_env_variable = os.getenv("TERM")
    is_screen = screen_env_variable is not None and screen_env_variable.startswith("screen")
    if is_screen or is_tmux:
        return True

    message = dedent(
        f"""{recommendation_text}
Do you want to continue running this script? (y/N) """
    )
    return get_confirmation_for(message)


class ExitCodes:
    # No exit code can be equal to 2 as ArgumentParser uses 2 if parsing failed
    SUCCESS = 0
    # exit code 1 is used if exception is thrown (in case of pyinstaller it is -1)
    ARGPARSE_ERROR = 2
    INTERRUPTED = 4
    CONFIG_ERROR = 5
    DATABASE_ERROR = 21
    PERMISSION_DENIED = 31
    GENERAL_FAILURE = 99


class PG:
    @dataclasses.dataclass(slots=True)
    class Filters:
        pg_catalog_indexes: Optional[bool] = None
        index_name_pattern: Optional[str] = None
        from_file: Optional[list["PG.IndexInfo"]] = None

    @dataclasses.dataclass(slots=True, frozen=True, eq=True)
    class IndexInfo:
        schema: Optional[str]
        name: str
        table: Optional[str]

        def __str__(self):
            return f"Index {self.name} on {self.schema}.{self.table}"

    def __init__(self, db_uri, application_name=None):
        if application_name:
            # max length of application name is 64 characters
            db_uri = self._set_application_name(application_name[:64], db_uri)
        try:
            self._connection = psycopg2.connect(db_uri)
            self._connection.set_session(autocommit=True)
        except psycopg2.DatabaseError as e:
            raise CLIException(f"Unable to connect to DB: {e}", ExitCodes.DATABASE_ERROR)

    @staticmethod
    def _set_application_name(application_name, db_uri):
        parsed_uri = urlparse(db_uri)
        parsed_query = parse_qs(parsed_uri.query)
        parsed_query["application_name"] = application_name
        parsed_uri = parsed_uri._replace(query=urlencode(parsed_query, doseq=True))
        db_uri = urlunparse(parsed_uri)
        return db_uri

    def disconnect(self):
        self._connection.close()

    def execute(self, query, params=None, verbose=False) -> list:
        log(f"Running: {query}", to_file_only=not verbose)
        cursor = self._connection.cursor()
        cursor.execute(query, params)
        return cursor.fetchall() if cursor.rowcount > 0 else []

    def get_indexes(self, filters: Filters) -> list["PG.IndexInfo"]:
        schema_condition = ""
        if filters.pg_catalog_indexes is not None:
            schema_condition = f"AND schema_info.nspname {'=' if filters.pg_catalog_indexes else '<>'} 'pg_catalog'"
        excluded_schema_names = ", ".join([f"'{name}'" for name in EXCLUDED_SCHEMAS])
        schema_condition += f" AND schema_info.nspname NOT IN ({excluded_schema_names})"
        query_params = {}
        index_condition = ""
        if filters.index_name_pattern:
            index_condition = "AND index_info.relname ~* %(index_name_pattern)s"
            query_params["index_name_pattern"] = filters.index_name_pattern
        query = f"""-- This CTE calculates all text-like types and types which are arrays of texts.
            WITH all_text_like_types AS (
                SELECT oid
                FROM pg_type
                WHERE
                    -- 'S' means "String types"
                    -- https://www.postgresql.org/docs/current/catalog-pg-type.html#CATALOG-TYPCATEGORY-TABLE
                    typcategory = 'S' OR
                    -- it is not possible to create index on json or xml types, thus only jsonb
                    typname = 'jsonb' OR
                    -- It is for index on column that type is "name" (names of PG objects).
                    -- For some reason the index'es type is cstring and its typcategory is 'P' (pseudo type)
                    typname = 'cstring'
                UNION
                -- In order to "decode" array type we need to query for additional row which is
                -- referenced by typelem value (oid of row describing the type of array elements).
                SELECT array_type.oid
                FROM pg_type array_type
                JOIN pg_type elem_type on elem_type.oid = array_type.typelem AND array_type.typelem <> 0
                WHERE
                    -- 'A' means Array types
                    array_type.typcategory = 'A' AND (
                        elem_type.typcategory = 'S' OR
                        elem_type.typname = 'jsonb' OR
                        elem_type.typname = 'cstring'
                    )
            )
            SELECT DISTINCT
                index_info.relname AS index_name,
                schema_info.nspname AS schema_name,
                table_info.relname AS table_name
            FROM pg_index
            JOIN pg_class index_info ON index_info.oid = pg_index.indexrelid
            JOIN pg_class table_info ON table_info.oid = pg_index.indrelid
            JOIN pg_namespace schema_info ON schema_info.oid = table_info.relnamespace
            JOIN pg_attribute attr on attr.attrelid = (schema_info.nspname || '.' || index_info.relname)::regclass::oid
            JOIN all_text_like_types on all_text_like_types.oid = attr.atttypid
            WHERE
                -- we take here only non-partitioned indexes for 2 reasons:
                -- - to avoid reindexing some indexes twice (partitioned index vs its child index)
                -- - to make reindexing more granular and increase parallelism of reindexing
                index_info.relkind = 'i'::char
                {schema_condition}
                {index_condition}
            ORDER BY
                schema_name,
                table_name,
                index_name
        """  # noqa S608
        result = []
        rows = self.execute(query, params=query_params)
        for row in rows:
            index_name, schema, table = row
            result.append(PG.IndexInfo(schema, index_name, table))
        return result


@dataclasses.dataclass(slots=True)
class ProgressStats:
    done_count: int
    failed_count: int
    total_count: int
    in_progress: list[PG.IndexInfo]
    failed: dict[PG.IndexInfo, str]
    failed_threads: list[str]


class ReindexState:
    def __init__(self, indexes, concurrently):
        self._to_be_done = copy.copy(indexes)
        self._number_of_all_indexes = len(indexes)
        self._failed = {}
        self._done = set()
        self._in_progress = set()
        self._failed_threads = []
        self._concurrently = concurrently

    def pop(self) -> PG.IndexInfo:
        index = self._pop_next_available()
        self._in_progress.add(index)
        return index

    def _pop_next_available(self) -> PG.IndexInfo:
        if self._concurrently:
            # Only one index on a single table can be reindex concurrently at a time.
            # Thus, we group them into batches per table to process them one by one.
            tables_in_progress = {index.table for index in self._in_progress}
            index = next(index for index in self._to_be_done if index.table not in tables_in_progress)
            self._to_be_done.remove(index)
            return index
        else:
            return self._to_be_done.pop(0)

    def succeeded(self, index):
        self._done.add(index)
        self._in_progress.remove(index)

    def failed(self, index, error):
        self._failed[index] = error
        self._in_progress.remove(index)

    def get_stats(self) -> ProgressStats:
        done_count = len(self._done)
        failed_count = len(self._failed)
        return ProgressStats(
            done_count=done_count,
            failed_count=failed_count,
            total_count=self._number_of_all_indexes,
            in_progress=list(self._in_progress),
            failed=copy.copy(self._failed),
            failed_threads=copy.copy(self._failed_threads),
        )

    def print_stats(self):
        stats = self.get_stats()
        in_progress = [index.name for index in stats.in_progress]
        description = ""
        if stats.failed_count:
            description += f", failed: {stats.failed_count}"
        if in_progress:
            description += f', in progress: {", ".join(in_progress)}'

        cli_print_progress(prefix="Processed: ", done=stats.done_count, total=stats.total_count, suffix=description)


@contextlib.contextmanager
def connect_to_db(*args, **kwargs):
    pg = PG(*args, **kwargs)
    try:
        yield pg
    finally:
        pg.disconnect()


def reindex(
    db_uri: str,
    *,
    concurrently: bool,
    description: str,
    filters: PG.Filters,
    list_only: bool,
    verbose: bool,
    workers: int,
):
    with connect_to_db(db_uri, REINDEX_DB_EXEC_NAME) as pg:
        indexes = pg.get_indexes(filters)

    if list_only:
        write_indexes_to_file(indexes, sys.stdout)
        return ExitCodes.SUCCESS, []

    logger.info(f"All indexes: {indexes}")
    if filters.from_file:
        indexes = filter_all_indexes_by_requested_indexes(indexes, filters.from_file)

    if len(indexes):
        log(f"Reindexing {len(indexes)} {description}")
    else:
        log(f"No {description} to be reindexed")
        return ExitCodes.SUCCESS, []
    reindex_state = ReindexState(indexes, concurrently)
    with ThreadPoolExecutor(max_workers=workers, thread_name_prefix="worker") as pool:
        pending_futures = {}
        last_update = time.time()
        previously_processed = 0
        while True:
            # Submit indexes to be processed to make work for all workers.
            # We cannot submit all indexes at once because, in the case of concurrently mode
            # wwe have to avoid running 2 reindex operations on indexes from one table.
            stats = reindex_state.get_stats()
            if stats.failed_count + stats.done_count >= stats.total_count:
                break
            in_progress_count = len(stats.in_progress)
            for _ in range(workers - in_progress_count):
                try:
                    index = reindex_state.pop()
                except (IndexError, StopIteration):
                    break
                future = pool.submit(
                    run_reindex_sql,
                    db_uri=db_uri,
                    index=index,
                    concurrently=concurrently,
                    verbose=verbose,
                )
                pending_futures[future] = index

            # print stats
            now = time.time()
            stats = reindex_state.get_stats()
            processed_indexes_count = stats.done_count + stats.failed_count + len(stats.in_progress)
            if now - last_update > 1 and previously_processed != processed_indexes_count:
                reindex_state.print_stats()
                previously_processed = processed_indexes_count
                last_update = now

            # wait for any index being rebuilt
            done, _ = futures.wait(pending_futures.keys(), timeout=1, return_when=FIRST_COMPLETED)
            for future in done:
                if future.exception():
                    reindex_state.failed(pending_futures[future], future.exception())
                else:
                    reindex_state.succeeded(pending_futures[future])
                pending_futures.pop(future)

            if stop_requested:
                log("Requested to stop by signal")
                raise SystemExit(ExitCodes.INTERRUPTED)

    stats = reindex_state.get_stats()
    log(f"Reindexing stats: {stats}", to_file_only=not verbose)
    log(f"\nSuccessfully reindexed: {stats.done_count}, failed: {stats.failed_count}")
    if stats.failed_count:
        log_error("Reindexing of the following indexes failed:")
        for index, error in stats.failed.items():
            log_error(f"{index} failed with error: {error}")
        return ExitCodes.DATABASE_ERROR, []
    return ExitCodes.SUCCESS, indexes


def filter_all_indexes_by_requested_indexes(
    all_indexes: list[PG.IndexInfo], requested_indexes: list[PG.IndexInfo]
) -> list[PG.IndexInfo]:
    result = []
    for requested_index in requested_indexes:
        if requested_index in all_indexes:
            result.append(requested_index)
        elif requested_index.schema is None:
            # Only names of indexes given in the input file.
            # Maybe there are more than one index matching the given name (in different schemas).
            for index in all_indexes:
                if index.name == requested_index.name:
                    result.append(index)
    return result


def run_reindex_sql(db_uri, index, concurrently, verbose):
    with connect_to_db(db_uri, REINDEX_DB_EXEC_NAME) as pg:
        concurrently = "CONCURRENTLY" if concurrently else ""
        log(f"Starting reindexing {index}", to_file_only=not verbose)
        pg.execute(f"REINDEX INDEX {concurrently} {index.schema}.{index.name}", verbose)
        log(f"Finished reindexing {index}", to_file_only=not verbose)


def create_parser(default_config_dir, default_log_dir):
    parser = argparse.ArgumentParser(
        prog=REINDEX_DB_EXEC_NAME,
        description=f"""Reindex text indexes in starfish DB
{TMUX_SCREEN_RECOMMENDATION}""",
        formatter_class=argparse.RawTextHelpFormatter,
        add_help=True,
        epilog=HELP_EPILOG,
    )
    parser.set_defaults(action_name="sfclient.actions.reindex_db.reindex_db_with_confirmation")
    parser.add_argument("-y", "--yes", action="store_true", default=False, help="Do not require confirmation")
    parser.add_argument("--ignore-no-tmux-and-screen", action="store_true", default=False, help=argparse.SUPPRESS)
    parser.add_argument(
        "--config-dir",
        type=str,
        default=default_config_dir,
        help=dedent(
            """As all starfish services should be stopped before running this command
it is required to specify config path to load config from"""
        ),
    )
    parser.add_argument(
        "--log-dir",
        type=str,
        default=default_log_dir,
        help=dedent("Logs directory"),
    )
    parser.add_argument(
        "--db-uri",
        type=str,
        metavar="PATTERN",
        help="URI to Postgres database to be reindexed. starfish db configured in the config is used by deafult.",
    )
    default_workers = 5
    parser.add_argument(
        "--workers",
        default=default_workers,
        type=int,
        help=dedent(
            f"""\
            Number of workers that will be used to parallelize reindexing.
            This argument is ignored when reindexing postgres indexes
            since it is not safe to reindex them in parallel.
            Default: {default_workers}"""
        ),
    )

    parser.add_argument(
        "--starfish-text-indexes",
        action=BooleanOptionalAction,
        default=False,
        help="reindex indexes other than postgres internal indexes",
    )
    parser.add_argument(
        "--pg-text-indexes",
        action=BooleanOptionalAction,
        default=False,
        help="""reindex postgres internal indexes
Do not use this option when Starfish is running.""",
    )
    parser.add_argument(
        "--index-name-pattern",
        type=str,
        metavar="PATTERN",
        help="reindex only indexes that name contains given pattern",
    )
    parser.add_argument(
        "--concurrently",
        action=BooleanOptionalAction,
        default=False,
        help="""reindex using concurrently mode, applies only to non-internal postgres indexes
This option is recommended when running the script when Starfish system is on.""",
    )
    parser.add_argument(
        "--list",
        action=BooleanOptionalAction,
        default=False,
        help="""Does not reindex any index. Only prints to STDOUT CSV list of indexes that would be reindexed.
Columns: schema, index name, table name.""",
    )
    parser.add_argument(
        "--from-file",
        metavar="PATH",
        type=argparse.FileType("r"),
        help="""Path to csv file containing list of indexes to be reindexed.
Accepts 2 formats:
    1. Columns as printed when --list option is used: schema, index name, table name
    2. Only one column: index name""",
    )

    log_group = parser.add_argument_group("Logging options")
    log_group.add_argument("-v", "--verbose", action="store_true", help="be verbose - set all log levels to DEBUG")

    return parser


def parse_args(args, default_config_dir, default_log_dir):
    parser = create_parser(default_config_dir, default_log_dir)
    parsed_args = parser.parse_args(args)
    return parsed_args


def setup_logging(log_dir: str):
    log_formatter = logging.Formatter(FILE_LOG_FORMAT)
    root_logger = logging.getLogger()
    root_logger.setLevel(logging.DEBUG)

    timestamp = datetime.datetime.fromtimestamp(time.time()).strftime("%Y%m%d_%H%M%S")
    file_path = str(Path(log_dir) / f"{REINDEX_DB_EXEC_NAME}-{timestamp}.log")
    fileHandler = logging.FileHandler(file_path)
    fileHandler.setFormatter(log_formatter)
    root_logger.addHandler(fileHandler)


def request_stop(signal_number, frame):
    global stop_requested
    stop_requested = True


def setup_cli():
    if sys.getdefaultencoding() != "utf-8":
        raise RuntimeError(f"Unexpected encoding: {sys.getdefaultencoding()}")

    signal.signal(signal.SIGTERM, request_stop)


def db_name_from_db_uri(db_uri: str):
    parsed_uri = urlparse(db_uri)
    return parsed_uri.path[1:]


class CLIException(Exception):
    def __init__(self, msg, exit_code):
        self.msg = msg
        self.exit_code = exit_code


def get_db_uri(config_dir: str):
    config_file = str(Path(config_dir) / "99-local.ini")
    if not os.access(config_file, os.R_OK):
        raise CLIException(f"No access to {config_file}.", ExitCodes.PERMISSION_DENIED)
    config = configparser.ConfigParser()
    try:
        config.read(config_file)
        return config.get("pg", "pg_uri")
    except configparser.Error as exc:
        logger.exception(f"Failed to get DB URI from config: {exc}")
        raise CLIException(f"Failed to get DB URI from config ({config_file}): {exc}", ExitCodes.CONFIG_ERROR)


def cli_exception_wrapper(main_function):
    @wraps(main_function)
    def wrapped_main(*args):
        try:
            return main_function(*args)
        except CLIException as exc:
            log_error(exc.msg)
            return exc.exit_code
        except psycopg2.Error as exc:
            log_error(exc)
            return ExitCodes.DATABASE_ERROR
        except KeyboardInterrupt:
            log_error("Interrupted by user")
            return ExitCodes.INTERRUPTED
        except SystemExit as exc:
            return exc.code
        except BaseException as exc:
            log_error(f"Unexpected exception: {exc} ({type(exc)})")
            logger.exception("Unexpected exception in main")
            return ExitCodes.GENERAL_FAILURE

    return wrapped_main


def humanized_time(time_):
    return datetime.datetime.fromtimestamp(time_).strftime("%Y-%m-%d %H:%M:%S")


def huminized_interval(interval_in_seconds: int):
    return str(datetime.timedelta(seconds=interval_in_seconds))


def read_indexes_from_file(file) -> list[PG.IndexInfo]:
    reader = csv.reader(file)
    indexes = []
    for row in reader:
        match row:
            case [index_name]:
                index = PG.IndexInfo(None, index_name, None)
            case [schema, index_name, table_name]:
                index = PG.IndexInfo(schema, index_name, table_name)
            case _:
                raise CLIException(f"Cannot parse row: {row}", ExitCodes.GENERAL_FAILURE)
        indexes.append(index)
    return indexes


def write_indexes_to_file(indexes: list[PG.IndexInfo], file, only_name=False):
    writer = csv.writer(file)
    for index in indexes:
        if only_name:  # used in tests
            writer.writerow([index.name])
        else:
            writer.writerow([index.schema, index.name, index.table])


def error_if_unused_indexes_from_file(from_file, reindexed_from_file):
    success = True
    reindexed_from_file_names = [index.name for index in reindexed_from_file]
    for index in from_file:
        if index in reindexed_from_file:
            continue

        if index.schema is None and index.name in reindexed_from_file_names:
            # given csv file had only index names
            continue

        log_error(f"Index not found: {index}")
        success = False

    return ExitCodes.SUCCESS if success else ExitCodes.GENERAL_FAILURE


@cli_exception_wrapper
def main(cmd_args):
    sfhome = Path(os.environ.get("SFHOME", DEFAULT_SFHOME))
    default_config_dir = str(sfhome / "etc")
    default_log_dir = str(sfhome / "log")
    args = parse_args(cmd_args, default_config_dir, default_log_dir)
    setup_cli()
    setup_logging(args.log_dir)

    log(f"Started with: {args}", to_file_only=True)

    if not args.starfish_text_indexes and not args.pg_text_indexes and not args.from_file:
        log_error("At least one of --pg-text-indexes, --starfish-text-indexes or --from-file option is required")
        return ExitCodes.ARGPARSE_ERROR

    if args.from_file and (args.starfish_text_indexes or args.pg_text_indexes):
        log_error("--from-file cannot be used along with --starfish-text-indexes or --pg-text-indexes options")
        return ExitCodes.ARGPARSE_ERROR

    if args.index_name_pattern and args.from_file:
        log_error("--index-name-pattern cannot be used with --from-file option")
        return ExitCodes.ARGPARSE_ERROR

    if args.from_file and args.list:
        log_error("--from-file cannot be used along with --list option")
        return ExitCodes.ARGPARSE_ERROR

    if args.pg_text_indexes and not args.starfish_text_indexes and args.concurrently:
        log_error("--concurrently option cannot be used to reindex Postgres internal indexes")
        return ExitCodes.ARGPARSE_ERROR

    if (
        not args.list
        and not args.yes
        and not get_confirmation_for(
            "Depending on the database size the operation can take hours or even days. "
            "Do you really want do reindex indexes (y/N)?: "
        )
    ):
        return ExitCodes.INTERRUPTED

    if (
        not args.list
        and not args.ignore_no_tmux_and_screen
        and not confirm_running_in_terminal_multiplexer(TMUX_SCREEN_RECOMMENDATION)
    ):
        return ExitCodes.INTERRUPTED

    db_uri = args.db_uri if args.db_uri else get_db_uri(args.config_dir)
    db_name = db_name_from_db_uri(db_uri)

    from_file = read_indexes_from_file(args.from_file) if args.from_file else None
    reindexed_from_file = []
    if from_file:
        log(f"From file indexes: {from_file}", to_file_only=not args.verbose)

    start_time = time.time()
    if not args.list:
        log(f"Started at {humanized_time(start_time)}")

    try:
        if args.pg_text_indexes or args.from_file:
            # Indexes from pg_catalog cannot be reindexed in parallel because that operations get deadlocked.
            # They also cannot be reindex concurrently. Thus, concurrently=False and workers=1.
            exit_code, _used_from_file = reindex(
                db_uri,
                concurrently=False,
                description=f"PG text indexes in {db_name} DB",
                filters=PG.Filters(
                    pg_catalog_indexes=True, index_name_pattern=args.index_name_pattern, from_file=from_file
                ),
                list_only=args.list,
                verbose=args.verbose,
                workers=1,
            )
            reindexed_from_file.extend(_used_from_file)
            if exit_code != ExitCodes.SUCCESS:
                log_error("Reindexing PG text indexes failed, interrupting")
                return exit_code

        if args.starfish_text_indexes or args.from_file:
            exit_code, _used_from_file = reindex(
                db_uri,
                concurrently=args.concurrently,
                description=f"starfish text indexes in {db_name} DB",
                filters=PG.Filters(
                    pg_catalog_indexes=False, index_name_pattern=args.index_name_pattern, from_file=from_file
                ),
                list_only=args.list,
                verbose=args.verbose,
                workers=args.workers,
            )
            reindexed_from_file.extend(_used_from_file)
            if exit_code != ExitCodes.SUCCESS:
                log_error("Reindexing starfish text indexes failed")
                return exit_code

        if args.from_file:
            return error_if_unused_indexes_from_file(args.from_file, reindexed_from_file)

        return ExitCodes.SUCCESS
    finally:
        end_time = time.time()
        if not args.list:
            log(f"Finished at {humanized_time(end_time)}, took: {huminized_interval(int(end_time - start_time))}")


if __name__ == "__main__":
    sys.exit(main(sys.argv[1:]))
