#!/bin/bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

DB_NAME="${DB_NAME:-starfish}"
NUM_INDEXES_IN_PARALLEL="${NUM_INDEXES_IN_PARALLEL:-3}"
SFHOME="${SFHOME:-/opt/starfish}"
# shellcheck source=scripts/installation/_pg_common.sh
source "${SFHOME}/data/installation/_pg_common.sh"
PSQL_PATH="$(find_psql_command "${STARFISH_PG_VERSION}")"
TABLESPACE_PREFIX="sf_repack_"
readonly DB_NAME NUM_INDEXES_IN_PARALLEL PSQL_PATH TABLESPACE_PREFIX

log() {
    if [ $# -gt 0 ]; then
        >&2 echo "$(date +"%Y-%m-%d %H:%M:%S"): $*"
    else
        local line
        while read -r line; do
            >&2 echo "$(date +"%Y-%m-%d %H:%M:%S"): ${line}"
        done
    fi
}

on_exit() {
    local exit_code=$?
    if [ "${exit_code}" -ne 0 ]; then
        log "FAILURE"
    fi
    exit "${exit_code}"
}

usage() {
    >&2 cat <<EOF
Usage: $0 <table> <temporary dir> [pg_repack args...]
or:
$0 --tables <table1> <table2> ... --tmp-dir <temporary dir> [pg_repack args...]

$0 eliminates PostgreSQL table bloat by copying live data from a table and
its indexes to a temporary directory, and then moving the now-compacted data
back to the original location.
This script has to be run as postgres user and requires that pg_repack be installed
to run.

This script does an 'offline repack', meaning that all Starfish activity has
to be completely stopped. The easiest way to do this is to stop any running
scans and jobs, then stop systemd and sf-agent services.
An offshoot of this is the 'read only' version, in which only queries are enabled.
To do this, stop sf-scans, sf-cron, sf-pgagent and sf-dispatcher within systemd.
Also stop all running scans and jobs, and pgagent jobs:

/opt/starfish/bin/configure_redash reports killall

Queries are able to be run as needed.

An online version of this is possible. Check with Starfish support for assistance.

The temporary directory has to be an absolute path and needs to be writable for
the postgres user.

This script repacks one table at a time. To identify tables that have a large
number of dead tuples, use sf check-config.

For the partitioned tables, if the passed table name refers to the given
partition, e.g. sf.file_current_part_1, only this partition will be repacked.
If the passed table name does not include partition, e.g. sf.file_current,
the script will repack all partitions.

Single Table Example:

$0 sf.job_result_current /path/to/temporary/space

Multiple Tables Example:

$0 --tables sf.file_current sf.file_history sf.dir_current sf.dir_history --tmp-dir /path/to/temp/dir/

and repacking selected partitions can be achieved in the same way:

$0 --tables sf.file_current_part_2 sf.file_current_part_5 --tmp-dir /path/to/temp/dir/

Please note that the order of arguments cannot be changed.
--tables + the list has to be before --tmp-dir + path.
EOF
}

trap on_exit EXIT

if [ $# -eq 0 ] || [ "$1" == '--help' ]; then
    usage
    exit 0
fi

if [ $# -lt 2 ]; then
    usage
    exit 2
fi

if [ "$(whoami)" != "postgres" ]; then
    echo >&2 "This script should be run as postgres user."
    exit 1
fi

tables=()

if [ "$1" == '--tables' ]; then
    shift
    while [ $# -gt 0 ]; do
        case $1 in
        --tmp-dir)
            shift
            readonly TEMP_DIR="${1}"  # absolute path
            shift
            break
            ;;
        *)
            tables+=("$1")
            ;;
        esac
        shift
    done
else
    tables+=("${1}")  # e.g. sf.job_result_current
    readonly TEMP_DIR="${2}"  # absolute path
    shift 2
fi

run_psql() {
    "${PSQL_PATH}" --dbname "${DB_NAME}" "$@"
}

is_table_partitioned() {
    local schema="$1"
    local relation="$2"
    local table_relkind
    table_relkind="$(run_psql --tuples-only --quiet --no-align <<_EOF
SELECT relkind
  FROM pg_class
 WHERE relname = '${relation}'
   AND relnamespace::regnamespace::text = '${schema}';
_EOF
)"

    [[ "${table_relkind}" == 'p' ]]
}

get_schema_from_table() {
    local schema_and_table="$1"
    echo "${schema_and_table}" | cut -d'.' -f1
}

get_relation_from_table() {
    local table="$1"
    echo "${table}" | cut -d'.' -f2-
}

get_postgresql_version(){
    local db_path
    db_path="$(run_psql --tuples-only --quiet --no-align --command "SHOW data_directory")"
    cat "${db_path}/PG_VERSION"
}

find_pg_repack(){
    if ! command -v pg_repack 2>&1; then  # ubuntu
        # centos:
        if [[ "${PG_VERSION:-""}" = "9.6" ]]; then
            command -v /usr/pgsql-9.6/bin/pg_repack 2>&1
        elif [[ "${PG_VERSION:-""}" = "13" ]]; then
            command -v /usr/pgsql-13/bin/pg_repack 2>&1
        fi
    fi
}

PG_VERSION="$(get_postgresql_version)"
readonly PG_VERSION

PG_REPACK="${PG_REPACK:-$(find_pg_repack)}"
readonly PG_REPACK

if [ -z "${PG_REPACK}" ]; then
    echo -e ""
        cat >&2 <<EOF
Your Starfish is configured to work with PostgreSQL version ${PG_VERSION:-""}
No pg_repack for that PostgreSQL version found on your system.
Please make sure pg_repack is installed and try again.
EOF
    exit 1
fi

alter_idle_in_transaction_session_timeout() {
    local new_value="$1"
    run_psql --command "ALTER SYSTEM SET idle_in_transaction_session_timeout='${new_value}'"
    log "Reloading PostgreSQL service"
    pg_reload_conf
    run_psql --quiet --command "ALTER SYSTEM RESET idle_in_transaction_session_timeout"
}

pg_reload_conf() {
    run_psql --command  "SELECT pg_reload_conf()" > /dev/null
}

drop_tablespaces() {
    for name in $(run_psql --no-align --tuples-only --command "SELECT spcname FROM pg_tablespace WHERE spcname LIKE '${TABLESPACE_PREFIX}%'"); do
        log "Dropping tablespace ${name}"
        run_psql --command "DROP TABLESPACE ${name}" || {
            log <<EOF
Dropping tablespace ${name} failed.
If it's not empty, repacking failed and the data of tables
${tables[@]} or their indexes are still in a temporary directory.
Do not remove the data manually as it would break Starfish.
Contact Starfish Support Team for assistance.
EOF
            return 1
        }
    done
}

disable_autovacuum() {
    # This also terminates running autovacuum on the table, because ALTER TABLE requires ACCESS EXCLUSIVE lock,
    # while autovacuum takes SHARE UPDATE EXCLUSIVE (https://www.postgresql.org/docs/9.6/static/explicit-locking.html).
    # Autovacuum is kind enough to stop if another process requires lock >= SHARE UPDATE EXCLUSIVE.
    local table="$1"
    run_psql <<_EOF
ALTER TABLE ${table} SET (autovacuum_enabled = false, toast.autovacuum_enabled = false);
_EOF
}

enable_autovacuum() {
    local table="$1"
    run_psql <<_EOF
ALTER TABLE ${table} SET (autovacuum_enabled = true, toast.autovacuum_enabled = true);
_EOF
}

move_table_to_default_tablespace() {
    local table="$1"
    local relation schema
    relation="$(get_relation_from_table "${table}")"
    schema="$(get_schema_from_table "${table}")"
    run_psql <<_EOF
DO \$\$
DECLARE
    index_name RECORD;
BEGIN
    ALTER TABLE ${table} SET TABLESPACE pg_default;
    FOR index_name IN
        SELECT schemaname AS ns, indexname AS name
        FROM pg_indexes
        WHERE schemaname = '${schema}' AND tablename = '${relation}'
    LOOP
        EXECUTE 'ALTER INDEX ' || quote_ident(index_name.ns) || '.' || quote_ident(index_name.name) || ' SET TABLESPACE pg_default';
    END LOOP;
END;
\$\$
_EOF
}

get_tables_to_repack() {
    local relation schema table
    for table in "${tables[@]}"; do
        relation="$(get_relation_from_table "${table}")"
        schema="$(get_schema_from_table "${table}")"
        if is_table_partitioned "${schema}" "${relation}"; then
            run_psql --tuples-only --quiet --no-align <<_EOF
SELECT
    nmsp_child.nspname || '.' || child.relname AS partition_name
FROM pg_inherits
    JOIN pg_class parent            ON pg_inherits.inhparent = parent.oid
    JOIN pg_class child             ON pg_inherits.inhrelid   = child.oid
    JOIN pg_namespace nmsp_parent   ON nmsp_parent.oid  = parent.relnamespace
    JOIN pg_namespace nmsp_child    ON nmsp_child.oid   = child.relnamespace
WHERE parent.relname='${relation}' and nmsp_parent.nspname = '${schema}'
_EOF
        else
            echo "${table}"
        fi
    done
}

main() {
    run_psql --command "DROP EXTENSION IF EXISTS pg_repack"
    run_psql --command "CREATE EXTENSION pg_repack"

    log "Altering idle_in_transaction_session_timeout to 96h"
    alter_idle_in_transaction_session_timeout "96h"
    log "Dropping tablespaces created by previous instances of this script (if any)"
    drop_tablespaces

    for table in $(get_tables_to_repack); do
        log "Repacking table ${table}"

        local tablespace_dir tablespace_name relation
        relation="$(get_relation_from_table "${table}")"
        tablespace_dir="$(mktemp -d "${TEMP_DIR}/${TABLESPACE_PREFIX}${relation}_$(date +"%Y%m%d_%H%M%S")_XXXX")"

        log "Creating temporary tablespace in ${tablespace_dir}"
        tablespace_name="$(basename "${tablespace_dir}" | tr '[:upper:]' '[:lower:]')"
        run_psql --command "CREATE TABLESPACE ${tablespace_name} LOCATION '${tablespace_dir}'"

        log "Disabling autovacuum"
        disable_autovacuum "${table}"

        log "Rewriting table to temporary tablespace"
        "${PG_REPACK}" --dbname "${DB_NAME}" --elevel DEBUG --table "${table}" --tablespace "${tablespace_name}" --moveidx --jobs "${NUM_INDEXES_IN_PARALLEL}" "$@"

        log "Moving repacked table back to default tablespace"
        move_table_to_default_tablespace "${table}"

        log "Enabling autovacuum"
        enable_autovacuum "${table}"
    done

    log "Dropping temporary tablespace"
    drop_tablespaces
    log "Reloading PostgreSQL service to restore idle_in_transaction_session_timeout to default"
    pg_reload_conf
    log "Success"
}

main "$@"
