#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

# INSTALL_SCRIPT_VERSION is generated automatically when creating self-extracting archive
readonly INSTALL_SCRIPT_VERSION=

_SCRIPT_DIR=$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )
readonly _SCRIPT_DIR

LOG_FILE="$(basename "${BASH_SOURCE[0]}" '.sh').log"
export LOG_FILE

# shellcheck source=scripts/installation/_common_install.sh
source "${_SCRIPT_DIR}/_common_install.sh"
# shellcheck source=scripts/installation/_pg_configure.sh
source "${_SCRIPT_DIR}/_pg_configure.sh"

usage() {
    >&2 cat <<__EOF
Usage: $0 [OPTION]...
$0 adjust PostgreSQL config on a local machine to optimize for Starfish usage.
By default, the script works in an interactive mode.
If any optional parameter is not given, the user will be prompted.

Options:
    --sf-db-path            path where Starfish will store its data;
                            default: ${DEFAULT_SF_DB_PATH}
    --sf-db-storage-type    storage type on which Starfish database is placed
                            possible values: hdd, raid (hdd), ssd, nvme; default: ${DEFAULT_SF_DB_STORAGE_TYPE}
    --use-defaults          work in non-interactive mode and use default values if possible
    --version               print Starfish version
    --help                  show this help
__EOF
}

show_version() {
    echo "${INSTALL_SCRIPT_VERSION}"
}

parse_cmdline_args() {
    log "Script called as: $0 $*"

    CLI_SF_DB_PATH=""
    CLI_SF_DB_STORAGE_TYPE=""
    CLI_USE_DEFAULTS="no"
    while [[ $# -gt 0 ]]; do
        case $1 in
        --sf-db-path)
            shift
            CLI_SF_DB_PATH="${1:-}"
            ;;
        --sf-db-storage-type)
            shift
            CLI_SF_DB_STORAGE_TYPE="${1:-}"
            ;;
        --use-defaults)
            CLI_USE_DEFAULTS="yes"
            ;;
        --version)
            show_version
            exit 0
            ;;
        --help)
            usage
            exit 0
            ;;
        *)
            break
            ;;
        esac
        shift
    done
    if (( $# != 0 )); then
        usage
        exit 2
    fi
}

collect_params() {
    load_default_installation_params_for_existing_installation
    parse_cmdline_args "$@"

    SF_DB_PATH="$(prompt_for_param "Enter path where Starfish database stores its data" "${CLI_SF_DB_PATH}" "${DEFAULT_SF_DB_PATH}" "${CLI_USE_DEFAULTS}")"
    SF_DB_STORAGE_TYPE=$(prompt_for_db_storage_type "${CLI_SF_DB_STORAGE_TYPE}" "${DEFAULT_SF_DB_STORAGE_TYPE}" "${CLI_USE_DEFAULTS}")

    log <<_EOF
Reconfiguring PostgreSQL with the following parameters:
    sf-db-path: "${SF_DB_PATH}"
    sf-db-storage-type: "${SF_DB_STORAGE_TYPE}"
_EOF
    save_default_installation_params
}

main() {
    LOG_FILE="$(basename "${BASH_SOURCE[0]}" '.sh').log"
    export LOG_FILE
    local pg_version

    fail_if_not_root
    fail_if_unknown_distribution
    collect_params "$@"

    pg_version=$(postgresql_version "${SF_DB_PATH}")
    reconfigure_postgresql "${SF_DB_PATH}" "${SF_DB_STORAGE_TYPE}" "${pg_version}" "${STARFISH_PG_CLUSTER_NAME}"
    if [[ "${_PG_APPLY_CONFIG_CHANGES}" = "restart" ]]; then
        log "restarting PostgreSQL ${pg_version}/${STARFISH_PG_CLUSTER_NAME}"
        restart_postgresql_service "${pg_version}" "${STARFISH_PG_CLUSTER_NAME}" "${STARFISH_PG_PORT}"
    elif [[ "${_PG_APPLY_CONFIG_CHANGES}" = "reload" ]]; then
        log "reload PostgreSQL ${pg_version}/${STARFISH_PG_CLUSTER_NAME}"
        reload_postgresql_service "${pg_version}" "${STARFISH_PG_CLUSTER_NAME}" "${STARFISH_PG_PORT}"
    fi
}

sourced() {
    [[ "${BASH_SOURCE[0]}" != "${0}" ]]
}

if ! sourced; then
    main "$@"
fi
