#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail
########################################################
#
#  Starfish pre-install check script
#
########################################################

SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
# shellcheck source=scripts/installation/_utils.sh
source "${SCRIPT_DIR}/_utils.sh"

# This script checks for conflicts on known ports that we need in operations

#*******************************************************
# Change Log
# v1.0 Aug 9, 2018 - doug; pre-check fro STAR-5302
# v1.01 April 24, 2019 - d.gold: modified to make less chatty
VERSION=1.01
DEBUG=0

OK=1

centos_pkg_count() {
    local pattern="$1"

    rpm -qa 2>/dev/null | grep --count "${pattern}" || true
}

ubuntu_pkg_count() {
    local pattern="$1"

    apt list --installed 2>/dev/null | grep --count "${pattern}" || true
}

pkg_count() {
    run_func_for_distro centos_pkg_count ubuntu_pkg_count "$@"
}

echo -e "Starfish pre-installation check, version $VERSION\\n"
echo "Checking whether Starfish is already installed."

if pkg_installed starfish; then
        echo "Starfish is already installed. Exiting."
        exit 0
fi

echo -e "\\nChecking whether PostgreSQL is already installed."

pgcount="$(pkg_count ^postgresql)"

if [[ $pgcount -gt 0 ]]; then
        echo "PostgreSQL is already installed. This may conflict with bundled version. We strongly recommend uninstalling existing PostgreSQL before installation."
        echo "Checking whether PostgreSQL is enabled."
        syscnt=$(systemctl list-unit-files | grep --count 'postgresql.*enabled')
        if [[ ${syscnt} -gt 0 ]]; then
                echo -e "PostgreSQL service is enabled. This needs to be disabled prior to installation of Starfish."
        else
                echo PostgreSQL service is not running. You can install Starfish without removing PostgreSQL, but will need to set alternatives for PostgreSQL commands after installation.
        fi
        OK=0
fi

echo -e "\\nChecking on ports used by Starfish"

for port in 80 443 5000 6379 8080; do
        LSOF=$(lsof -P -i :"$port">/dev/null)
        PORTS=$?
        if [ $DEBUG -eq 1 ];then
           echo "LSOF output is $LSOF"
        fi
        if [[ $PORTS == 0 ]]; then
                echo "A service is running on port $port. This will conflict with Starfish web and redash interfaces."
                #echo -e "$stuff" | head -n 2
                #echo "-----"
                OK=0
        fi

done

for port in 5432 5435; do
        # echo "checking port $port"
        LSOF=$(lsof -P -i :"$port">/dev/null)
        DBPORTS=$?
        if [ $DEBUG -eq 1 ];then
           echo "LSOF output is $LSOF"
        fi
        if [[ $DBPORTS == 0 ]]; then
                echo -e "A service is running on port $port. This will conflict with Postgres"
                #echo -e "$stuff" | head -n 2
                #echo "-----"
                OK=0
        fi
done

#echo "checking starfish API ports. Any thing below here may end up conflicting with Starfish services API."
END=30030
for ((port=30000;port<=END;port++)); do
        #echo "checking port $port"
        LSOF=$(lsof -P -i:"$port")
        SFPORTS=$?
        if [ $DEBUG -eq 1 ];then
           echo "LSOF output is $LSOF"
        fi
        if [[ $SFPORTS == 0 ]]; then
                echo "A service is running on port $port. This will conflict with Starfish"
                #echo -e "$stuff" | head -n 2
                #echo "------"
                OK=0
        fi
        export LSOF
done

if [[ $OK == 0 ]]; then
        echo
        echo -e "\\nFound potential issues during precheck. Please address."
else
        echo -e "\\nPrecheck looks OK. No conflicts"
fi
