#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

# INSTALL_SCRIPT_VERSION is generated automatically when creating self-extracting archive
readonly INSTALL_SCRIPT_VERSION=
LOG_FILE="$(basename "${BASH_SOURCE[0]}" '.sh').log"
readonly LOG_FILE
readonly STARFISH_DB_THRESHOLD="${STARFISH_DB_THRESHOLD:-$((64 * 1024 * 1024 * 1024))}"

SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
readonly SCRIPT_DIR

# shellcheck source=scripts/installation/_common_install.sh
source "${SCRIPT_DIR}/_common_install.sh"


usage() {
    >&2 cat <<__EOF
Usage: $0 [OPTION]...
$0 installs Starfish and Redash with all required dependencies (including database) on the local machine.
By default, the script works in an interactive mode. If any optional parameter is not given, the user will be prompted.

Starfish options:
    --sf-repo-url REPO_URL                  Starfish repository URL provided by Starfish Corporation
    --sf-db-path DB_PATH                    local, non-nfs path where Starfish will store its data;
                                            Starfish database requires ~2GB of storage per 1M files;
                                            there should be at least ${STARFISH_DB_THRESHOLD} bytes free in the directory;
                                            default: ${DEFAULT_SF_DB_PATH}
    --sf-db-storage-type                    storage type on which Starfish database is placed;
                                            possible values: hdd, raid (hdd), ssd, nvme; default: ${DEFAULT_SF_DB_STORAGE_TYPE}
    --sf-disable-backup                     Do not setup backup for Starfish database
    --sf-backup-path                        Backup location for Starfish database.
                                            Starfish installer will install PostgreSQL database backup
                                            with the following backup policy:
                                            - full backup performed every Saturday
                                              (required disk space: ~15% of the original database)
                                            - incremental backup run every weekday
                                              (required disk space: depends on amount of daily changes on scanned filesystems.
                                              After initial volume scan, this generally will not exceed 10% of database size)
                                            - retention of backups set to 2 weeks: 2 full backups, 10 incrementals
                                            - PostgreSQL archivelog mode enabled. Point in time recoveries possible
                                             from any point in time within the 2 week backup period.
                                            - disk space required for backup: 100-200% of original database size depending on
                                            the number of new files written
                                            - configuration files: /etc/pgbackrest.conf, $(get_starfish_cron_file)
                                            Note: Backup path should be on a different filesystem, preferably on a different server.
                                            See the Starfish documentation for details on modifying backup policy.
    --sf-listen-address LISTEN_ADDRESS      Hostname or IP address which Starfish will listen on (default: ${DEFAULT_SF_LISTEN_ADDRESS})
                                            It is recommended to use a public address.
    --sf-disable-reports                    Disable sending all diagnostic reports.
                                            By default, the reports are sent in an email every day at 04:00 AM.
    --sf-disable-reports-to-starfish-team   Disable sending diagnostic reports to Starfish Corporation team.
                                            Disabling them is not recommended, because diagnostic reports help
                                            Starfish support team to quickly investigate any problems with this installation.
    --sf-disable-reports-attachments        Disable attaching diagnostic data for automatic monitoring.
                                            Disabling these is not recommended, as these make Starfish support team
                                            aware of the problems in short period of time.
    --sf-enable-obfuscated-reports-to-starfish-team    Obfuscate all the fields of the diagnostic reports
                                            that may contain confidential information.
    --sf-reports-extra-emails-errors-only EMAILS   Space-separated list of emails to which diagnostic error reports will be sent.
    --sf-reports-extra-emails EMAILS        Space-separated list of emails to which diagnostic error and warning reports will be sent.
    --sf-reports-from-email EMAIL           Custom FROM email for diagnostic reports.
    --sf-reports-ignored-errors-pattern PATTERN     Bash-style egrep pattern. If some errors in the diagnostic report are expected,
                                            they can be defined here and will not be counted as errors, therefore
                                            '(ERROR)' phrase will not be added to the email subject.
    --sf-disable-proxy-check                Forces installation to continue when proxy settings are found.
    --sf-offline-installation               Performs an offline installation. Requires specifying --sf-repo-dir.
    --sf-repo-dir REPO_DIR                  Directory containing a package repository from a Starfish offline installation bundle.

Redash options:
    --redash-email EMAIL                    Email address used to login to Redash; default: ${DEFAULT_REDASH_EMAIL}
    --redash-password PASSWORD              Password used to login to Redash
    --redash-only                           Install and configure Redash only. This option requires that Starfish is
                                            already installed.
Database options:
    --backup-only                           Setup starfish database backup without Starfish installation

Miscellaneous:
    --use-defaults                          work in non-interactive mode and use default values if possible
    --version                               print Starfish version
    --help                                  show this help
__EOF
}

show_version() {
    echo "${INSTALL_SCRIPT_VERSION}"
}

parse_cmdline_args() {
    local args=( "$@" )

    CLI_INSTALL_STARFISH="yes"
    CLI_SF_REPO_URL=""
    CLI_SF_LISTEN_ADDRESS=""
    CLI_SF_CONTINUE_ON_PROXY=""
    CLI_SF_DB_PATH=""
    CLI_SF_DB_STORAGE_TYPE=""
    CLI_SF_CONFIGURE_BACKUP=""
    CLI_SF_BACKUP_PATH=""
    CLI_SF_BACKUP_RETENTION=""
    CLI_SF_ENABLE_DIAG=""
    CLI_SF_SEND_DIAG_TO_SF=""
    CLI_SF_ATTACH_JSON_TO_EMAIL=""
    CLI_SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY=""
    CLI_SF_DIAG_EXTRA_EMAILS=""
    CLI_SF_DIAG_FROM_EMAIL=""
    CLI_SF_DIAG_IGNORED_ERRORS_PATTERN=""
    CLI_INSTALL_REDASH="yes"
    CLI_REDASH_EMAIL=""
    CLI_REDASH_PASSWORD=""
    CLI_REDASH_VERSION=10
    CLI_ADD_PORT_TO_CONFIG_SERVICE_URL=""
    CLI_PG_VERSION="13"
    CLI_USE_DEFAULTS="no"

    while [ $# -gt 0 ]; do
        case $1 in
        --log-file)
            shift
            LOG_FILE="$1"
            ;;
        --sf-repo-url|--sf-repo-dir)
            shift
            CLI_SF_REPO_URL="${1:-}"
            ;;
        # --sf-listen-ip is deprecated
        --sf-listen-address|--sf-listen-ip)
            shift
            CLI_SF_LISTEN_ADDRESS="${1:-}"
            ;;
        --sf-db-path)
            shift
            CLI_SF_DB_PATH="${1:-}"
            ;;
        --sf-db-storage-type)
            shift
            CLI_SF_DB_STORAGE_TYPE="${1:-}"
            ;;
        --sf-disable-backup)
            CLI_SF_CONFIGURE_BACKUP="no"
            ;;
        --sf-enable-backup)
            CLI_SF_CONFIGURE_BACKUP="yes"
            ;;
        --sf-backup-path)
            CLI_SF_CONFIGURE_BACKUP="yes"
            shift
            CLI_SF_BACKUP_PATH="${1:-}"
            ;;
        --sf-backup-retention)
            CLI_SF_CONFIGURE_BACKUP="yes"
            shift
            CLI_SF_BACKUP_RETENTION="${1:-}"
            if ! is_positive_integer "${CLI_SF_BACKUP_RETENTION}"; then
                log "--sf-backup-retention requires positive integer, ${CLI_SF_BACKUP_RETENTION} was given."
                usage
                exit 2
            fi
            ;;
        --sf-disable-reports)
            CLI_SF_ENABLE_DIAG="no"
            ;;
        --sf-disable-reports-to-starfish-team)
            CLI_SF_SEND_DIAG_TO_SF="no"
            ;;
        --sf-disable-reports-attachments)
            CLI_SF_ATTACH_JSON_TO_EMAIL="no"
            ;;
        --sf-enable-obfuscated-reports-to-starfish-team)
            CLI_SF_SEND_DIAG_TO_SF="obfuscated"
            ;;
        --sf-disable-proxy-check)
            CLI_SF_CONTINUE_ON_PROXY="yes"
            ;;
        --sf-reports-extra-emails)
            shift
            CLI_SF_DIAG_EXTRA_EMAILS="${1:-}"
            ;;
        --sf-reports-extra-emails-errors-only)
            shift
            CLI_SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY="${1:-}"
            ;;
        --sf-reports-from-email)
            shift
            CLI_SF_DIAG_FROM_EMAIL="${1:-}"
            ;;
        --sf-reports-ignored-errors-pattern)
            shift
            CLI_SF_DIAG_IGNORED_ERRORS_PATTERN="${1:-}"
            ;;
        --sf-offline-installation)
            export SF_OFFLINE_INSTALLATION=yes
            ;;
        --redash-only)
            CLI_INSTALL_REDASH="yes"
            CLI_INSTALL_STARFISH="no"
            CLI_SF_CONFIGURE_BACKUP="no"
            ;;
        --backup-only)
            CLI_INSTALL_REDASH="no"
            CLI_INSTALL_STARFISH="no"
            CLI_SF_CONFIGURE_BACKUP="yes"
            ;;
        --no-redash)
            CLI_INSTALL_REDASH="no"
            ;;
        --redash-version)
            shift
            CLI_REDASH_VERSION="${1:-}"
            ;;
        --pg-version)
            shift
            CLI_PG_VERSION="${1:-}"
            ;;
        --redash-email)
            shift
            CLI_REDASH_EMAIL="${1:-}"
            ;;
        --redash-password)
            shift
            CLI_REDASH_PASSWORD="${1:-}"
            ;;
        --add-port-to-config-service-url)
            CLI_ADD_PORT_TO_CONFIG_SERVICE_URL="yes"
            ;;
        --use-defaults)
            CLI_USE_DEFAULTS="yes"
            ;;
        --version)
            show_version
            exit 0
            ;;
        --help)
            usage
            exit 0
            ;;
        *)
            break
            ;;
        esac
        shift
    done
    export LOG_FILE
    log "Script called with arguments: $0 " ${args[@]+"${args[@]}"}

    if (( $# != 0 )); then
        usage
        exit 2
    fi
}

prompt_for_repo_url() {
    if ! empty_string "${CLI_SF_REPO_URL}"; then
        echo "${CLI_SF_REPO_URL}"
    else
        if empty_string "${DEFAULT_SF_REPO_URL}"; then
            # default value was not loaded from file
            if [ "${CLI_USE_DEFAULTS}" == "yes" ]; then
                log "Can't use default value for --sf-repo-url."
                exit 2
            fi
            if ! is_offline_installation; then
                input_not_empty_value "Enter Starfish repository URL provided by Starfish Corporation: "
            else
                input_not_empty_value "Enter Starfish offline repository directory path: "
            fi
        else
            if [ "${CLI_USE_DEFAULTS}" == "yes" ]; then
                echo "${DEFAULT_SF_REPO_URL}"
            else
                if ! is_offline_installation; then
                    input_with_default_value "Enter Starfish repository URL provided by Starfish Corporation: " "${DEFAULT_SF_REPO_URL}"
                else
                    input_with_default_value "Enter Starfish offline repository directory path: " "${DEFAULT_SF_REPO_URL}"
                fi
            fi
        fi
    fi
}

prompt_for_password() {
    local message="$1"
    local cmdline_value="$2"
    local default_value="$3"
    local use_defaults="$4"
    local min_pass_len="${6:-6}"
    local value password

    if ! empty_string "${cmdline_value}"; then
        echo "${cmdline_value}"
        return
    fi
    if [ "${use_defaults}" == "yes" ]; then
        echo "${default_value}"
        return
    fi

    while true; do
        read -s -r -p "${message} [min len=${min_pass_len}, default=${default_value}]: " value
        password="${value:-${default_value}}"
        if [ ${#password} -ge "${min_pass_len}" ]; then
            read -s -r -p "Confirm redash password [min len=${min_pass_len}, default=${default_value}]: " value_2
            password_2="${value_2:-${default_value}}"
            if [[ "${password}" == "${password_2}" ]]; then
                break
            fi
        fi
    done
    echo "${password}"
}

ask_for_install_continue_due_to_proxy() {
    if is_proxy_set; then
        SF_CONTINUE_ON_PROXY="$(prompt_for_yes_or_no "Proxy settings detected
Current proxy settings are:
$(env | grep --extended-regexp --ignore-case "(all|http[s]?|no)_proxy=")

Please make sure that these settings are correct. no_proxy should be set for 127.0.0.1 and ${SF_LISTEN_ADDRESS}
Would you like to continue with the current settings?" "${CLI_SF_CONTINUE_ON_PROXY}" "${DEFAULT_SF_CONTINUE_ON_PROXY}" "${CLI_USE_DEFAULTS}")"

        if [[ "${SF_CONTINUE_ON_PROXY}" == "no" ]]; then
            log "Installation interrupted by user due to potentially problematic proxy settings."
            exit 1
        fi
    fi
}

ask_for_diagnostic_emails_config() {

    SF_SEND_DIAG_TO_SF="${DEFAULT_SF_SEND_DIAG_TO_SF}"
    SF_ATTACH_JSON_TO_EMAIL="${DEFAULT_SF_ATTACH_JSON_TO_EMAIL}"
    SF_OBFUSCATE_CONFIDENTIAL_INFO="${DEFAULT_SF_OBFUSCATE_CONFIDENTIAL_INFO}"
    SF_DIAG_EXTRA_EMAILS="${DEFAULT_SF_DIAG_EXTRA_EMAILS}"
    SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY="${DEFAULT_SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY}"
    SF_DIAG_FROM_EMAIL="${DEFAULT_SF_DIAG_FROM_EMAIL}"
    SF_DIAG_IGNORED_ERRORS_PATTERN="${DEFAULT_SF_DIAG_IGNORED_ERRORS_PATTERN}"

    cat <<_EOF

You can configure Starfish to send nightly diagnostic/status reports. Emails may be sent to Starfish support or any
custom email to help monitor your installation.
A few questions will be asked to configure those emails and their contents.
All the following options can be modified later by editing $(get_sysconfig_file).

_EOF
    SF_ENABLE_DIAG="$(choose_default_value_if_no_cli_value "${CLI_SF_ENABLE_DIAG}" "${DEFAULT_SF_ENABLE_DIAG}")"
    if [[ $SF_ENABLE_DIAG = "yes" ]]; then
        SF_SEND_DIAG_TO_SF="$(prompt_for_yes_or_no "Would you like nightly diagnostic/status reports to be sent to Starfish support?" "${CLI_SF_SEND_DIAG_TO_SF}" "${DEFAULT_SF_SEND_DIAG_TO_SF}" "${CLI_USE_DEFAULTS}" "obfuscated")"
        SF_OBFUSCATE_CONFIDENTIAL_INFO="no"
        if [[ $SF_SEND_DIAG_TO_SF = "obfuscated" ]]; then
            SF_SEND_DIAG_TO_SF="yes"
            SF_OBFUSCATE_CONFIDENTIAL_INFO="yes"
        fi
        SF_ATTACH_JSON_TO_EMAIL="$(choose_default_value_if_no_cli_value "${CLI_SF_ATTACH_JSON_TO_EMAIL}" "${DEFAULT_SF_ATTACH_JSON_TO_EMAIL}")"
        SF_DIAG_IGNORED_ERRORS_PATTERN="$(choose_default_value_if_no_cli_value "${CLI_SF_DIAG_IGNORED_ERRORS_PATTERN}" "${DEFAULT_SF_DIAG_IGNORED_ERRORS_PATTERN}")"
        SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY="$(prompt_for_param "Would you like nightly diagnostic/status reports (errors only) to be sent to other (internal) recipients? If yes, list the emails here, space separated. Otherwise press enter " "${CLI_SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY}" "${DEFAULT_SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY}" "${CLI_USE_DEFAULTS}")"
        echo ""
        SF_DIAG_EXTRA_EMAILS="$(prompt_for_param "Would you like nightly diagnostic/status reports (errors and warnings) to be sent to other (internal) recipients? If yes, list the emails here, space separated. Otherwise press enter " "${CLI_SF_DIAG_EXTRA_EMAILS}" "${DEFAULT_SF_DIAG_EXTRA_EMAILS}" "${CLI_USE_DEFAULTS}")"
        echo ""
        SF_DIAG_FROM_EMAIL="$(prompt_for_param "Would you like to set the from address? If so, enter it here, otherwise press enter to use the system default" "${CLI_SF_DIAG_FROM_EMAIL}" "${DEFAULT_SF_DIAG_FROM_EMAIL}" "${CLI_USE_DEFAULTS}")"
        if [[ "${SF_SEND_DIAG_TO_SF}" != "yes" ]] && empty_string "${SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY}" && empty_string "${SF_DIAG_EXTRA_EMAILS}"; then
            SF_ENABLE_DIAG="no"
        fi
    fi
}

collect_installation_params() {
    local incrementals_count
    load_default_installation_params
    parse_cmdline_args "$@"
    # hack to install PG to /opt/starfish/pg/9.6 if --pg-version 9.6 is used;
    # can be removed when fully migrated to PG 13: load_default_installation_params() can't be called after parse_cmdline_args(),
    # because usage() refers to DEFAULT_* variables
    DEFAULT_SF_DB_PATH="${DEFAULT_SF_DB_PATH/13/${CLI_PG_VERSION}}"

    SF_REPO_URL="$(prompt_for_repo_url)"
    SF_LISTEN_ADDRESS="$(prompt_for_param "Enter host name or IP address (public is recommended) for Starfish to listen on" "${CLI_SF_LISTEN_ADDRESS}" "${DEFAULT_SF_LISTEN_ADDRESS}" "${CLI_USE_DEFAULTS}")"
    if ! is_address_resolvable "${SF_LISTEN_ADDRESS}"; then
        log "Could not resolve specified Starfish listen address: ${SF_LISTEN_ADDRESS}. Invalid address or unknown hostname."
        log "Starfish installer unable to continue, exiting. Please update and verify name resolution and then rerun installer."
        exit 1
    fi
    ask_for_install_continue_due_to_proxy
    SF_DB_PATH="$(prompt_for_param "Starfish database requires ~2GB of storage per 1M files. Enter a local, non-nfs path where database will store its data" "${CLI_SF_DB_PATH}" "${DEFAULT_SF_DB_PATH}" "${CLI_USE_DEFAULTS}")"
    SF_DB_STORAGE_TYPE=$(prompt_for_db_storage_type "${CLI_SF_DB_STORAGE_TYPE}" "${DEFAULT_SF_DB_STORAGE_TYPE}" "${CLI_USE_DEFAULTS}")
    SF_CONFIGURE_BACKUP="$(prompt_for_yes_or_no "Would you like to set up a backup for Starfish database" "${CLI_SF_CONFIGURE_BACKUP}" "${DEFAULT_SF_CONFIGURE_BACKUP}" "${CLI_USE_DEFAULTS}")"
    SF_BACKUP_PATH=""
    SF_BACKUP_RETENTION="2"
    if [ "${SF_CONFIGURE_BACKUP}" = "yes" ]; then
        SF_BACKUP_PATH="$(prompt_for_param "Enter a backup path for Starfish database" "${CLI_SF_BACKUP_PATH}" "${DEFAULT_SF_BACKUP_PATH}" "${CLI_USE_DEFAULTS}")"
        SF_BACKUP_RETENTION="$(prompt_for_positive_integer "Enter a backup retention count for Starfish database" "${CLI_SF_BACKUP_RETENTION}" "${DEFAULT_SF_BACKUP_RETENTION}" "${CLI_USE_DEFAULTS}")"
        incrementals_count=$(( SF_BACKUP_RETENTION * 5 ))
        cat <<_EOF

Starfish installer will install PostgreSQL database backup with the following backup policy:
- full backup performed every Saturday (required disk space: ~15% of the original database).
- incremental backup run every weekday (required disk space: depends on amount of daily changes
  on scanned filesystems. After initial volume scan, this generally will not exceed 10% of database
  size)
- retention of backups set to ${SF_BACKUP_RETENTION} weeks: ${SF_BACKUP_RETENTION} full backups, ${incrementals_count} incrementals
- PostgreSQL archivelog mode enabled. Point in time recoveries possible from any point in time
  within the 2 week backup period.
- disk space required for backup: 100-200% of original database size depending on the number of new files written
- configuration files: /etc/pgbackrest.conf, $(get_starfish_cron_file)
- backup storage location: ${SF_BACKUP_PATH}

Note: Backup path should be on a different filesystem, preferably on a different server.
See the Starfish documentation for details on modifying backup policy.

_EOF
    fi

    ask_for_diagnostic_emails_config
    REDASH_EMAIL=""
    REDASH_PASSWORD=""
    # get proper redash credentials only when needed
    if [[ "${CLI_INSTALL_REDASH}" = "yes" ]]; then
        cat <<_EOF
Starfish installer installs Redash to provide data visualizations.
_EOF
        REDASH_EMAIL="$(prompt_for_param "Enter username (Redash \"email address\") used to log in to Redash" "${CLI_REDASH_EMAIL}" "${DEFAULT_REDASH_EMAIL}" "${CLI_USE_DEFAULTS}")"
        REDASH_PASSWORD="$(prompt_for_password "Enter password used to log in to Redash" "${CLI_REDASH_PASSWORD}" "${REDASH_DEFAULT_PASSWORD}" "${CLI_USE_DEFAULTS}")"
    fi

    log <<_EOF
Installing Starfish with the following parameters:
    sf-repo-url: "${SF_REPO_URL}"
    sf-listen-address: "${SF_LISTEN_ADDRESS}"
    sf-disable-proxy-check: "${SF_CONTINUE_ON_PROXY:-${CLI_SF_CONTINUE_ON_PROXY}}"
    sf-db-path: "${SF_DB_PATH}"
    sf-db-storage-type: "${SF_DB_STORAGE_TYPE}"
    sf-enable-backup: "${SF_CONFIGURE_BACKUP}"
    sf-backup-path: "${SF_BACKUP_PATH}"
    sf-backup-retention: "${SF_BACKUP_RETENTION}"
    sf-enable-diagnostic-reports: "${SF_ENABLE_DIAG}"
    sf-diagnostic-reports-to-starfish-team: "${SF_SEND_DIAG_TO_SF}"
    sf-obfuscated-reports-to-starfish-team: "${SF_OBFUSCATE_CONFIDENTIAL_INFO}"
    sf-diagnostic-reports-attachments: "${SF_ATTACH_JSON_TO_EMAIL}"
    sf-diagnostic-reports-extra-emails: "${SF_DIAG_EXTRA_EMAILS}"
    sf-diagnostic-reports-extra-emails-errors-only: "${SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY}"
    sf-diagnostic-reports-from-email: "${SF_DIAG_FROM_EMAIL}"
    sf-diagnostic-reports-ignored-errors-pattern: "${SF_DIAG_IGNORED_ERRORS_PATTERN}"
    redash-email: "${REDASH_EMAIL}"
    redash-password: "*****"
_EOF
    save_default_installation_params
}

create_pg_uri() {
    local listen_address="$1"
    local pass="$2"
    local port="5432"
    local user="starfish"
    local database="starfish"

    local user_password

    user_password="$(encode_user_password "${user}"):$(encode_user_password "${pass}")"
    echo "postgresql://${user_password}@$(host_part_from_hostname "${listen_address}"):${port}/${database}"
}

create_opt_starfish_dir() {
    mkdir --parents "${SFHOME}"
    mkdir --parents "${STARFISH_PG_DIR}"
}

install_postgresql() {
    local listen_address="$1"
    local password="$2"
    local db_path="$3"
    local db_storage_type="$4"
    local backup_path="$5"
    local pg_version="$6"
    local retention="$7"

    ALLOW_FOR_INTERNAL_SCRIPT=1 DB_THRESHOLD="${STARFISH_DB_THRESHOLD}" "${SCRIPT_DIR}/_postgresql.sh" \
        --log-file "${LOG_FILE}" \
        --listen-address "${listen_address}" \
        --password "${password}" \
        --db-path "${db_path}" \
        --db-storage-type "${db_storage_type}" \
        --backup-path "${backup_path}" \
        --pg-version "${pg_version}" \
        --retention "${retention}"
}

install_postgresql_backup_only() {
    local db_path="$1"
    local backup_path="$2"
    local retention="$3"

    ALLOW_FOR_INTERNAL_SCRIPT=1 DB_THRESHOLD="${STARFISH_DB_THRESHOLD}" "${SCRIPT_DIR}/_postgresql.sh" \
        --backup-only \
        --db-path "${db_path}" \
        --backup-path "${backup_path}" \
        --retention "${retention}"
}

install_starfish() {
    local repo_url="$1"
    local listen_address="$2"
    local pg_uri="$3"
    shift 3

    ALLOW_FOR_INTERNAL_SCRIPT=1 "${SCRIPT_DIR}/_starfish.sh" \
        --log-file "${LOG_FILE}" \
        --repo-url "${repo_url}" \
        --listen-address "${listen_address}" \
        --pg-uri "${pg_uri}" \
        "$@"
}

install_redash() {
    local email="$1"
    local password="$2"
    local sf_listen_address="$3"
    local redash_version="$4"

    ALLOW_FOR_INTERNAL_SCRIPT=1 "${SCRIPT_DIR}/_redash.sh" \
        --log-file "${LOG_FILE}" \
        --email "${email}" \
        --password "${password}" \
        --redash-host "${sf_listen_address}" \
        --redash-version "${redash_version}"
}

enable_pg_backups() {
    set_sysconfig_param PG_BACKUP_CONFIGURED true

    log "If you enabled Starfish database backups, verify they are enabled in $(get_sysconfig_file)"
}

main() {
    local pg_pass pg_uri
    local disable_diag_reports="" disable_diag_reports_to_sf="" disable_json_diag_reports_to_sf="" no_redash=""
    local add_port_to_config_service_url=""

    log "Installer version: $(show_version)"

    pre_verifications_starfish
    collect_installation_params "$@"

    set_convenient_umask
    if [ "${CLI_INSTALL_STARFISH}" = "yes" ]; then
        fail_if_not_enough_space_for_db "${SF_DB_PATH}" "${STARFISH_DB_THRESHOLD}"
        if [ "${SF_CONFIGURE_BACKUP}" = "yes" ]; then
            fail_if_not_enough_space_for_db "${SF_BACKUP_PATH}" "${STARFISH_DB_THRESHOLD}"
        fi

        install_prerequisites
        if is_offline_installation; then
            setup_local_repo "${SF_REPO_URL}" "$(starfish_offline_repo_name)"
        fi

        create_opt_starfish_dir  # if user decides to keep database inside /opt/starfish
        pg_pass="$(generate_secret_key 24)"
        install_postgresql "${SF_LISTEN_ADDRESS}" "${pg_pass}" "${SF_DB_PATH}" "${SF_DB_STORAGE_TYPE}" "${SF_BACKUP_PATH}" "${CLI_PG_VERSION}" "${SF_BACKUP_RETENTION}"

        pg_uri="$(create_pg_uri "${SF_LISTEN_ADDRESS}" "${pg_pass}")"
        if [[ "${SF_ENABLE_DIAG}" != "yes" ]]; then
            disable_diag_reports="yes"
        fi
        if [[ "${SF_SEND_DIAG_TO_SF}" != "yes" ]]; then
            disable_diag_reports_to_sf="yes"
        fi
        if [[ "${SF_ATTACH_JSON_TO_EMAIL}" != "yes" ]]; then
            disable_json_diag_reports_to_sf="yes"
        fi
        if [[ "${SF_OBFUSCATE_CONFIDENTIAL_INFO}" = "yes" ]]; then
            enable_confidential_obfuscation="yes"
        fi
        if [[ "${CLI_INSTALL_REDASH}" = "no" ]]; then
            no_redash="yes"
        fi
        if [[ "${CLI_ADD_PORT_TO_CONFIG_SERVICE_URL}" = "yes" ]]; then
            add_port_to_config_service_url="yes"
        fi
        install_starfish "${SF_REPO_URL}" "${SF_LISTEN_ADDRESS}" "${pg_uri}"\
            ${no_redash:+"--no-redash"} \
            ${add_port_to_config_service_url:+"--add-port-to-config-service-url"} \
            ${disable_diag_reports:+"--disable-diag-reports"} \
            ${disable_diag_reports_to_sf:+"--disable-diag-reports-to-sf"} \
            ${disable_json_diag_reports_to_sf:+"--disable-json-diag-reports-to-sf"} \
            ${enable_confidential_obfuscation:+"--enable-confidential-obfuscation"} \
            --diag-reports-extra-emails \""${SF_DIAG_EXTRA_EMAILS}"\" \
            --diag-reports-extra-emails-errors-only \""${SF_DIAG_EXTRA_EMAILS_ERRORS_ONLY}"\" \
            --diag-reports-from-email "${SF_DIAG_FROM_EMAIL}" \
            --diag-reports-ignored-errors-pattern \""${SF_DIAG_IGNORED_ERRORS_PATTERN}"\"

        if [ "${SF_CONFIGURE_BACKUP}" = "yes" ]; then
            enable_pg_backups
        fi
    fi

    if [[ "${CLI_INSTALL_STARFISH}" = "no" && "${SF_CONFIGURE_BACKUP}" = "yes" ]]; then
        if empty_string "${SF_BACKUP_PATH}"; then
            log "Database backup path not specified."
            exit 1
        fi
        fail_if_not_enough_space_for_db "${SF_BACKUP_PATH}" "${STARFISH_DB_THRESHOLD}"
        install_postgresql_backup_only "${SF_DB_PATH}" "${SF_BACKUP_PATH}" "${SF_BACKUP_RETENTION}"
        enable_pg_backups
    fi

    if [ "${CLI_INSTALL_REDASH}" != "no" ]; then
        install_redash "${REDASH_EMAIL}" "${REDASH_PASSWORD}" "${SF_LISTEN_ADDRESS}" "${CLI_REDASH_VERSION}"
    fi

    log "###### Starfish Installation is Complete ######"
    if [ "${SF_ENABLE_DIAG}" = "no" ]; then
        log "Sending Starfish diagnostic reports is disabled. It's recommended to enable it by editing $(get_sysconfig_file)."
    else
        if ! command -v mail &> /dev/null; then
            log "\`mail\` command is missing. Starfish daily diagnostic report will not be sent."
        fi
        log <<_EOF

Please run a database baseline and send it to Starfish support. To do so:
su postgres
/usr/pgsql-13/bin/pgbench -i
/usr/pgsql-13/bin/pgbench -c 4 -j 4 -T 60 -P 2
(On Ubuntu, use /bin/pgbench or /usr/lib/postgresql/13/bin/pgbench)

It is also recommended to test sending Starfish diagnostic report by running: 'echo  "Test Message" | mail -s "Starfish Test Message" <USER_EMAIL_ADDRESS>'
After running the command, please check the inbox of <USER_EMAIL_ADDRESS> if the test email was received.

You can also run /opt/starfish/bin/report_stats.sh command so Starfish can verify that the email was received.
_EOF
    fi
}

sourced() {
    [[ "${BASH_SOURCE[0]}" != "${0}" ]]
}

if ! sourced; then
    main "$@"
fi
