#!/bin/bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

_log() {
    local message="$*"

    echo -e "${message}" >&2
    if [[ -n "${LOG_FILE:-""}" ]]; then
        # -p not --parents as it might be used on Mac OS in development
        mkdir -p "$(dirname "${LOG_FILE}")"
        echo -e "$(date +'%Y-%m-%d %H:%M:%S') ${message}" >> "${LOG_FILE}"
    fi
}

log() {
    if [ $# -gt 0 ]; then
        _log "$@"
    else
        local line
        while read -r line; do
            _log "${line}"
        done
    fi
}

failure() {
    echo -e "ERROR: $*" >&2
    exit 1
}

fail_if_not_root() {
    local root_uid=0
    if [ "${EUID}" != ${root_uid} ]; then
        failure "You must be root to run this script."
    fi
}

wait_for() {
    # there is a duplicate function in run-sf-lib.sh, used in tests only
    # it's now too hard to unify those two
    local condition="$1"
    local timeout="$2"

    for _ in $(seq "${timeout}"); do
        if ${condition}; then
            return 0
        fi
        sleep 1
    done
    log "Waiting for \"${condition}\" timed out after ${timeout} seconds"
    return 1
}

command_exists() {
    local cmd_name="$1"

    command -v "${cmd_name}" >/dev/null
}

path_exists() {
    local path="$1"

    test -e "${path}"
}

dir_exists() {
    local path="$1"

    test -d "${path}"
}

file_readable() {
    local path="$1"

    test -r "${path}"
}

file_exists() {
    local path="$1"

    test -f "${path}"
}

symlink_exists() {
    local path="$1"

    test -L "${path}"
}

is_owned_by_current_user() {
    local path="$1"

    test -O "${path}"
}

is_dir_empty() {
    local path="$1"

    test -z "$(ls --almost-all "${path}")"
}

empty_string() {
    local str="$1"

    test -z "${str}"
}

get_owner() {
    local path="$1"

    stat --format "%U" "${path}"
}

user_exists() {
    local username="$1"

    getent passwd "${username}" >/dev/null 2>&1
}

distro_get_id() {
    # used to be lsb_release --id --short
    # grep ... | cut should be equivalent
    # lsb_release is not installed in Docker images so this one works
    # it has the downside that `lsb_release` can in theory provide
    # backwards compat if /etc/lsb-release format changes

    # examples: LinuxMint, Ubuntu
    grep DISTRIB_ID /etc/lsb-release | cut -f 2 -d =
}

ubuntu_get_major_version() {
    # used to be lsb_release --codename --short
    # see distro_get_id

    # examples: focal, jammy
    grep DISTRIB_CODENAME /etc/lsb-release | cut -f 2 -d =
}

centos_get_major_version() {
    # works on RedHat as well
    sed --regexp-extended --quiet 's/.* ([0-9])\.[0-9].*/\1/p' /etc/redhat-release
}

centos_makecache() {
    if [[ "$(centos_get_major_version)" -lt 8 ]]; then
        run_with_sudo_if_not_root yum makecache fast
    else
        run_with_sudo_if_not_root yum makecache
    fi
}

get_distribution() {
    local distrib_id release_file

    if file_exists /etc/system-release; then
        release_file="$(basename "$(readlink -ev /etc/system-release)")"
        # cut off `-release` suffix
        echo "${release_file%-release}"
    elif file_exists /etc/lsb-release; then
        distrib_id=$(distro_get_id)
        if [[ "${distrib_id}" != "Ubuntu" && "${distrib_id}" != "LinuxMint" ]]; then
            failure "Distribution ${distrib_id} not supported"
        fi
        echo ubuntu
    elif [[ "$(uname)" == Darwin ]]; then
        echo macos
    else
        failure "Unable to detect Linux distribution"
    fi
}

fail_if_unknown_distribution() {
    get_distribution >/dev/null
}

raise_unsupported_ubuntu_distro() {
    failure "Unsupported Ubuntu distribution: $(ubuntu_get_major_version)"
}

is_development_environment() {
    [[ "${DEV_ENV:-""}" = "1" ]]
}

execute_on_production_environment() {
    local fail_func=$1
    if is_development_environment; then
        return
    fi
    ${fail_func}
}

map_to_ubuntu_distro() {
    local codename mapped_distro_name

    codename="$(ubuntu_get_major_version)"

    case "${codename}" in
        # after https://linuxmint.com/download_all.php
        xenial|sarah|serena|sonya|sylvia)  mapped_distro_name=xenial ;;
        trusty|qiana|rebecca|rafaela|rosa)  mapped_distro_name=trusty ;;
        bionic|tara|tessa|tina|tricia)  mapped_distro_name=bionic ;;
        focal|ulyana|ulyssa|una)  mapped_distro_name=focal ;;
        jammy|vanessa|vera|victoria)  mapped_distro_name=jammy ;;
        disco)
            execute_on_production_environment raise_unsupported_ubuntu_distro
            mapped_distro_name=disco ;;
        *)
            raise_unsupported_ubuntu_distro
    esac
    echo "${mapped_distro_name}"
}

run_func_for() {
    local distribution="$1"
    local centos_func="$2"
    local ubuntu_func="$3"
    shift 3

    case "${distribution}" in
    centos | redhat | oracle | rocky | alma*)
        ${centos_func} "$@"
        ;;
    ubuntu)
        ${ubuntu_func} "$@"
        ;;
    *)
        failure "Unsupported distribution: ${distribution}"
        ;;
    esac
}

run_func_for_distro() {
    local distro

    distro="$(get_distribution)"
    if is_development_environment && [[ "${distro}" == macos ]]; then
        >&2 echo "MacOS: not running $*"
    else
        run_func_for "${distro}" "$@"
    fi
}

run_with_sudo_if_not_root() {
    if [[ "$(whoami)" != "root" ]]; then
        sudo -E "$@"
    else
        "$@"
    fi
}

running_inside_container() {
   path_exists /.dockerenv || path_exists /run/.containerenv  # docker or podman
}

centos_pkg_installed() {
    local pkg="$1"

    # rpm --query always works even if yum DB is locked
    # so this function can be run safely from post-install RPM script
    rpm --query --info "${pkg}" > /dev/null 2>&1
}

ubuntu_pkg_installed() {
    local pkg="$1"
    local status=""

    # dpkg-query always works, even if apt DB is locked
    # so this function can be run safely from post-install DEB script

    # dpkg-query may exit with exit code 1 if the package is not known
    # but it may also return 0 and return "not installed" or "config-files"
    status=$(dpkg-query --show --showformat='${db:Status-Status}' "${pkg}" 2>/dev/null)
    test "${status}" == "installed"
}

pkg_installed() {
    run_func_for_distro centos_pkg_installed ubuntu_pkg_installed "$@"
}

centos_pkg_install() {
    local pkg="$1"

    if [[ "$#" -gt 1 ]]; then
        failure "centos_pkg_install can be used to install a single package only"
    fi

    if ! centos_pkg_installed "${pkg}"; then
        echo "Installing ${pkg}..."
        run_with_sudo_if_not_root yum install --assumeyes "${pkg}"
    fi
}

ubuntu_pkg_install() {
    local pkg="$1"

    if [[ "$#" -gt 1 ]]; then
        failure "ubuntu_pkg_install can be used to install a single package only"
    fi

    if ! ubuntu_pkg_installed "${pkg}"; then
        echo "Installing ${pkg}..."
        run_with_sudo_if_not_root apt-get install -y "${pkg}"
    fi
}

pkg_install() {
    local pkg

    for pkg in "$@"; do
        run_func_for_distro centos_pkg_install ubuntu_pkg_install "${pkg}"
    done
}

centos_pkg_update() {
    local pkg="$1"

    run_with_sudo_if_not_root yum update --assumeyes "${pkg}"
}

ubuntu_pkg_update() {
    local pkg="$1"

    run_with_sudo_if_not_root apt-get install -y "${pkg}"
}

pkg_update() {
    echo "Updating $*..."
    run_func_for_distro centos_pkg_update ubuntu_pkg_update "$@"
}

centos_pkg_available() {
    local pkg="$1"

    yum list available "${pkg}" > /dev/null 2>&1
}

ubuntu_pkg_available() {
    local pkg="$1"

    apt-get install --dry-run "${pkg}" > /dev/null 2>&1
}

pkg_available() {
    run_func_for_distro centos_pkg_available ubuntu_pkg_available "$@"
}

centos_get_sysconfig_file() {
    echo "/etc/sysconfig/starfish"
}

ubuntu_get_sysconfig_file() {
    echo "/etc/default/starfish"
}

get_sysconfig_file() {
    run_func_for_distro centos_get_sysconfig_file ubuntu_get_sysconfig_file
}

get_report_stats_crontab_file() {
    echo "/etc/cron.d/starfish"
}

set_sysconfig_param() {
    local name="$1"
    local value="$2"
    local sysconfig_file="${3:-""}"

    if [[ -z "${sysconfig_file}" ]]; then
        sysconfig_file="$(get_sysconfig_file)"
    fi

    if file_exists "${sysconfig_file}"; then
        if ! grep "${name}" "${sysconfig_file}" >/dev/null; then
            log "Parameter ${name} not found in ${sysconfig_file}"
        else
            run_with_sudo_if_not_root sed --in-place "s/[# ]*${name}=.*/${name}=${value}/" "${sysconfig_file}"
        fi
    fi
}

source_starfish_sysconfig() {
    local sysconfig_file

    sysconfig_file="$(get_sysconfig_file)"
    if [ -r "${sysconfig_file}" ]; then
        # shellcheck source=starfish/config/sysconfig-starfish
        source "${sysconfig_file}"
    fi
}

array_contains() {
    local n=$#
    local value=${!n}
    for ((i=1;i < $#;i++)) {
        if [ "${!i}" == "${value}" ]; then
            return 0
        fi
    }
    return 1
}

protect_file() {
    # make file visible to starfish group only
    local cfg_file="$1"
    local mod="${2:-0640}"
    local owner="${3:-root}"
    local group="${4:-starfish}"

    if ! chown "${owner}"."${group}" "${cfg_file}"; then
        failure "Failed to change owner to ${owner}.${group} of ${cfg_file}"
    fi
    if ! chmod "${mod}" "${cfg_file}"; then
        failure "Failed to change permissions of ${cfg_file}"
    fi
}

create_protected_file() {
    local filename="$1"
    local mod="${2:-0640}"
    local owner="${3:-root}"
    local group="${4:-starfish}"

    if ! file_exists "${filename}"; then
        log "${filename} not found, creating"
        mkdir --parents "$(dirname "${filename}")"
        touch "${filename}"
    fi
    protect_file "${filename}" "${mod}" "${owner}" "${group}"
}

centos_get_starfish_repo_file() {
    echo "/etc/yum.repos.d/starfish.repo"
}

ubuntu_get_starfish_repo_file() {
    echo "/etc/apt/sources.list.d/starfish.list"
}

centos_sysstat_conf_file() {
    echo "/etc/sysconfig/sysstat"
}

ubuntu_sysstat_conf_file() {
    echo "/etc/sysstat/sysstat"
}

get_sysstat_config_file() {
    run_func_for_distro centos_sysstat_conf_file ubuntu_sysstat_conf_file
}

setup_starfish_misc_repo() {
    local _utils_dir

    _utils_dir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

    # this function can be called from production scripts - it's then called as root user so no need to sudo
    # and some sites do not allow "sudo" for "root"
    # if called as a non-root user - it most likely means it's in devel/testing setting so try to elevate with sudo
    # add-repo-gpg-key and add-starfish-misc-repo won't work without root privileges
    run_with_sudo_if_not_root "${_utils_dir}/add-repo-gpg-key.sh"
    run_with_sudo_if_not_root "${_utils_dir}/add-starfish-misc-repo.sh"
}

install_prerequisites() {
    if ! is_offline_installation; then
        setup_starfish_misc_repo
        run_func_for_distro centos_install_epel true
    fi
}

add_starfish_misc_repo() {
    local _utils_dir

    _utils_dir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

    if ! is_offline_installation; then
        "${_utils_dir}/add-starfish-misc-repo.sh" --no-cache-update
    else
        echo "Offline installation mode active, Starfish misc repo will not be added."
    fi
}

centos_install_epel() {
    local major_version rpm_url=""

    if ! centos_pkg_installed epel-release; then
        # this function can be called from production scripts - it's then called as root user so no need to sudo
        # and some sites do not allow "sudo" for "root"
        # if called as a non-root user - it most likely means it's in devel/testing setting so try to elevate with sudo
        # add-repo-gpg-key and add-starfish-misc-repo won't work without root privileges

        # yum exits with 1 if installing from rpm and the package is already installed
        if centos_pkg_available epel-release; then
            # on CentOS epel-release is available from base
            run_with_sudo_if_not_root yum install --assumeyes epel-release
        else
            # installing from rpm, because no epel-release package on RedHat
            major_version="$(centos_get_major_version)"
            case "${major_version}" in
            7 | 8 | 9)
                rpm_url="https://dl.fedoraproject.org/pub/epel/epel-release-latest-${major_version}.noarch.rpm"
                ;;
            *)
                failure "Detected not supported major version: ${major_version}, /etc/redhat-release: $(cat /etc/redhat-release)"
                ;;
            esac
            run_with_sudo_if_not_root yum install --assumeyes "${rpm_url}"
        fi
    fi
}

insert_after_line() {
    local file="$1"
    local existing_line_in_file="$2"
    local new_line_to_insert="$3"
    sed --in-place "/${existing_line_in_file}/a ${new_line_to_insert}" "${file}"
}

is_loopback_or_incorrect() {
    local addr="$1"
    local ip

    # try to resolve IP and remove loopback addresses (127.0.0.0/8 IPv4 and ::1 IPv6)
    ip=$(getent ahosts "${addr}" | grep STREAM | grep --invert-match ^127 | grep --invert-match ^::1 || true)

    # if nothing is left it's either loopback or incorrect address
    [[ -z "${ip}" ]]
}

get_first_ip_address_or_localhost() {
    local ret

    ret=$(hostname --all-ip-addresses 2>/dev/null | cut -f 1 -d " " || true)
    if [[ -z "${ret}" ]]; then
        # if getting IP fails, declare failure and call it "localhost"
        ret=localhost
    fi

    echo -n "${ret}"
}

get_hostname() {
    local ret

    # hostname -f may fail with "hostname: Unknown host", --all-fqdns does not
    # there may potentially be a few "hostnames", take the first one
    ret=$(hostname --all-fqdns 2>/dev/null | cut -f 1 -d " " || true)
    if [[ -n "${ret}" ]]; then
        if ! is_loopback_or_incorrect "${ret}"; then
            echo -n "${ret}"
            return
        fi
    fi
    # if can't find hostname, get first IP address
    get_first_ip_address_or_localhost
}

glob_matches_at_least_once() {
    local pattern="$1"

    compgen -G "${pattern}" > /dev/null
}

setup_local_repo() {
    run_func_for_distro setup_yum_local_repo setup_apt_local_repo "$@"
}

create_apt_local_repo() {
    local pkg_dir="$1"

    # creating local apt repo because dpkg does not fetch dependencies and all the scripts assume that
    # apt-get install starfish works
    # https://askubuntu.com/a/176546

    run_with_sudo_if_not_root apt-get install -y dpkg-dev apt-utils
    cd "${pkg_dir}"
    # --multiversion is needed in case there are packages for multiple distribution revisions
    # for example: sf-examples_1.0.0-22.focal_amd64.deb and sf-examples_1.0.0-22.focal_amd64.deb
    # Both versions will be added to "Packages"
    dpkg-scanpackages --multiversion . > Packages
    apt-ftparchive release . > Release
}

update_with_apt_local_repo() {
    local pkg_dir="$1"
    local repo_name="$2"

    echo "deb [trusted=yes] file:${pkg_dir} ./" > "/etc/apt/sources.list.d/${repo_name}.list"
    apt-get update
}

setup_apt_local_repo() {
    local pkg_dir="$1"
    local repo_name="${2:-starfish-$(basename "${pkg_dir}")}"

    # Our offline bundles come with repo structure already created.
    if ! is_offline_installation; then
        create_apt_local_repo "${pkg_dir}"
    fi
    update_with_apt_local_repo "${pkg_dir}" "${repo_name}"
}

create_yum_local_repo() {
    local pkg_dir="$1"

    if [[ "$(centos_get_major_version)" -lt 8 ]]; then
        centos_pkg_install createrepo
    else
        centos_pkg_install createrepo_c
    fi
    createrepo "${pkg_dir}"
}

update_with_yum_local_repo() {
    local pkg_dir="$1"
    local repo_name="${2:-""}"
    local build repo_id repo_file

    if [[ -z "${repo_name}" ]]; then
        build="$(basename "${pkg_dir}")"
        repo_name="Starfish ${build}"
        repo_id="starfish-${build}"
        repo_file="/etc/yum.repos.d/starfish-${build}.repo"
    else
        repo_id="${repo_name}"
        repo_file="/etc/yum.repos.d/${repo_name}.repo"
    fi

    cat > "${repo_file}" <<EOF
[${repo_id}]
name=${repo_name}
baseurl=file://${pkg_dir}
enabled=1
gpgcheck=0
protect=1
EOF
    if is_offline_installation && [[ "$(centos_get_major_version)" -ge 8 ]]; then
        # convince EL 8/9 yum to install certain non-modular packages from the offline repo
        echo "module_hotfixes=1" >> "${repo_file}"
    fi
    centos_makecache
}

setup_yum_local_repo() {
    local pkg_dir="$1"
    local repo_name="${2:-starfish-$(basename "${pkg_dir}")}"

    # Our offline bundles come with repo structure already created.
    if ! is_offline_installation; then
        create_yum_local_repo "${pkg_dir}"
    fi
    update_with_yum_local_repo "${pkg_dir}" "${repo_name}"
}

_get_mail_help() {
    mail --help 2>&1 || true
}

get_attachment_option() {
    # some clients have lowercase -a for attachments, other have -A ...
    if _get_mail_help | grep -- '-A, --attach=FILE' > /dev/null; then
        # mail.mailutils
        echo "-A"
    else
        if _get_mail_help | grep -- 'Usage: mail .* -a FILE' > /dev/null ; then
            # mailx
            echo "-a"
        fi
    fi
}

get_timezone() {
    timedatectl status --no-pager | grep "Time zone" | sed --regexp-extended "s|.*Time zone: ([-a-zA-Z/_0-9\+]+)(.*)|\1|"
}

extract_hostname_from_url() {
    local url="$1"

    echo "${url}" | cut -f 3 -d / | cut -f 1 -d :
}

distr_to_pkg_format() {
    local distribution=$1

    run_func_for "${distribution}" "echo rpm" "echo deb"
}

is_ipv6_address() {
    local address="$1"

    [[ $(echo "${address}" | grep --count --fixed-strings :) -gt 0 ]]
}

ipv6_enabled() {
    # assume that IPv6 is enabled if any interface has any IPv6 addr
    test -n "$(ip -6 addr)"
}

encode_user_password() {
    local text="$1"
    local length="${#1}"
    local c i

    # Encode user and password fields.
    # From https://datatracker.ietf.org/doc/html/rfc1738.html#section-3.1
    # Within the user and password field, any ":", "@", or "/" must be encoded.

    for (( i = 0; i < length; i++ )); do
        c="${text:${i}:1}"
        case "${c}" in
            [^:@/%])
                printf '%s' "${c}"
                ;;
            *)
                # '${c} is not a bug, it's intended
                printf '%%%02X' "'${c}"
                ;;
        esac
    done
}

host_part_from_hostname() {
    local hostname="$1"

    # https://datatracker.ietf.org/doc/html/rfc2732.html requires
    # to surround literal IPv6 address with `[]`, otherwise no-op
    if is_ipv6_address "${hostname}"; then
        echo "[${hostname}]"
    else
        echo "${hostname}"
    fi
}

run_func_for_service_manager() {
    local systemd_list_units_pattern="$1"
    local systemd_func="$2"
    local supervisord_grep_condition="$3"
    local supervisord_func="$4"
    shift 4

    if systemctl_knows_about "${systemd_list_units_pattern}"; then
        ${systemd_func} "$@"
    elif [[ $(command -v supervisorctl) ]]; then
        if [[ $(supervisorctl avail | grep --count "${supervisord_grep_condition}") -gt 0 ]]; then
           ${supervisord_func} "$@"
        fi
    fi
}

get_systemd_sf_target_name() {
    echo "starfish.target"
}

get_supervisord_sf_group_name() {
    echo "sf:"
}

# Echoes a newline-delimited list of members of the unit, recursively (i.e. units that have this unit listed under
# PartOf parameter, either directly or recursively through intermediate units), plus the name of unit itself.
_systemd_unit_members_list() {
    local unit_name="$1"
    local depth="${2:-1}"
    local members

    if [[ "${depth}" -ge 10 ]]; then
        failure "Invalid systemd unit configuration: 'PartOf=' cycle highly likely (including ${unit_name}). Exiting."
    fi
    echo "${unit_name}"
    members=$(systemctl show --plain --no-legend --property=ConsistsOf "${unit_name}" | sed s/ConsistsOf=//)
    for member in ${members}; do
        _systemd_unit_members_list "${member}" $((depth+1))
    done
}

systemd_unit_members_list() {
    local resetxtrace=0

    if shopt -q -o xtrace; then
        resetxtrace=1
    fi
    # disable debug because logs from loops waiting for services to start become unbearably long
    set +x

    # reload systemd config
    # when upgrading sf-nginx2 package to a version with sf-nginx2.service file
    # it may happen that after upgrade sf-nginx.service gets removed but systemd is not reloaded after it is removed
    # so systemd still thinks that starfish.target depends on sf-nginx.service
    # and we wait for sf-nginx.service to start which will never happen
    systemctl daemon-reload
    _systemd_unit_members_list "$@"

    if [[ "${resetxtrace}" -eq 1 ]]; then
        set -x
    fi
}

systemd_unit_members_count() {
    local unit_name="$1"

    systemd_unit_members_list "${unit_name}" | wc -w
}

# running == "active" state
systemd_unit_running_members_count() {
    local unit_name="$1"

    # shellcheck disable=SC2046
    systemctl is-active $(systemd_unit_members_list "${unit_name}") | grep --count ^active$
}

# stopped == "inactive" or "failed" state
systemd_unit_stopped_members_count() {
    local unit_name="$1"

    # shellcheck disable=SC2046
    systemctl is-active $(systemd_unit_members_list "${unit_name}") | grep --count --extended-regexp "^(inactive|failed)$"
}

systemd_all_unit_members_running() {
    local unit_name="$1"

    if systemctl_knows_about "${unit_name}"; then
        [[ "$(systemd_unit_running_members_count "${unit_name}")" == "$(systemd_unit_members_count "${unit_name}")" ]]
    fi
}

systemd_all_unit_members_stopped() {
    local unit_name="$1"

    if systemctl_knows_about "${unit_name}"; then
        [[ "$(systemd_unit_stopped_members_count "${unit_name}")" == "$(systemd_unit_members_count "${unit_name}")" ]]
    fi
}

systemd_check_unit_running() {
    local unit_name="$1"
    shift

    if systemctl_knows_about "${unit_name}"; then
        test "$(systemd_unit_running_members_count "${unit_name}")" "$@"
    fi
}

systemd_check_unit_stopped() {
    local unit_name="$1"
    shift

    if systemctl_knows_about "${unit_name}"; then
        test "$(systemd_unit_stopped_members_count "${unit_name}")" "$@"
    fi
}

systemctl_knows_about() {
    local unit="$1"

    test "$(systemctl --no-pager --no-legend --all list-units "${unit}" | wc -l)" -gt 0
}

curl_silent_to_stdout() {
    # --show-error When used with -s, --silent, it makes curl show an error message if it fails.
    #
    # --silent Silent or quiet mode. Don't show progress meter or error messages.
    #          Makes Curl mute. It will still output the data you ask for, potentially even to the termi nal/stdout
    #          unless you redirect it.
    #
    # --fail   (HTTP) Fail silently (no output at all) on server errors. This is mostly done to better enable scripts
    #          etc to better deal with failed attempts. In normal cases when an HTTP server fails to deliver a document,
    #          it returns an HTML document stating so (which often also describes why and more).
    #          This flag will prevent  curl from outputting that and return error 22.
    #          This  method  is  not  fail-safe and there are occasions where non-successful response codes will slip
    #          through, especially when authentication is involved (response codes 401 and 407).
    #
    # --location (HTTP) If the server reports that the requested page has moved to a different location (indicated with
    #            a Location: header and a 3XX response code), this option will make curl redo the request on the new
    #            place. If used together with -i, --include or -I, --head, headers from all requested pages will be
    #            shown.
    #            When  authentication  is  used,  curl  only  sends its credentials to the initial host.
    #            If a redirect takes curl to a different host, it won't be able to intercept the user+password.
    #            See also --location-trusted on how to change this.
    #            You can limit the amount of redirects to follow by using the --max-redirs option.
    #            When curl follows a redirect and the request is not a plain GET (for example POST or PUT), it will do
    #            the following request with a GET if the HTTP response was 301, 302, or 303.
    #            If the response code was any other 3xx code, curl will re-send the following request using the same
    #            unmodified method.
    #            You  can  tell  curl  to  not  change  the non-GET request method to GET after a 30x response by using
    #            the dedicated options for that: --post301, --post302 and --post303.

    # to work around problem with certificate errors add "--insecure"
    # for wget it used to be replace --check-certificate with --no-check-certificate
    curl --fail --location --silent --show-error "$@"
}

curl_silent_to_file() {
    # --remote-name Write output to a local file named like the remote file we get. (Only the file part of the remote
    #               file is used, the path is cut off.)
    #               The  file  will be saved in the current working directory. If you want the file saved in a different
    #               directory, make sure you change the current working directory before invoking curl with this option.
    #               The remote file name to use for saving is extracted from the given URL, nothing else, and if it
    #               already exists it will be overwritten. If you want  the  server to  be  able to choose the file name
    #               refer to -J, --remote-header-name which can be used in addition to this option.
    #               If the server chooses a file name and that name already exists it will not be overwritten.
    #
    #               There is no URL decoding done on the file name. If it has %20 or other URL encoded parts of the name,
    #               they will end up as-is as file name.
    #
    #               You may use this option as many times as the number of URLs you have.
    curl_silent_to_stdout --remote-name "$@"
}

starfish_offline_installation_marker_file() {
    STARFISH_ETC_DIR=$(readlink --verbose --canonicalize-missing "${SFHOME:-/opt/starfish}/etc")

    echo "${STARFISH_ETC_DIR}/offline_install"
}

is_offline_installation() {
    test "${SF_OFFLINE_INSTALLATION:-no}" == yes || file_exists "$(starfish_offline_installation_marker_file)"
}

starfish_offline_repo_name() {
    echo "starfish-offline-repo"
}

centos_add_repo() {
    local name="$1"
    local section="$2"
    local url="$3"
    local file="$4"

    cat << EOF | tee "$(centos_get_starfish_repo_file)" >/dev/null
[${section}]
name=${name}
baseurl=${url}
gpgcheck=1
enabled=1
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-STARFISH
EOF
}

ubuntu_add_repo() {
    local url="$1"
    local file="$2"
    local codename

    codename=$(map_to_ubuntu_distro)

    case "${codename}" in
    focal | jammy)
        ;;
    *)
        log "Codename '${codename}' not supported"
        exit 1
        ;;
    esac

    echo "deb ${url} ${codename} main" > "${file}"
}

centos_enable_packages_cache() {
    # Using set_ini_property would be better, but we don't want to install augeas before enabling the cache.
    sed -i "/keepcache=/d" /etc/yum.conf
    echo "keepcache=1" >>/etc/yum.conf

    if [[ "$(centos_get_major_version)" -ge 8 ]]; then
        sed -i "/keepcache=/d" /etc/dnf/dnf.conf
        echo "keepcache=1" >>/etc/dnf/dnf.conf
    fi
}

ubuntu_enable_packages_cache() {
    cat <<EOF >/etc/apt/apt.conf.d/99enablecache
Binary::apt::APT::Cache "/var/cache/apt";
Binary::apt::APT::Cache::Archives "archives";
Binary::apt::APT::Keep-Downloaded-Packages "true";
APT::Keep-Downloaded-Packages "true";
EOF
    # some Docker images have cofiguration which removes `*.deb` from cache after installation
    rm --verbose --force /etc/apt/apt.conf.d/docker-clean
}

enable_packages_cache() {
    run_func_for_distro centos_enable_packages_cache ubuntu_enable_packages_cache
}

centos_clean_packages_cache() {
    yum clean all
}

ubuntu_clean_packages_cache() {
    apt-get clean
}

clean_packages_cache() {
    run_func_for_distro centos_clean_packages_cache ubuntu_clean_packages_cache
}

ubuntu_gather_packages() {
    local repo_url="$1"
    local target_dir="$2"

    if dir_exists "${repo_url}"; then
        # In case of an installation from a directory, Starfish packages won't be present in the apt cache.
        find "${repo_url}" -name "*.deb" -exec cp {} "${target_dir}" \;
    fi

    find /var/cache/apt -name "*.deb" -exec cp {} "${target_dir}" \;
}

centos_gather_packages() {
    local repo_url="$1"
    local target_dir="$2"

    if dir_exists "${repo_url}"; then
        # In case of an installation from a directory, Starfish packages won't be present in the yum/dnf cache.
        find "${repo_url}" -name "*.rpm" -exec cp {} "${target_dir}" \;
    fi

    if [[ "$(centos_get_major_version)" -lt 8 ]]; then
        find /var/cache/yum -name "*.rpm" -exec cp {} "${target_dir}" \;
    else
        find /var/cache/dnf -name "*.rpm" -exec cp {} "${target_dir}" \;
    fi
}

gather_packages() {
    run_func_for_distro centos_gather_packages ubuntu_gather_packages "$@"
}
