#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
readonly SCRIPT_DIR
LOG_FILE="$(basename "${BASH_SOURCE[0]}" '.sh').log"
# shellcheck source=scripts/installation/_common_install.sh
source "${SCRIPT_DIR}/_common_install.sh"
# shellcheck source=scripts/installation/_ssl.sh
source "${SCRIPT_DIR}/_ssl.sh"

centos_get_starfish_repo_url() {
    file_exists "$(centos_get_starfish_repo_file)" ||
        {
            log "File $(centos_get_starfish_repo_file) doesn't exist."
            exit 1
        }
    grep "baseurl" "$(centos_get_starfish_repo_file)" | cut -d'=' -f2
}

centos_add_starfish_repo_file() {
    local repo_url="$1"

    log "Adding Starfish repository from URL: ${repo_url}"
    centos_add_repo "Starfish Repository" Starfish "${repo_url}" "$(centos_get_starfish_repo_file)"
}

ubuntu_get_starfish_repo_url() {
    file_exists "$(ubuntu_get_starfish_repo_file)" ||
        {
            log "File $(ubuntu_get_starfish_repo_file) doesn't exist."
            exit 1
        }
    cut -d' ' -f2 "$(ubuntu_get_starfish_repo_file)"
}

ubuntu_add_starfish_repo_file() {
    local repo_url="$1"

    log "Adding Starfish repository from URL: ${repo_url}"

    ubuntu_add_repo "${repo_url}" "$(ubuntu_get_starfish_repo_file)"
    apt-get update
}

add_starfish_repo_file() {
    run_func_for_distro centos_add_starfish_repo_file ubuntu_add_starfish_repo_file "$@"
}

centos_install_starfish() {
    local packages=("$@")

    yum install --assumeyes "${packages[@]}" || {
        log "There was an error while installing packages: ${packages[*]}. See yum logs for more details."
        local repo_file
        repo_file="$(centos_get_starfish_repo_file)"
        if ! file_exists "${repo_file}"; then
            log "File ${repo_file} doesn't exist. Starfish repository hasn't been installed. Run install-Starfish.sh"
        else
            log "If the repository URL is not correct ($(centos_get_starfish_repo_url)), edit ${repo_file} manually and run the script again."
        fi
        exit 1
    }
}

ubuntu_install_starfish() {
    local packages=("$@")
    local repo_file

    # --force-confdef and --force-confold because install scripts creates config before installing packages
    # this means that config files in *.deb packages may conflict with the ones on disk and dpkg will ask
    # the operator what to do, unless given --force-conf... options:
    # from "man dpkg:
    #   confold: If a conffile has been modified and the version in the package did change, always keep the
    #            old version without prompting, unless the --force-confdef is also specified, in which case the default
    #            action is preferred.
    #
    #   confdef: If a conffile has been modified and the version in the package did change, always choose the
    #            default action without prompting. If there is no default action it will stop to ask  the  user  unless
    #            --force-confnew  or  --force-confold  is also been given, in which case it will use that to decide the
    #            action.
    apt-get -o Dpkg::Options::='--force-confdef' -o Dpkg::Options::='--force-confold' install -y --force-yes "${packages[@]}" || {
        log "There was an error while installing packages: ${packages[*]}. See apt logs for more details."
        repo_file="$(ubuntu_get_starfish_repo_file)"
        if ! file_exists "${repo_file}"; then
            log "File ${repo_file} doesn't exist. Starfish repository hasn't been installed. Run install-Starfish.sh"
        else
            log "If the repository URL is not correct ($(ubuntu_get_starfish_repo_url)), edit ${repo_file} manually and run the script again."
        fi
        exit 1
    }
}

install_starfish() {
    run_func_for_distro centos_install_starfish ubuntu_install_starfish "$@"
}

_set_existing_starfish_config_param() {
    local config_file="$1"
    local name="$2"
    local value="$3"

    # setup parameter if it's already present in config
    # if possible use parameter defined within comment
    sed --in-place "s|[#; ]*${name}=.*$|${name}=${value}|" "${config_file}"

    grep --quiet "^${name}=${value}" "${config_file}" # return false if parameter does not exist in config
}

configure_https() {
    local cert_name="sf-ssl"
    local cert_file="${STARFISH_ETC_DIR}/${cert_name}.crt"
    local key_file="${STARFISH_ETC_DIR}/${cert_name}.key"

    log "Configuring HTTPS in ${STARFISH_ETC_LOCAL_INI} using ${cert_name}.(key|crt)"

    mkdir --parents "${STARFISH_ETC_DIR}"

    create_ssl_cert "${STARFISH_ETC_DIR}" "${cert_name}" "Starfish"

    protect_file "${key_file}" 0600 root root

    set_ini_property "${STARFISH_ETC_LOCAL_INI}" "global" "ssl_certificate_file" "${cert_file}"
    set_ini_property "${STARFISH_ETC_LOCAL_INI}" "global" "ssl_private_key_file" "${key_file}"
}

configure_nginx() {
    local listen_address="$1"
    local nginx_local_file="${SFHOME}/nginx/etc/conf.d/starfish/local.conf"

    log "Configuring nginx in ${nginx_local_file} to listen on ${listen_address}:443"

    create_protected_file "${nginx_local_file}" 0644 root root
    echo "listen $(host_part_from_hostname "${listen_address}"):443 default_server ssl http2;" >"${nginx_local_file}"
}

configure_starfish() {
    local listen_address="$1"
    local pg_uri="$2"
    local add_port_to_config_service_url="$3"
    local config_service_url
    local keyfile
    local cfg_keyfile="${STARFISH_ETC_DIR}/license"
    local licenses_dir="${STARFISH_ETC_DIR}/licenses"
    local local_ini_exists_before

    # Check if 99-local.ini exists (from previous installation). If it does, skip adding the comment
    # further in the function. Set to 0 if the file does not exist at this moment.
    local_ini_exists_before=$(
        [[ ! -e "${STARFISH_ETC_LOCAL_INI}" ]]
        echo $?
    )

    # starfish group most likely does not exist yet, create those files protected as much as possible
    # post-install.sh of starfish package will change the group to starfish and permissions to 0640
    create_protected_file "${STARFISH_ETC_LOCAL_INI}" 0600 root root
    create_protected_file "${STARFISH_ETC_SERVICE_INI}" 0600 root root
    create_protected_file "${STARFISH_ETC_SECRETS_INI}" 0600 root root

    # 0 means that the local INI file did not exist before creation attempt above.
    # This condition is necessary because create_protected_file does not always create the file from scratch.
    if [[ ${local_ini_exists_before} -eq 0 ]]; then
        cat <<"EOF" >"${STARFISH_ETC_LOCAL_INI}"
; This file is used for customer specific changes that are local to this Starfish server.
; It *should not* be put under configuration management control nor reverted or locked
; in any way because it may break some necessary changes in the future, particularly
; those related to security, credentials, and authorization.
; Customers wishing to have a file that is fully under their control to contain
; customizations relating to scans, jobs, agents, etc. should use a different file.
; Creating a file at /opt/starfish/etc/99-site.ini would be one possibility and will not be
; overwritten by Starfish now or at any time in the future.

EOF
    fi

    configure_https
    configure_nginx "${listen_address}"

    log "Configuring Starfish in ${STARFISH_ETC_LOCAL_INI}"

    config_service_url="https://$(host_part_from_hostname "${listen_address}")"
    if [[ "${add_port_to_config_service_url}" == "yes" ]]; then
        config_service_url="${config_service_url}:30003"
    fi
    set_ini_property "${STARFISH_ETC_SERVICE_INI}" global config_service_url "${config_service_url}"

    set_ini_property "${STARFISH_ETC_LOCAL_INI}" global bind_host "${listen_address}"
    set_ini_property "${STARFISH_ETC_LOCAL_INI}" global starfish_url_prefix "https://$(host_part_from_hostname "${listen_address}")"
    set_ini_property "${STARFISH_ETC_LOCAL_INI}" global default_agent_service_url "https://$(host_part_from_hostname "${listen_address}"):30002"

    set_ini_property "${STARFISH_ETC_LOCAL_INI}" pg pg_uri "${pg_uri}"

    keyfile="$(locate_keyfile "${SCRIPT_DIR}/..")"
    if ! file_exists "${cfg_keyfile}"; then
        log "Copying keyfile ${keyfile} to ${cfg_keyfile}"
        cp "${keyfile}" "${cfg_keyfile}"
    else
        log "Using existing keyfile ${keyfile}"
    fi

    if ! dir_exists "${licenses_dir}"; then
        mkdir --parents --verbose "${licenses_dir}"
    fi
}

configure_diagnostic_reports() {
    local enable_diag_reports_to_sf="$1"
    local enable_json_diag_reports_to_sf="$2"
    local enable_confidential_obfuscation="$3"
    local extra_emails="$4"
    local extra_emails_errors_only="$5"
    local diag_reports_from_email="$6"
    local diag_reports_ignored_errors_pattern="$7"

    set_sysconfig_param SENDING_DIAGNOSTIC_REPORTS_ENABLED true
    set_sysconfig_param SENDING_DIAGNOSTIC_REPORTS_TO_STARFISH_ENABLED "${enable_diag_reports_to_sf}"
    set_sysconfig_param SENDING_DIAGNOSTIC_JSON_TO_STARFISH_ENABLED "${enable_json_diag_reports_to_sf}"
    set_sysconfig_param OBFUSCATE_CONFIDENTIAL_INFO_ENABLED "${enable_confidential_obfuscation}"
    # extra emails could be set in starfish cron file, but this would make any future updates of the cron file
    # more difficult as package managers refuse to update/remove already changed files
    set_sysconfig_param DIAGNOSTIC_REPORTS_EXTRA_EMAILS "${extra_emails}"
    set_sysconfig_param DIAGNOSTIC_REPORTS_EXTRA_EMAILS_ERRORS_ONLY "${extra_emails_errors_only}"
    set_sysconfig_param DIAGNOSTIC_REPORTS_FROM_EMAIL "${diag_reports_from_email}"
    set_sysconfig_param DIAGNOSTIC_REPORTS_IGNORED_ERRORS_PATTERN "${diag_reports_ignored_errors_pattern}"
}

ubuntu_install_local_starfish() {
    local package_dir="$1"
    shift
    local packages=("$@")

    if ! is_offline_installation; then
        # In offline mode this is done by install.sh
        setup_apt_local_repo "${package_dir}"
    fi
    ubuntu_install_starfish "${packages[@]}"
}

centos_install_local_starfish() {
    local package_dir="$1" # also known as repo_dir in 'prepare_starfish'
    shift
    local packages=("$@")

    if ! is_offline_installation; then
        # In offline mode this is done by install.sh
        setup_yum_local_repo "${package_dir}"
    fi
    centos_install_starfish "${packages[@]}"
}

install_local_starfish() {
    run_func_for_distro centos_install_local_starfish ubuntu_install_local_starfish "$@"
}

prepare_starfish() {
    local repo_url="$1"
    local listen_address="$2"
    local pg_uri="$3"
    local install_redash="$4"
    local add_port_to_config_service_url="$5"
    local package_dir

    local base_packages=(starfish sf-agent)
    if [[ "${install_redash}" = "yes" ]]; then
        base_packages+=(sf-redash)
    fi

    configure_starfish "${listen_address}" "${pg_uri}" "${add_port_to_config_service_url}"
    if dir_exists "${repo_url}"; then
        # repo_url is a directory, install from locally provided deb or rpm files.
        package_dir=${repo_url}
        install_local_starfish "${package_dir}" "${base_packages[@]}"
    else
        if ! is_offline_installation; then
            add_starfish_repo_file "${repo_url}"
        fi
        install_starfish "${base_packages[@]}"
    fi
    start_starfish
}

main() {
    fail_if_internal_script_run_directly

    local repo_url=""
    local listen_address="127.0.0.1"
    local pg_uri=""
    local enable_diag_reports="yes"
    local enable_diag_reports_to_sf="true"
    local enable_json_diag_reports_to_sf="true"
    local enable_confidential_obfuscation="false"
    local diag_reports_extra_emails=""
    local diag_reports_extra_emails_errors_only=""
    local diag_reports_from_email=""
    local diag_reports_ignored_errors_pattern=""
    local install_redash="yes"
    local add_port_to_config_service_url="no"
    local args=("$@")

    while [ $# -gt 0 ]; do
        case $1 in
        --log-file)
            shift
            LOG_FILE="$1"
            ;;
        --repo-url)
            shift
            repo_url="$1"
            ;;
        --listen-address)
            shift
            listen_address="$1"
            ;;
        --pg-uri)
            shift
            pg_uri="$1"
            ;;
        --disable-diag-reports)
            enable_diag_reports="no"
            ;;
        --disable-diag-reports-to-sf)
            enable_diag_reports_to_sf="false"
            ;;
        --disable-json-diag-reports-to-sf)
            enable_json_diag_reports_to_sf="false"
            ;;
        --enable-confidential-obfuscation)
            enable_confidential_obfuscation="true"
            ;;
        --diag-reports-extra-emails-errors-only)
            shift
            diag_reports_extra_emails_errors_only="$1"
            ;;
        --diag-reports-extra-emails)
            shift
            diag_reports_extra_emails="$1"
            ;;
        --diag-reports-from-email)
            shift
            diag_reports_from_email="$1"
            ;;
        --diag-reports-ignored-errors-pattern)
            shift
            diag_reports_ignored_errors_pattern="$1"
            ;;
        --no-redash)
            install_redash="no"
            ;;
        --add-port-to-config-service-url)
            add_port_to_config_service_url="yes"
            ;;
        *)
            break
            ;;
        esac
        shift
    done
    export LOG_FILE
    log "Script called with arguments: $0 " ${args[@]+"${args[@]}"}

    if (($# != 0)); then
        log "Unrecognised arguments" "$@"
        exit 2
    fi

    fail_if_not_root
    fail_if_selinux_enforcing

    if empty_string "${repo_url}"; then
        log "No value for --repo-url."
        exit 1
    fi
    if empty_string "${pg_uri}"; then
        log "No value for --pg-uri."
        exit 1
    fi
    set_convenient_umask
    prepare_starfish "${repo_url}" "${listen_address}" "${pg_uri}" "${install_redash}" "${add_port_to_config_service_url}"
    if [ "${enable_diag_reports}" = "yes" ]; then
        configure_diagnostic_reports \
            "${enable_diag_reports_to_sf}" \
            "${enable_json_diag_reports_to_sf}" \
            "${enable_confidential_obfuscation}" \
            "${diag_reports_extra_emails}" \
            "${diag_reports_extra_emails_errors_only}" \
            "${diag_reports_from_email}" \
            "${diag_reports_ignored_errors_pattern}"
    fi
}

sourced() {
    [[ "${BASH_SOURCE[0]}" != "${0}" ]]
}

if ! sourced; then
    main "$@"
fi
