#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************
set -euo pipefail

SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
readonly SCRIPT_DIR

LOG_FILE="$(basename "${BASH_SOURCE[0]}" '.sh').log"

# shellcheck source=scripts/installation/_common_install.sh
source "${SCRIPT_DIR}/_common_install.sh"

moment_date_format_from_locale() {
    # d_fmt uses strftime characters (see man strftime),
    # REDASH_DATE_FORMAT is MomentJS date format https://momentjs.com/docs/#/displaying/format/
    # following sed expression is based on
    # https://github.com/benjaminoakes/moment-strftime/blob/1886cabc4b07d13e3046ae075d357e7aad92ea93/lib/moment-strftime.js#L10
    locale d_fmt | sed \
          -e 's/%a/EEE/g' \
          -e 's/%a/ddd/g' \
          -e 's/%A/dddd/g' \
          -e 's/%b/MMM/g' \
          -e 's/%B/MMMM/g' \
          -e 's/%c/lll/g' \
          -e 's/%d/DD/g' \
          -e 's/%-d/D/g' \
          -e 's/%e/D/g' \
          -e 's/%F/YYYY-MM-DD/g' \
          -e 's/%H/HH/g' \
          -e 's/%-H/H/g' \
          -e 's/%I/hh/g' \
          -e 's/%-I/h/g' \
          -e 's/%j/DDDD/g' \
          -e 's/%-j/DDD/g' \
          -e 's/%k/H/g' \
          -e 's/%l/h/g' \
          -e 's/%m/MM/g' \
          -e 's/%-m/M/g' \
          -e 's/%M/mm/g' \
          -e 's/%-M/m/g' \
          -e 's/%p/A/g' \
          -e 's/%P/a/g' \
          -e 's/%S/ss/g' \
          -e 's/%-S/s/g' \
          -e 's/%u/E/g' \
          -e 's/%w/d/g' \
          -e 's/%W/WW/g' \
          -e 's/%x/ll/g' \
          -e 's/%X/LTS/g' \
          -e 's/%y/YY/g' \
          -e 's/%Y/YYYY/g' \
          -e 's/%z/ZZ/g' \
          -e 's/%Z/z/g' \
          -e 's/%f/SSS/g'
}

systemd_restart_redash() {
    local redash_service_manager_name="$1"

    log "systemctl restart ${redash_service_manager_name}.target"
    systemctl restart "${redash_service_manager_name}.target" | log
}

supervisord_restart_redash() {
    local redash_service_manager_name="$1"

    log "supervisorctl restart ${redash_service_manager_name}:*"
    supervisorctl restart "${redash_service_manager_name}:*" | log
}

restart_redash() {
    local redash_version="$1"
    local redash_service_manager_name

    redash_service_manager_name="$(get_redash_service_manager_name "${redash_version}")"

    log "Restarting Redash..."
    # restart as we might run into the following race condition
    # * packages are installed and systemd/supervisord config uploaded
    # * systemd/supervisord sf-redash target/group gets started but Redash (and thus Celery) is not yet configured
    # * Celery (started by systemd/supervisord) tries to use default PG and Redis, founds none, exits with error
    # * Celery gets restarted by systemd/supervisord, fails again and again
    # * after a few tries systemd/supervisord moves Celery to "failed" state
    # systemctl start XYZ.target does try to start "failed" processes, but supervisorctl start XYZ:* doesn't
    run_func_for_service_manager "${redash_service_manager_name}.target" systemd_restart_redash "${redash_service_manager_name}" supervisord_restart_redash "${redash_service_manager_name}"

    log "waiting for all ${redash_service_manager_name} services to start"
    wait_for "_check_running_services ${redash_service_manager_name} -gt 0" 30
    wait_for "_check_not_running_services ${redash_service_manager_name} -eq 0" 30
}

update_redash_env_settings() {
    local redash_host="$1"
    local redash_version="$2"

    local restart=false
    local redash_env

    redash_env="$(get_redash_env_filename "${redash_version}")"

    if ! file_exists "${redash_env}"; then
        log "update_redash_env_settings: ${redash_env} does not exist, nothing to update"
        return
    fi

    if [[ $(grep --count REDASH_HOST "${redash_env}") -eq 0 ]]; then
        log "Adding REDASH_HOST to ${redash_env}"
        insert_after_line "${redash_env}" "^REDASH_DATE_FORMAT" "REDASH_HOST=https://${redash_host}/redash"
        restart=true
    fi

    if [[ $(grep --count REDASH_MAIL_DEFAULT_SENDER "${redash_env}") -eq 0 ]]; then
        log "Adding REDASH_MAIL_DEFAULT_SENDER to ${redash_env}"
        insert_after_line "${redash_env}" "^REDASH_LOG_LEVEL" "REDASH_MAIL_DEFAULT_SENDER=starfish@${redash_host}"
        restart=true
    fi

    if [[ $(grep --count "#use the following to configure sending" "${redash_env}") -eq 0 ]]; then
        log "Adding REDASH_MAIL comments to ${redash_env}"
        cat >> "${redash_env}" <<EOF
#use the following to configure sending mail server if you want Redash to send out e-mails
#by default SMTP server on localhost:25 is used
#REDASH_MAIL_PASSWORD=
#REDASH_MAIL_PORT=25
#REDASH_MAIL_SERVER=localhost
#REDASH_MAIL_USERNAME=
#REDASH_MAIL_USE_SSL=false
#REDASH_MAIL_USE_TLS=false
EOF
        restart=true
    fi

    if [[ $(grep --count "#Redash JWT" "${redash_env}") -eq 0 ]]; then
        log "Adding Redash JWT settings to ${redash_env}"
        cat >> "${redash_env}" <<EOF

#Redash JWT
REDASH_JWT_LOGIN_ENABLED=true
REDASH_JWT_AUTH_PUBLIC_CERTS_URL="file:///opt/starfish/etc/jwtRS256.pem"
REDASH_JWT_AUTH_ISSUER="sf-auth"
REDASH_JWT_AUTH_AUDIENCE="Starfish"
REDASH_JWT_AUTH_COOKIE_NAME="sf-jwt-token"
EOF
        restart=true
    fi

    if [[ $(grep --count REDASH_FEATURE_ALLOW_CUSTOM_JS_VISUALIZATIONS "${redash_env}") -eq 0 ]]; then
        cat >> "${redash_env}" <<EOF

#Enable "Custom" Chart Type
REDASH_FEATURE_ALLOW_CUSTOM_JS_VISUALIZATIONS=true
EOF
        restart=true
    fi

    if [[ $(grep --count REDASH_INVITATION_TOKEN_MAX_AGE "${redash_env}") -eq 0 ]]; then
        cat >> "${redash_env}" <<EOF

#Make invitation links sent by email valid for 1 hour
REDASH_INVITATION_TOKEN_MAX_AGE=3600
EOF
        restart=true
    fi

    if [[ $(grep --count REDASH_VERSION_CHECK "${redash_env}") -eq 0 ]]; then
        log "Adding REDASH_VERSION_CHECK to ${redash_env}"
        insert_after_line "${redash_env}" "^REDASH_REDIS_URL" "REDASH_VERSION_CHECK=false"
        restart=true
    fi

    # we would like to check whether upgrade from some old redash works; in such case
    # in old SF instances CSRF was not on - here we simulate it (if redash version is different than 10, then
    # we would skip enabling CSRF)
    if [[ "${redash_version}" -eq 10 ]]; then
        if [[ $(grep --count REDASH_ENFORCE_CSRF "${redash_env}") -eq 0 ]]; then
            log "Adding REDASH_ENFORCE_CSRF to ${redash_env}"
            insert_after_line "${redash_env}" "^REDASH_VERSION_CHECK" "REDASH_ENFORCE_CSRF=true"
            restart=true
        fi
        if [[ $(grep --count REDASH_ENFORCE_CSRF_FOR_AUTH_USERS "${redash_env}") -eq 0 ]]; then
            log "Adding REDASH_ENFORCE_CSRF_FOR_AUTH_USERS to ${redash_env}"
            insert_after_line "${redash_env}" "^REDASH_ENFORCE_CSRF" "REDASH_ENFORCE_CSRF_FOR_AUTH_USERS=true"
            restart=true
        fi
        if [[ $(grep --count REDASH_CSRF_TIME_LIMIT "${redash_env}") -eq 0 ]]; then
            log "Adding REDASH_CSRF_TIME_LIMIT to ${redash_env}"
            insert_after_line "${redash_env}" "^REDASH_ENFORCE_CSRF_FOR_AUTH_USERS" "REDASH_CSRF_TIME_LIMIT=999888000"
            restart=true
        fi
    fi

    if [[ $(grep --count REDASH_DEFAULT_PROFILE_IMAGE "${redash_env}") -eq 0 ]]; then
        log "Adding REDASH_DEFAULT_PROFILE_IMAGE to ${redash_env}"
        cat >> "${redash_env}" <<EOF

# Change the icon in the bottom left corner of Redash to a custom icon
# To set, put a png (200x200 pixels) in /opt/starfish/gui/img, and enable
# the REDASH_DEFAULT_PROFILE_IMAGE variable. This should be set relative
# to the /opt/starfish/gui directory.
# Restart sf-redash10.target to enable.
#REDASH_DEFAULT_PROFILE_IMAGE=/img/logo_icon.png
EOF
    fi

    if [[ "${restart}" == true ]]; then
        restart_redash "${redash_version}"
    fi
}

install_redash() {
    local redash_version="$1"
    local redis_port="$2"
    local pg_port="$3"
    local gunicorn_listen_address="$4"
    local overwrite="$5"
    local install_pkgs="$6"
    local redash_host="$7"
    local restart_redash_if_already_installed="${8:-false}"

    local redis_pwd postgres_pwd cookie_secret redash_date_format redash_dir redash_pg_dir redash_redis_dir
    local redash_env_file redash_pg_cluster_name pg_version gunicorn_config_file

    redash_dir="$(get_redash_dir "${redash_version}")"
    redash_pg_cluster_name="$(get_redash_pg_cluster_name "${redash_version}")"
    redash_pg_dir="$(get_redash_pg_dir "${redash_version}")"
    redash_redis_dir="$(get_redash_redis_dir "${redash_version}")"
    redash_env_file="$(get_redash_env_filename "${redash_version}")"

    if dir_exists "${redash_pg_dir}"; then
        update_redash_env_settings "${redash_host}" "${redash_version}"
        if [[ "${overwrite}" == false ]]; then
            log "Redash is already installed in ${redash_dir}"
            if [[ "${restart_redash_if_already_installed}" == true ]]; then
                restart_redash "${redash_version}"
            fi
            exit 0
        fi
    fi

    pg_version="$(get_redash_pg_version "${redash_version}")"
    if [[ "${install_pkgs}" == true ]]; then
        install_prerequisites

        # sf-redash contains "configure_redash" which works with Redash 1, 8 and 10
        # redash/redashX packages contains Redash 1/Redash X open source software

        # if Starfish repo not yet available - install redash only, without sf-redash with configure_redash
        # if Starfish repo available - install sf-redash and redash (sf-redash depends of redash now but it wasn't the
        # case in old versions of packages)
        if pkg_available sf-redash; then
            pkg_install sf-redash
        fi
        # need to have PGDG repos to install redash10 package as it requires PostgreSQL 13 package
        install_redash_postgresql "${pg_version}" "${LOG_FILE}"
        pkg_install "$(get_redash_pkg_name "${redash_version}")"
    else
        # make sure that starfish-misc repo is always configured, even if not installing packages
        add_starfish_misc_repo
    fi

    redis_pwd="$(generate_secret_key 24)"
    postgres_pwd="$(generate_secret_key 24)"
    cookie_secret="$(generate_secret_key 24)"
    configure_redash_postgresql "${pg_port}" "${redash_pg_dir}" "${redash_pg_cluster_name}" "${postgres_pwd}" "${LOG_FILE}" "${overwrite}" "${pg_version}"
    install_redash_redis "${redis_port}" "${redash_redis_dir}" "${redis_pwd}" "${LOG_FILE}" "${install_pkgs}"

    if [[ "${redash_version}" -eq 10 ]]; then
        gunicorn_config_file="gunicorn.conf.py"
    else
        gunicorn_config_file="gunicorn.conf"
    fi
    sed -i "s/^bind.*$/bind='${gunicorn_listen_address}'/" "${redash_dir}/${gunicorn_config_file}"
    redash_date_format="$(moment_date_format_from_locale)"
    cat > "${redash_env_file}" <<EOF
PYTHONUNBUFFERED=0
REDASH_COOKIE_SECRET=${cookie_secret}
REDASH_DATABASE_URL=postgresql://${REDASH_POSTGRES_USERNAME}:${postgres_pwd}@localhost:${pg_port}/${REDASH_POSTGRES_DB_NAME}
REDASH_DATE_FORMAT="${redash_date_format}"
REDASH_HOST=https://${redash_host}/redash
REDASH_LOG_LEVEL=INFO
REDASH_MAIL_DEFAULT_SENDER=starfish@${redash_host}
REDASH_REDIS_URL=redis://redash:${redis_pwd}@localhost:${redis_port}/0
REDASH_VERSION_CHECK=false
REDASH_ENFORCE_CSRF=true
REDASH_ENFORCE_CSRF_FOR_AUTH_USERS=true
REDASH_CSRF_TIME_LIMIT=999888000
#use the following to configure sending mail server if you want Redash to send out e-mails
#by default SMTP server on localhost:25 is used
#REDASH_MAIL_PASSWORD=
#REDASH_MAIL_PORT=25
#REDASH_MAIL_SERVER=localhost
#REDASH_MAIL_USERNAME=
#REDASH_MAIL_USE_SSL=false
#REDASH_MAIL_USE_TLS=false

#Redash JWT
REDASH_JWT_LOGIN_ENABLED=true
REDASH_JWT_AUTH_PUBLIC_CERTS_URL="file:///opt/starfish/etc/jwtRS256.pem"
REDASH_JWT_AUTH_ISSUER="sf-auth"
REDASH_JWT_AUTH_AUDIENCE="Starfish"
REDASH_JWT_AUTH_COOKIE_NAME="sf-jwt-token"

#Enable "Custom" Chart Type
REDASH_FEATURE_ALLOW_CUSTOM_JS_VISUALIZATIONS=true

#Make invitation links sent by email valid for 1 hour
REDASH_INVITATION_TOKEN_MAX_AGE=3600

# Change the icon in the bottom left corner of Redash to a custom icon
# To set, put a png (200x200 pixels) in /opt/starfish/gui/img, and enable
# the REDASH_DEFAULT_PROFILE_IMAGE variable. This should be set relative
# to the /opt/starfish/gui directory.
# Restart sf-redash10.target to enable.
#REDASH_DEFAULT_PROFILE_IMAGE=/img/logo_icon.png
EOF

# REDASH_JWT_AUTH_AUDIENCE, REDASH_JWT_AUTH_ISSUER have to match values used by sf-auth service

    chown -R redash "${redash_env_file}"
    chmod -R go= "${redash_env_file}"
}

_systemd_check_running_services() {
    local services="$1"
    shift 1

    test "$(systemd_unit_running_members_count "${services}.target")" "$@"
}

_supervisord_check_running_services() {
    local services="$1"
    shift 1

    test "$(supervisorctl status | grep "${services}" | grep --count RUNNING)" "$@"
}

_check_running_services() {
    local services="$1"
    shift

    run_func_for_service_manager "${services}.target" _systemd_check_running_services "${services}" _supervisord_check_running_services "${services}" "$@"
}

_systemd_check_not_running_services() {
    local services="$1"
    shift 1

    test "$(systemd_unit_stopped_members_count "${services}.target")" "$@"
}

_supervisord_check_not_running_services() {
    local services="$1"
    shift 1

    test "$(supervisorctl status | grep "${services}" | grep --invert-match --count RUNNING)" "$@"
}

_check_not_running_services() {
    local services="$1"
    shift

    run_func_for_service_manager "${services}.target" _systemd_check_not_running_services "${services}" _supervisord_check_not_running_services "${services}" "$@"
}

create_redash_db_tables() {
    local redash_version="$1"

    run_redash_manage "${redash_version}" "Creating Redash DB tables" database create_tables
}

configure_redash() {
    local configure_redash="${STARFISH_BIN_DIR}/configure_redash"

    if ! file_exists "${configure_redash}"; then
        log "${configure_redash} not found. Run install-Starfish.sh"
        return 1
    fi

    log "Configuring Redash"

    "${configure_redash}" "$@"
}

create_reports_db_user() {
    local password
    local pgagent_schema="pgagent"

    log "Creating reports DB user (${REPORTS_USER_IN_SF_CLUSTER})"

    password="$(get_ini_property "${STARFISH_ETC_SECRETS_INI}" reports reports_user_password)"
    if [[ "${password}" == "" ]]; then
        password="$(generate_secret_key 24)"
        set_ini_property "${STARFISH_ETC_SECRETS_INI}" reports reports_user_password "${password}"
    fi

    ensure_postgresql_user_exists "${STARFISH_PG_PORT}" "${REPORTS_USER_IN_SF_CLUSTER}" "${password}" nosuperuser
    # Here, we only grant permissions to postgres.pgagent schema (so we connect to postgres DB).
    # sf schemas in starfish DB may not exist when this script is run
    # thus granting permissions to sf schemas is done in "blocking migrations" in pgloader.
    (
        psql_as_postgres "${STARFISH_PG_PORT}" postgres <<EOF
GRANT USAGE ON SCHEMA public, ${pgagent_schema} TO ${REPORTS_USER_IN_SF_CLUSTER};
GRANT SELECT ON ALL TABLES IN SCHEMA public, ${pgagent_schema} TO ${REPORTS_USER_IN_SF_CLUSTER};
-- make all new (not existing now) tables accessible by the user when they are created
ALTER DEFAULT PRIVILEGES IN SCHEMA public, ${pgagent_schema} GRANT SELECT ON TABLES TO ${REPORTS_USER_IN_SF_CLUSTER};
EOF
        psql_as_postgres "${STARFISH_PG_PORT}" "${STARFISH_PG_DB_NAME}" <<EOF
CREATE SCHEMA IF NOT EXISTS AUTHORIZATION ${REPORTS_USER_IN_SF_CLUSTER};
EOF
    ) |& tee --append "${LOG_FILE}"
}

reset_redash_datasources() {
    configure_redash reset_datasources
}

main() {
    fail_if_internal_script_run_directly

    local email="${REDASH_DEFAULT_EMAIL}"
    local password="${REDASH_DEFAULT_PASSWORD}"
    local redash_version=10

    local redis_port pg_port gunicorn_listen_address help_message
    local args=( "$@" )

    local redash_host
    local configure="true"
    local install_pkgs="true"
    local overwrite="false"
    local start="true"
    local update_env_only="false"
    local create_reports_db_user_only="false"
    local reset_reports_db_user_only="false"

    redash_host="$(get_hostname)"

    while [ $# -gt 0 ]; do
        case $1 in
        --log-file)
            shift
            LOG_FILE="$1"
            ;;
        --email)
            shift
            email="$1"
            ;;
        --password)
            shift
            password="$1"
            ;;
        --redash-version)
            shift
            redash_version="$1"
            ;;
        --pg-port)
            shift
            # backwards compat in upgrade
            if [[ "$1" -eq "${REDASH1_PG_PORT}" ]]; then
                redash_version=1
            elif [[ "$1" -eq "${REDASH8_PG_PORT}" ]]; then
                redash_version=8
            elif [[ "$1" -eq "${REDASH10_PG_PORT}" ]]; then
                redash_version=10
            else
                log "Unknown PG port value $1"
                exit 2
            fi
            ;;
        --redash-host)
            shift
            redash_host="$1"
            ;;
        --no-install-pkgs)
            # may be called from migrate post-install script in RPM/DEB
            # this means we should have all packages installed and we can't use apt/yum as it'll result in a deadlock
            install_pkgs="false"
            ;;
        --no-configure)
            configure="false"
            ;;
        --overwrite)
            overwrite="true"
            ;;
        --no-overwrite)
            overwrite="false"
            ;;
        --no-start)
            start="false"
            ;;
        --update-env-only)
            # override any other option, calls update_redash_env_settings and exits
            update_env_only="true"
            ;;
        --create-reports-db-user-only)
            # override any other option, calls create_reports_db_user and exits
            create_reports_db_user_only="true"
            ;;
        --reset-reports-db-user-only)
            # override any other option, calls create_reports_db_user and exits
            reset_reports_db_user_only="true"
            ;;
        *)
            break
            ;;
        esac
        shift
    done
    export LOG_FILE
    log "Script called with arguments: $0 " ${args[@]+"${args[@]}"}

    if (( $# != 0 )); then
        log "Unknown option $*"
        exit 2
    fi

    if [[ "${update_env_only}" == true ]]; then
        update_redash_env_settings "${redash_host}" "${redash_version}"
        return
    fi

    if [[ "${create_reports_db_user_only}" == true ]]; then
        create_reports_db_user
        return
    fi

    if [[ "${reset_reports_db_user_only}" == true ]]; then
        create_reports_db_user
        reset_redash_datasources
        return
    fi

    redis_port="$(get_redash_redis_port "${redash_version}")"
    pg_port="$(get_redash_pg_port "${redash_version}")"
    gunicorn_listen_address="$(get_redash_gunicorn_address "${redash_version}")"

    if ! is_correct_format_email_address "${email}"; then
        log "Given email address '${email}' is incorrect. Please, type in correct email address."
        exit 1
    fi
    execute_on_production_environment fail_if_not_root

    set_convenient_umask

    install_redash "${redash_version}" "${redis_port}" "${pg_port}" "${gunicorn_listen_address}" "${overwrite}" "${install_pkgs}" "${redash_host}" "${start}"
    if [[ "${start}" == true ]]; then
        restart_redash "${redash_version}"
        create_redash_db_tables "${redash_version}"
        if [[ "${configure}" == true ]]; then
            # Setup Redash admin user
            # This step wither create new admin user with given email and password
            # or reset existing admin user password (if that user already exists)
            help_message=$(configure_redash --help)
            if [[ $help_message == *setup* ]]; then
                configure_redash setup --save-credentials --yes --user "${email}" --password "${password}" --url "${gunicorn_listen_address}"
                # Setup Redash reports
                configure_redash --yes
            else
                # This code is for testing purpose only. Production code uses `setup` command above.
                configure_redash default --yes --user "${email}" --password "${password}" --url "${gunicorn_listen_address}"
            fi
        fi
    fi
}

sourced() {
    [[ "${BASH_SOURCE[0]}" != "${0}" ]]
}

if ! sourced; then
    main "$@"
fi
