#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

_POSTGRESQL_SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
DB_THRESHOLD="${DB_THRESHOLD:-$((64 * 1024 * 1024 * 1024))}"
readonly _POSTGRESQL_SCRIPT_DIR DB_THRESHOLD

LOG_FILE="$(basename "${BASH_SOURCE[0]}" '.sh').log"

# shellcheck source=scripts/installation/_common_install.sh
source "${_POSTGRESQL_SCRIPT_DIR}/_common_install.sh"
# shellcheck source=scripts/installation/_ssl.sh
source "${_POSTGRESQL_SCRIPT_DIR}/_ssl.sh"
# shellcheck source=scripts/installation/_pg_configure.sh
source "${_POSTGRESQL_SCRIPT_DIR}/_pg_configure.sh"

readonly PG_LOCALE="${PG_LOCALE:-en_US.UTF-8}"

log_and_exit() {
    log "$@"
    exit 1
}

lang_from_pg_locale() {
    echo "${PG_LOCALE}" | cut -f 1 -d _ | cut -f 1 -d . | tr '[:upper:]' '[:lower:]'
}

_centos_install_postgresql_pkgs() {
    local pg_version="$1"
    local install_devel_pkg="$2"
    local major_version lang
    local -a pkgs

    major_version=$(centos_get_major_version)

    if [[ "${pg_version}" == 15 ]]; then
        pkgs=(
            pg_repack_15
            pgagent_15
            postgresql15
            postgresql15-contrib
            postgresql15-debuginfo
            postgresql15-libs
            postgresql15-llvmjit
            postgresql15-plpython3
            postgresql15-server
            sf-pg-activity
        )
        if [[ "${install_devel_pkg}" == true ]]; then
            pkgs+=(postgresql15-devel)
        fi
    elif [[ "${pg_version}" == 13 ]]; then
        pkgs=(
            pg_repack_13
            pgagent_13
            postgresql13
            postgresql13-contrib
            postgresql13-debuginfo
            postgresql13-libs
            postgresql13-llvmjit
            postgresql13-plpython3
            postgresql13-server
            sf-pg-activity
        )
        if [[ "${major_version}" -ge 8 ]]; then
            # under CentOS 8 without glibc-langpack-en initdb will fail with
            # `initdb: error: invalid locale name "en_US.UTF-8"`
            # if PG_LOCALE is set to en_US.UTF-8
            lang=$(lang_from_pg_locale)
            if [[ "${lang}" != c ]]; then
                pkgs+=("glibc-langpack-${lang}")
            fi

            # postgresql13-plpython3-debuginfo not available on CentOS 7
            pkgs+=(postgresql13-plpython3-debuginfo postgresql13-server-debuginfo)
        fi

        if [[ "${install_devel_pkg}" == true ]]; then
            pkgs+=(postgresql13-devel)
        fi
    elif [[ "${pg_version}" == 9.6 ]]; then
        pkgs=(
            pgagent_96
            pg_repack96
            postgresql96
            postgresql96-contrib
            # postgresql96-debuginfo - not available in archived repo for CentOS 8
            postgresql96-libs
            postgresql96-server
            sf-pg-activity
        )
        if [[ "${install_devel_pkg}" == true ]]; then
            pkgs+=(postgresql96-devel)
        fi
    else
        log_and_exit "Unsupported PG version ${pg_version}"
    fi

    yum --assumeyes install "${pkgs[@]}"
}

centos_update_to_newest_pgdg_repo() {
    local pgdg_redhat_repo_url

    # Repo URL comes from https://www.postgresql.org/download/linux/redhat/
    pgdg_redhat_repo_url="https://download.postgresql.org/pub/repos/yum/reporpms/EL-$(centos_get_major_version)-x86_64/pgdg-redhat-repo-latest.noarch.rpm"
    if centos_pkg_installed pgdg-redhat-repo; then
        # remove old package before installing new one is the only way I've found to reinstall repo file
        # even using "reinstall" option does not restore contents of /etc/yum.repos.d/pgdg-redhat-all.repo
        yum remove --assumeyes pgdg-redhat-repo
    fi
    yum --assumeyes install "${pgdg_redhat_repo_url}"
}

centos_install_postgresql() {
    local pg_version="$1"
    local install_devel_pkg="$2"
    local -a extra_repos

    # augeas is needed for _pg_configure to edit PostgreSQL config files
    centos_pkg_install augeas
    if ! is_offline_installation; then
        centos_install_epel # needed for pgagent that depends on psutil
        centos_update_to_newest_pgdg_repo
        if [[ "${pg_version}" == 15 ]]; then
            extra_repos=(pgdg15 pgdg15-debuginfo) # PG 15 is not used in product and that's why it's not enabled by default
        fi
        centos_enable_pgdg_repos "${extra_repos[@]+"${extra_repos[@]}"}" # CentOS 7 still comes with bash < 4.4
        if [[ "${pg_version}" == 9.6 ]]; then
            centos_enable_archived_repo_for_pg_96
        fi
    fi
    # Rocky Linux Docker image doesn't have diff installed
    centos_pkg_install diffutils
    _centos_install_postgresql_pkgs "${pg_version}" "${install_devel_pkg}"
}

ubuntu_install_postgresql() {
    local pg_version="$1"
    local install_devel_pkg="$2"
    local keyring=/usr/share/keyrings/pgdg.gpg
    local -a pkgs
    ubuntu_distro=$(map_to_ubuntu_distro)

    if ! is_offline_installation; then
        ubuntu_add_signing_key https://apt.postgresql.org/pub/repos/apt/ACCC4CF8.asc "${keyring}" B97B0AFCAA1A47F044F244A07FCC7D46ACCC4CF8
        ubuntu_add_apt_source /etc/apt/sources.list.d/pgdg.list "${keyring}" http://apt.postgresql.org/pub/repos/apt/ main "${ubuntu_distro}-pgdg"
    fi

    apt-get update
    # postgresql-common has /etc/postgresql-common directory
    apt-get install -y postgresql-common
    # don't create main cluster automatically
    mkdir --parents /etc/postgresql-common/createcluster.d
    echo "create_main_cluster = false" >/etc/postgresql-common/createcluster.d/starfish.conf

    # augeas is needed for _pg_configure to edit PostgreSQL config files
    pkgs=(pgagent augeas-tools sf-pg-activity)
    if [[ "${install_devel_pkg}" == true ]]; then
        pkgs+=(libpq-dev)
    fi

    if [[ "${pg_version}" == 15 ]]; then
        pkgs+=(
            postgresql-15
            postgresql-15-dbgsym
            postgresql-15-repack
            postgresql-plpython3-15
            postgresql-plpython3-15-dbgsym
        )
    elif [[ "${pg_version}" == 13 ]]; then
        pkgs+=(
            postgresql-13
            postgresql-13-dbgsym
            postgresql-13-repack
            postgresql-plpython3-13
            postgresql-plpython3-13-dbgsym
        )
    elif [[ "${pg_version}" == 9.6 ]]; then
        pkgs+=(
            postgresql-9.6
            postgresql-9.6-dbg
            postgresql-9.6-repack
            postgresql-contrib-9.6
        )
    else
        log_and_exit "Unsupported PG version ${pg_version}"
    fi

    apt-get install -y "${pkgs[@]}"

    if ! user_exists pgagent; then
        groupadd --system pgagent || true
        useradd --gid pgagent --system --shell /bin/false --comment "pgAgent Job Schedule" pgagent
    fi

    if [[ "${pg_version}" == 9.6 ]]; then
        # pgagent from PGDG does not support 9.6 anymore, but SQL files in all the PG versions
        # are just symlinks to 14
        ln --symbolic --force --verbose --relative /usr/share/postgresql/14/extension/pgagent* --target-directory=/usr/share/postgresql/9.6/extension
    fi
}

ensure_postgresql_database_exists() {
    local port="$1"
    local db_name="$2"

    if ! postgresql_database_exists "${port}" "${db_name}"; then
        # database can't be created inside a transaction
        psql_as_postgres "${port}" postgres "CREATE DATABASE ${db_name};"
    fi
}

install_postgresql() {
    local pg_version="$1"

    log "Installing PostgreSQL ${pg_version}"

    run_func_for_distro centos_install_postgresql ubuntu_install_postgresql "$@"
}

centos_initdb() {
    local pg_version="$1"
    local pg_cluster_name="$2"
    # initdb under CentOS does not care about the port number - it only sets up files in a directory
    # local unused_port="$3"
    local db_path="$4"
    local data_checksums="$5"
    local postgresql_setup_script service_name description initdb_log unit_file_path

    if [[ "${pg_version}" = "15" ]]; then
        postgresql_setup_script="$(centos_get_pgbin_dir "${pg_version}")/postgresql-15-setup"
    elif [[ "${pg_version}" = "13" ]]; then
        postgresql_setup_script="$(centos_get_pgbin_dir "${pg_version}")/postgresql-13-setup"
    elif [[ "${pg_version}" = "9.6" ]]; then
        postgresql_setup_script="$(centos_get_pgbin_dir "${pg_version}")/postgresql96-setup"
    else
        log_and_exit "Unsupported PG version ${pg_version}"
    fi

    service_name=$(get_pg_service_name "${pg_cluster_name}" "${pg_version}")
    description=$(get_pg_service_description "${pg_cluster_name}")
    unit_file_path="/etc/systemd/system/${service_name}.service"

    # Centos PG package, unlike the Ubuntu one, comes with a single unit file instead of a proper template one.
    # Because of this, there is really no way to multi-instance PG without either using the deprecated .include
    # instruction or copying the unit file and possibly losing any future upstream changes.
    # Creating a new PG service instance by symlinking the package unit file and enhancing it with drop-ins doesn't
    # work, because newer versions of systemd refuse to operate on linked unit files. Hardlinking is no good either,
    # because upgrading PG doesn't overwrite the package file, but deletes+creates it (destroying the linkage).
    # We shall use the package unit file (postgresql-${pg_version}.service) for our main (SF) PG instance and copy
    # it for other instances, e.g. Redash.
    if ! file_exists "${unit_file_path}" && [[ "${service_name}" != "postgresql-${pg_version}" ]]; then
        cp "/usr/lib/systemd/system/postgresql-${pg_version}.service" "${unit_file_path}"
    fi
    mkdir --parents "/etc/systemd/system/${service_name}.service.d"
    cp "${_POSTGRESQL_SCRIPT_DIR}/pg-drop-in-centos_01_wants-network-online_kill-mode-process.conf" "/etc/systemd/system/${service_name}.service.d/01_wants-network-online_kill-mode-process.conf"
    sed -i \
        -e "s|<%= PG_VERSION %>|${pg_version}|g" \
        -e "s|<%= DESCRIPTION %>|${description}|g" \
        -e "s|<%= DB_PATH %>|${db_path}|g" "/etc/systemd/system/${service_name}.service.d/01_wants-network-online_kill-mode-process.conf"
    # ask systemd to load new ${service_name}.service file
    systemctl daemon-reload
    # make sure systemd starts up the DB upon bootup
    if [[ ! $(systemctl is-enabled "${service_name}") =~ .*enabled.* ]]; then
        systemctl enable "${service_name}"
    fi
    if [[ "${data_checksums}" = "true" ]]; then
        data_checksums="--data-checksums"
    else
        data_checksums=""
    fi
    if ! PGSETUP_INITDB_OPTIONS="--locale=${PG_LOCALE} ${data_checksums}" ${postgresql_setup_script} initdb "${service_name}"; then
        initdb_log=$(centos_get_default_pgdata "${pg_version}" initdb.log)
        if [[ -f "${initdb_log}" ]]; then
            cat "${initdb_log}" >&2
            exit 1
        fi
    fi

    dd if=/dev/zero of="${db_path}/.delete_me_in_emergency" bs=1M count=100
}

ubuntu_initdb() {
    local pg_version="$1"
    local pg_cluster_name="$2"
    local port="$3"
    local db_path="$4"
    local data_checksums="$5"
    local -a extra_args
    if [[ "${data_checksums}" = "true" ]]; then
        extra_args+=(--data-checksums)
    fi

    service_name=$(get_pg_service_name "${pg_cluster_name}" "${pg_version}")
    mkdir -p "/etc/systemd/system/${service_name}.service.d"
    cp "${_POSTGRESQL_SCRIPT_DIR}/pg-drop-in-ubuntu_01_wants-network-online.conf" "/etc/systemd/system/${service_name}.service.d/01_wants-network-online.conf"
    # ask systemd to load new ${service_name}.service file
    systemctl daemon-reload
    # if there was a cluster of the same name - drop it
    pg_dropcluster --stop "${pg_version}" "${pg_cluster_name}" 2>/dev/null || true
    if [[ "${PG_LOCALE}" != C ]]; then
        # otherwise pg_createcluster will fail with
        # initdb: error: invalid locale name "en_US.UTF-8"
        locale-gen "${PG_LOCALE}" || true
    fi
    pg_createcluster --locale="${PG_LOCALE}" "${pg_version}" "${pg_cluster_name}" -d "${db_path}" -p "${port}" -- "${extra_args[@]+"${extra_args[@]}"}"

    dd if=/dev/zero of="${db_path}/.delete_me_in_emergency" bs=1M count=100
}

initdb() {
    local pg_version="$1"
    local pg_cluster_name="$2"
    local port="$3"
    local db_path="$4"
    local data_checksums="$5"
    local drop_existing_cluster="$6"
    local real_db_path

    log "Stopping PostgreSQL ${pg_version} cluster '${pg_cluster_name}' on port ${port} if there is any"
    stop_postgresql_service "${pg_version}" "${pg_cluster_name}" "${port}" 2>/dev/null || true
    if symlink_exists "${db_path}" && ! path_exists "${db_path}"; then
        log "${db_path} is a broken symlink to $(readlink --canonicalize-missing "${db_path}"), removing"
        # if db_path is a symlink and the target doesn't exist - remove the symlink, it's not useful anymore
        rm -f -- "${db_path}"
    fi
    if path_exists "${db_path}"; then
        real_db_path="$(readlink --canonicalize-existing --verbose "${db_path}")"
        if [[ "${real_db_path}" != "${db_path}" ]]; then
            log "${db_path} is a symlink to ${real_db_path}, following"
            db_path="${real_db_path}"
        fi
        if is_dir_empty "${db_path}"; then
            chown postgres:postgres "${db_path}"
            chmod 700 "${db_path}"
        elif [[ "${drop_existing_cluster}" = "prompt" ]]; then
            prompt_to_remove_path "${db_path}"
        elif [[ "${drop_existing_cluster}" = "true" ]]; then
            log "removing ${db_path}"
            rm -rf -- "${db_path}"
        fi
    fi
    log "Creating PostgreSQL ${pg_version} cluster '${pg_cluster_name}' in ${db_path}"
    run_func_for_distro centos_initdb ubuntu_initdb "$@"
}

create_user_and_db_in_postgresql() {
    local port="$1"
    local username="$2"
    local password="$3"
    local db_name="$4"

    ensure_postgresql_user_exists "${port}" "${username}" "${password}" superuser
    ensure_postgresql_database_exists "${port}" "${db_name}"
}

enable_pgadmin() {
    local port="$1"
    local db_name="$2"

    psql_as_postgres "${port}" postgres <<EOF
CREATE EXTENSION IF NOT EXISTS adminpack;
EOF
}

enable_pgagent() {
    local port="$1"
    local db_name="$2"
    local db_config_template="$3"

    if [[ "${db_config_template}" == redash ]]; then
        # no need to install pgagent on the Redash DB as it's not used by Redash
        return
    fi

    psql_as_postgres "${port}" postgres <<EOF
CREATE EXTENSION IF NOT EXISTS pgagent;

DO
\$body\$
BEGIN
    IF NOT EXISTS (SELECT * FROM pg_catalog.pg_user WHERE usename = 'pgagent') THEN
        CREATE ROLE pgagent LOGIN NOSUPERUSER INHERIT NOCREATEDB NOCREATEROLE NOREPLICATION;
    END IF;
END
\$body\$;

GRANT ALL ON SCHEMA pgagent TO pgagent;
GRANT ALL ON ALL TABLES IN SCHEMA pgagent TO pgagent;
GRANT ALL ON ALL SEQUENCES IN SCHEMA pgagent TO pgagent;
GRANT ALL ON ALL FUNCTIONS IN SCHEMA pgagent TO pgagent;
EOF

    psql_as_postgres "${port}" "${db_name}" <<EOF
GRANT CONNECT ON DATABASE ${db_name} TO pgagent;
EOF
}

copy_ctl_scripts() {
    local pg_version="$1"
    local pg_cluster_name="$2"
    local port="$3"
    local db_path="$4"

    cp --force "${_POSTGRESQL_SCRIPT_DIR}/_pg_common.sh" "${_POSTGRESQL_SCRIPT_DIR}/_utils.sh" "${db_path}"
    cat <<EOF >"${db_path}/_params.sh"
# no-license-check
readonly PG_VERSION="${pg_version}"
readonly PG_CLUSTER_NAME="${pg_cluster_name}"
readonly PORT="${port}"
EOF
    cat <<'EOF' >"${db_path}/ctl.sh"
#!/usr/bin/env bash
# no-license-check

set -euo pipefail

readonly SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

source "${SCRIPT_DIR}/_pg_common.sh"
source "${SCRIPT_DIR}/_params.sh"

usage() {
    >&2 echo "Usage: $0 (start|stop|restart|reload)"
    exit 1
}

if [[ $# -lt 1 ]]; then
    usage
fi

case $1 in
start)
    start_postgresql_service "${PG_VERSION}" "${PG_CLUSTER_NAME}" "${PORT}"
    ;;
stop)
    stop_postgresql_service "${PG_VERSION}" "${PG_CLUSTER_NAME}" "${PORT}"
    ;;
restart)
    restart_postgresql_service "${PG_VERSION}" "${PG_CLUSTER_NAME}" "${PORT}"
    ;;
reload)
    reload_postgresql_service "${PG_VERSION}" "${PG_CLUSTER_NAME}" "${PORT}"
    ;;
*)
    usage
    ;;
esac
EOF
    chmod 755 "${db_path}/ctl.sh"
}

configure_pgbackrest() {
    local stanza="$1"
    local db_path="$2"
    local backup_path="$3"
    local pg_version="$4"
    local pg_cluster_name="$5"
    local port="$6"
    local retention="$7"

    local pg_config_dir dereferenced_db_path postgresql_config_file
    local pgbackrest_config_file="/etc/pgbackrest.conf"
    local sf_tmp_dir="${SFHOME}/tmp"
    local pgbackrest_tmp_dir="${sf_tmp_dir}/pgbackrest"
    local backup_processes_cnt="$(($(nproc --all) / 4 == 0 ? 1 : $(nproc --all) / 4))"
    local pgbackrest_spool_path="${STARFISH_PG_DIR}/pgbackrest_queue"

    pg_config_dir="$(get_pgconfig_dir "${db_path}" "${pg_version}" "${pg_cluster_name}")"
    postgresql_config_file="$(get_pg_local_conf_file "${pg_config_dir}")"

    mkdir --parents "${backup_path}"
    chmod 750 "${backup_path}"
    chown postgres:postgres "${backup_path}"

    # workaround for https://github.com/pgbackrest/pgbackrest/issues/1312
    mkdir --parents "${pgbackrest_tmp_dir}"
    chmod 1777 "${sf_tmp_dir}"
    chmod 1777 "${pgbackrest_tmp_dir}"

    mkdir --parents "${pgbackrest_spool_path}"
    chmod 750 "${pgbackrest_spool_path}"
    chown postgres:postgres "${pgbackrest_spool_path}"

    dereferenced_db_path="$(dereference_db_path "${db_path}")"
    cat <<EOF >"${pgbackrest_config_file}"
[global]
repo-path=${backup_path}
# requires space for 3 full backups
retention-full=${retention}
process-max=${backup_processes_cnt}
# automatically call pg_stop_backup() if previous backup failed
stop-auto=y
start-fast=y
lock-path=${pgbackrest_tmp_dir}
archive-async=y
spool-path=${pgbackrest_spool_path}

[${stanza}]
db-path=${dereferenced_db_path}
EOF

    set_pg_config_params "${postgresql_config_file}" \
        archive_command "'pgbackrest --stanza=${stanza} archive-push %p'" \
        archive_mode "on" \
        max_wal_senders "3" \
        wal_level "hot_standby"

    # restart with new configuration
    stop_postgresql_service "${pg_version}" "${pg_cluster_name}" "${port}"

    # stanza-create command no longer supports "--force" option so remove it manually to ensure it doesn't exist
    echo "Make sure stanza ${stanza} does not exist"
    run_as_postgres "pgbackrest --stanza=${stanza} --log-level-console=info stanza-delete" || true
    find "${backup_path}" -mindepth 1 -delete || true
    find "${pgbackrest_spool_path}" -mindepth 1 -delete || true

    start_postgresql_service "${pg_version}" "${pg_cluster_name}" "${port}"

    run_as_postgres "pgbackrest --stanza=${stanza} --log-level-console=info stanza-create"
    run_as_postgres "pgbackrest --stanza=${stanza} --log-level-console=info check"
}

configure_backup() {
    local db_path="$1"
    local backup_path="$2"
    local pg_version="$3"
    local pg_cluster_name="$4"
    local port="$5"
    local retention="$6"
    local install_pkgs="$7"
    local stanza

    if [[ "${pg_cluster_name}" == "main" ]]; then
        stanza="starfish"
    else
        stanza="${pg_cluster_name}"
    fi

    if [[ "${install_pkgs}" = "true" ]]; then
        pkg_install pgbackrest
    fi
    configure_pgbackrest "${stanza}" "${db_path}" "${backup_path}" "${pg_version}" "${pg_cluster_name}" "${port}" "${retention}"
}

main() {
    fail_if_internal_script_run_directly

    local listen_address="127.0.0.1"
    local port="${STARFISH_PG_PORT}"
    local username="starfish"
    local password="starfish"
    local backup_only="false"
    local drop_existing_cluster="prompt"
    local db_name="${STARFISH_PG_DB_NAME}"
    local pg_version="${STARFISH_PG_VERSION}"
    local pg_cluster_name="${STARFISH_PG_CLUSTER_NAME}"
    local db_config_template="${STARFISH_PG_CONFIG_TEMPLATE}"
    local db_path=""
    local install_pkgs="true"
    local install_devel_pkg="false"
    local create_cluster="true"
    local data_checksums="true"
    local retention="2"
    local db_storage_type backup_path pg_config_dir
    local args=("$@")

    db_storage_type="$(get_default_db_storage_type)"
    backup_path="$(get_default_pgbackup_path)"

    while [[ $# -gt 0 ]]; do
        case $1 in
        --log-file)
            shift
            LOG_FILE="$1"
            ;;
        --listen-address)
            shift
            listen_address="$1"
            ;;
        --port)
            shift
            port="$1"
            ;;
        --username)
            shift
            username="$1"
            ;;
        --password)
            shift
            password="$1"
            ;;
        --db-name)
            shift
            db_name="$1"
            ;;
        --db-path)
            shift
            db_path="$1"
            ;;
        --db-storage-type)
            shift
            db_storage_type="$1"
            ;;
        --db-config-template)
            shift
            db_config_template="$1"
            ;;
        --drop-existing-cluster)
            drop_existing_cluster="true"
            ;;
        --backup-path)
            shift
            backup_path="$1"
            ;;
        --backup-only)
            backup_only="true"
            ;;
        --no-backup)
            backup_path=""
            ;;
        --pg-version)
            shift
            pg_version="$1"
            ;;
        --pg-cluster-name)
            shift
            pg_cluster_name="$1"
            ;;
        --no-install-pkgs)
            # may be called from post-install script in RPM/DEB
            # this means we should have all packages installed and we can't use apt/yum as it'll result in a deadlock
            install_pkgs="false"
            ;;
        --install-devel-pkg)
            # devel packages are needed to compile libpq in development environment
            install_devel_pkg="true"
            ;;
        --create-cluster)
            create_cluster="true"
            ;;
        --no-create-cluster)
            create_cluster="false"
            ;;
        --no-data-checksums)
            data_checksums="false"
            ;;
        --retention)
            shift
            retention="$1"
            ;;
        *)
            break
            ;;
        esac
        shift
    done
    export LOG_FILE
    log "Script called with arguments: $0 " ${args[@]+"${args[@]}"}

    if (($# != 0)); then
        log "Unknown parameters specified: $*"
        exit 2
    fi
    if [[ -z "${db_path}" ]]; then
        db_path="$(get_default_pgdata "${pg_version}" "${pg_cluster_name}")"
    fi
    if [[ "${backup_only}" == "true" ]]; then
        if ! dir_exists "${db_path}"; then
            log_and_exit "Cannot configure backup, because db path ${db_path} does not exist."
        fi
        pg_version=$(postgresql_version "${db_path}")
    fi

    fail_if_not_root
    fail_if_selinux_enforcing

    if [[ "${backup_only}" == "true" && -z "${backup_path}" ]]; then
        log_and_exit "Cannot run in backup-only mode with no backup path specified."
    fi

    if [[ "${backup_only}" == "false" ]]; then
        fail_if_not_enough_space_for_db "${db_path}" "${DB_THRESHOLD}"

        set_convenient_umask

        pg_config_dir="$(get_pgconfig_dir "${db_path}" "${pg_version}" "${pg_cluster_name}")"
        if [[ "${install_pkgs}" == "true" ]]; then
            install_prerequisites
            install_postgresql "${pg_version}" "${install_devel_pkg}"
        fi

        if [[ "${create_cluster}" == "true" ]]; then
            initdb "${pg_version}" "${pg_cluster_name}" "${port}" "${db_path}" "${data_checksums}" "${drop_existing_cluster}"
            configure_postgresql_initial "${listen_address}" "${port}" "${db_path}" "${pg_version}" "${pg_cluster_name}" "${db_storage_type}" "${db_config_template}"
            copy_ctl_scripts "${pg_version}" "${pg_cluster_name}" "${port}" "${db_path}"
            start_postgresql_service "${pg_version}" "${pg_cluster_name}" "${port}"
            create_user_and_db_in_postgresql "${port}" "${username}" "${password}" "${db_name}"
            enable_pgadmin "${port}" "${db_name}"
            enable_pgagent "${port}" "${db_name}" "${db_config_template}"
            enable_pg_stat_extension "${port}" "${db_name}"
        fi
    fi

    if ! empty_string "${backup_path}"; then
        fail_if_not_enough_space_for_db "${backup_path}" "${DB_THRESHOLD}"
        configure_backup "${db_path}" "${backup_path}" "${pg_version}" "${pg_cluster_name}" "${port}" "${retention}" "${install_pkgs}"
    fi
}

sourced() {
    [[ "${BASH_SOURCE[0]}" != "${0}" ]]
}

if ! sourced; then
    main "$@"
fi
