#!/usr/bin/env bash
#***********************************************************************************************************
#
# Starfish Storage Corporation ("Starfish") CONFIDENTIAL
# Unpublished Copyright (c) 2011 - present Starfish Storage Corporation, All Rights Reserved.
#
# NOTICE: This file and its contents (1) constitute Starfish's "External Code" under Starfish's most-recent
# Limited Software End-User License Agreement, and (2) is and remains the property of Starfish. The
# intellectual and technical concepts contained herein are proprietary to Starfish and may be covered by
# U.S. and/or foreign patents or patents in process, and are protected by trade secret or copyright law.
# Dissemination of this information or reproduction of this material is strictly forbidden unless prior
# written permission is obtained from Starfish. Access to the source code contained herein is hereby
# forbidden to anyone except (A) current Starfish employees, managers, or contractors who have executed
# confidentiality or nondisclosure agreements explicitly covering such access, and (B) licensees of
# Starfish's software.
#
# ANY REPRODUCTION, COPYING, MODIFICATION, DISTRIBUTION, PUBLIC PERFORMANCE, OR PUBLIC DISPLAY OF OR
# THROUGH USE OF THIS SOURCE CODE WITHOUT THE EXPRESS WRITTEN CONSENT OF STARFISH IS STRICTLY PROHIBITED
# AND IS IN VIOLATION OF APPLICABLE LAWS AND INTERNATIONAL TREATIES. THE RECEIPT OR POSSESSION OF THIS
# FILE OR ITS CONTENTS AND/OR RELATED INFORMATION DOES NOT CONVEY OR IMPLY ANY RIGHTS TO REPRODUCE,
# DISCLOSE, OR DISTRIBUTE ITS CONTENTS, OR TO MANUFACTURE, USE, OR SELL ANYTHING THAT IT MAY DESCRIBE, IN
# WHOLE OR IN PART.
#
# FOR U.S. GOVERNMENT CUSTOMERS REGARDING THIS DOCUMENTATION/SOFTWARE
#   These notices shall be marked on any reproduction of this data, in whole or in part.
#   NOTICE: Notwithstanding any other lease or license that may pertain to, or accompany the delivery of,
#   this computer software, the rights of the Government regarding its use, reproduction and disclosure are
#   as set forth in Section 52.227-19 of the FARS Computer Software-Restricted Rights clause.
#   RESTRICTED RIGHTS NOTICE: Use, duplication, or disclosure by the Government is subject to the
#   restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer
#   Software clause at DFARS 52.227-7013.
#
#***********************************************************************************************************

set -euo pipefail

_PG_CONFIGURE_SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
readonly _PG_CONFIGURE_SCRIPT_DIR

# shellcheck source=scripts/installation/_yum_repo.sh
source "${_PG_CONFIGURE_SCRIPT_DIR}/_yum_repo.sh"

_PG_APPLY_CONFIG_CHANGES=""
# from https://github.com/jberkus/annotated.conf/blob/master/annotated.csv
_PG_CONFIG_PARAMS_REQUIRE_RESTART=(
    allow_system_table_mods
    archive_mode
    autovacuum_freeze_max_age
    autovacuum_max_workers
    autovacuum_multixact_freeze_max_age
    bonjour
    bonjour_name
    cluster_name
    config_file
    data_directory
    dynamic_shared_memory_type
    event_source
    external_pid_file
    hba_file
    hot_standby
    huge_pages
    ident_file
    listen_addresses
    logging_collector
    max_connections
    max_files_per_process
    max_locks_per_transaction
    max_logical_replication_workers
    max_pred_locks_per_transaction
    max_prepared_transactions
    max_replication_slots
    max_wal_senders
    max_worker_processes
    old_snapshot_threshold
    port
    shared_buffers
    shared_preload_libraries
    superuser_reserved_connections
    track_activity_query_size
    track_commit_timestamp
    unix_socket_directories
    unix_socket_group
    unix_socket_permissions
    wal_buffers
    wal_level
    wal_log_hints
)

pg_check_changed_params() {
    local old_file="$1"
    local new_file="$2"
    local changed_param changed_params diff_text

    # check the differences between old and new config files
    # grep out filenames so that only changes are left
    # grep out the lines that didn't change
    diff_text="$(diff -u "${old_file}" "${new_file}" \
        | grep --invert-match --extended-regexp "(${old_file}|${new_file})" \
        | grep "^[+-]" || true)"

    log "\\n${old_file} will be changed:\\n\\n${diff_text}\\n"

    # cut only second byte and further from each line to remove + or - in the beginning of patch line
    # remove empty lines
    # cut only parameters' names
    # get unique parameters
    changed_params="$(echo "${diff_text}" \
        | cut --bytes=2- \
        | grep --invert-match "^$" \
        | cut --fields 1 --delimiter = \
        | sort --unique || true)"

    for changed_param in ${changed_params}; do
        # shellcheck disable=SC2199
        if array_contains "${_PG_CONFIG_PARAMS_REQUIRE_RESTART[@]}" "${changed_param}"; then
            log "will restart PostgreSQL because ${changed_param} changed"
            pg_applied_config_changes restart
            break
        else
            pg_applied_config_changes reload
        fi
    done
}

set_pg_config_params() {
    local file_path file_path_new temp_file key val date

    temp_file="$(mktemp .sf.postgresql.augeas.XXXXX)"
    file_path="$(readlink --canonicalize-missing "$1")"
    file_path_new="${file_path}.augnew"
    shift

    cat > "${temp_file}" <<EOF
set /augeas/load/Postgresql/lens Postgresql.lns
set /augeas/load/Postgresql/incl ${file_path}
load
EOF
    while [[ $# -gt 0 ]]; do
        key="$1"
        shift
        val="$1"
        shift
        echo "set /files${file_path}/${key}" "${val}" >> "${temp_file}"
    done
    echo "save" >> "${temp_file}"

    augtool --new --include="${_PG_CONFIGURE_SCRIPT_DIR}" --noautoload < "${temp_file}"
    rm -f -- "${temp_file}"

    if [[ -e "${file_path_new}" ]]; then
        # augeas changed the config file
        chown --reference="${file_path}" "${file_path_new}"
        chmod --reference="${file_path}" "${file_path_new}"

        date="$(date +"%Y%m%d_%H%M%S")"
        pg_check_changed_params "${file_path}" "${file_path_new}"
        mv -f "${file_path}" "${file_path}.${date}.bak"
        mv -f "${file_path_new}" "${file_path}"
    fi
}

get_pg_config_param() {
    local file_path
    file_path="$(readlink --canonicalize-missing "$1")"
    local key="$2"

    # augtool prints something like: /path/etc/01-service.ini/agent/ssl_certificate_file = /etc/sf-agent:localhost.localdomain.crt
    augtool --include "${_PG_CONFIGURE_SCRIPT_DIR}" --noautoload <<EOF | cut --only-delimited -f 2- -d = | tail -c +2
set /augeas/load/Postgresql/lens Postgresql.lns
set /augeas/load/Postgresql/incl ${file_path}
load
get /files${file_path}/${key}
EOF
}

pg_applied_config_changes() {
    local change_type="$1"

    if [[ "${change_type}" == "restart" ]]; then
        _PG_APPLY_CONFIG_CHANGES="restart"
    elif [[ "${change_type}" == "reload" ]] && [[ "${_PG_APPLY_CONFIG_CHANGES}" != "restart" ]]; then
        _PG_APPLY_CONFIG_CHANGES="reload"
    fi
}

fail_if_missing_config_file() {
    local local_config_file="$1"

    file_exists "${local_config_file}" || {
        log "Missing configuration file: ${local_config_file}"
        log "Did you install PostgreSQL with install-Starfish.sh script?"
        exit 1
    }
}

configure_hba() {
    local listen_address="$1"
    local pgconfig_dir="$2"
    local hba_config_file="${pgconfig_dir}/pg_hba.conf"
    local temp_file

    # if there is no commented out lines with samenet
    if ! grep --quiet --extended-regexp "^[^#].*samenet" "${hba_config_file}"; then
        temp_file="$(mktemp --tmpdir="${pgconfig_dir}" pg_hba.conf.sf.XXXXX)"
        cp --preserve=mode,ownership "${hba_config_file}" "${temp_file}"

        insert_after_line "${temp_file}" '# TYPE.*METHOD' 'hostssl all all samenet md5'
        insert_after_line "${temp_file}" 'hostssl all all samenet md5' 'hostnossl all all samehost md5'

        mv --force --backup=existing "${temp_file}" "${hba_config_file}"
        pg_applied_config_changes reload
    fi
}

configure_pg_to_use_local_config_file() {
    local pgconfig_dir="$1"
    local port="$2"
    local global_config_file="${pgconfig_dir}/postgresql.conf"
    local local_config_file include_line

    local_config_file="$(get_pg_local_conf_file "${pgconfig_dir}")"
    include_line="include '$(basename "${local_config_file}")'"

    chown --reference "${global_config_file}" "${local_config_file}"
    grep "${include_line}" "${global_config_file}" > /dev/null || {
        echo >> "${global_config_file}"
        echo "${include_line}" >> "${global_config_file}"
    }

    # set port in the global config file as Ubuntu pg_lsclusters gets "port" from this file
    set_pg_config_params "${global_config_file}" \
        port "${port}"
}

create_local_conf() {
    local local_config_file="$1"
    local listen_address="$2"
    local port="$3"

    if ! file_exists "${local_config_file}"; then
        cat > "${local_config_file}" <<EOF
listen_addresses = '${listen_address}'
port = ${port}
EOF
    fi
}

postgresql_version() {
    local db_path="$1"

    cat "${db_path}/PG_VERSION"
}

postgresql_compound_version() {
    local db_path="$1"
    local version major minor

    version="$(cat "${db_path}/PG_VERSION")"
    major="$(echo "${version}" | cut -f 1 -d .)"
    if [[ "${version}" = *.* ]]; then
        minor="$(echo "${version}" | cut -f 2 -d .)"
    else
        minor=0
    fi
    echo "$(( major*10 + minor ))"
}

configure_machine_independent_params() {
    local local_config_file="$1"
    local pg_compound_version="$2"
    local db_path="$3"
    local pg_cluster_name="$4"
    local db_config_template="$5"
    local dereferenced_db_path

    log "Configuring PostgreSQL ${pg_cluster_name} in ${db_path} with ${db_config_template} template"

    dereferenced_db_path="$(dereference_db_path "${db_path}")"

    set_pg_config_params "${local_config_file}" \
        cluster_name "'${pg_cluster_name}'" \
        event_source "'${pg_cluster_name}'" \
        data_directory "'${dereferenced_db_path}'" \
        \
        log_autovacuum_min_duration 60s \
        log_checkpoints on \
        log_connections on \
        log_destination "'stderr'" \
        log_directory "'pg_log'" \
        log_disconnections on \
        log_filename "'postgresql-%Y-%m-%d_%H%M%S.log'" \
        logging_collector on \
        log_line_prefix "'%m [%p-%l] app=%a %q%u@%d '" \
        log_lock_waits on \
        log_min_duration_statement 3000ms \
        log_min_messages "${PG_LOG_MIN_MESSAGES:-info}" \
        log_rotation_age 1d \
        log_rotation_size 1GB \
        log_temp_files 0 \
        \
        shared_preload_libraries "'pg_stat_statements,auto_explain'" \
        \
        pg_stat_statements.track all \
        pg_stat_statements.max 500 \
        \
        auto_explain.log_min_duration 30s \
        auto_explain.log_analyze true \
        auto_explain.log_timing false \
        auto_explain.log_nested_statements true \
        \
        autovacuum_max_workers 6 \
        autovacuum_freeze_max_age 1000000000 \
        vacuum_freeze_table_age 1000000000 \
        \
        cpu_index_tuple_cost 0.001 \
        cpu_operator_cost 0.0005 \
        \
        track_commit_timestamp on

    if [[ "${db_config_template}" = "redash" ]]; then
        set_pg_config_params "${local_config_file}" \
            max_connections 50 \
            default_statistics_target 1000
    elif [[ "${db_config_template}" = "starfish" ]]; then
        set_pg_config_params "${local_config_file}" \
            max_connections 1000 \
            \
            temp_buffers 2GB \
            maintenance_work_mem 2GB \
            \
            checkpoint_timeout 15min \
            checkpoint_completion_target 0.9 \
            checkpoint_warning 60s \
            \
            max_wal_size 18GB \
            min_wal_size 6GB \
            wal_buffers 128MB \
            wal_compression on \
            \
            max_locks_per_transaction 1000 \
            cursor_tuple_fraction 1 \
            \
            default_statistics_target 200 \
            \
            geqo_threshold 15 \
            from_collapse_limit 15 \
            join_collapse_limit 15 \
            \
            hot_standby off
    else
        log "Unsupported DB config template: ${db_config_template}"
        exit 1
    fi

    if [[ "${pg_compound_version}" -lt 96 ]]; then
        # PG 9.5 or earlier, should rather not happen, rest of configuration is for 9.6+
        return
    fi

    if [[ "${pg_compound_version}" -lt 100 ]]; then
        set_pg_config_params "${local_config_file}" password_encryption on
    else
        set_pg_config_params "${local_config_file}" password_encryption scram-sha-256
    fi

    if [[ "${pg_compound_version}" -lt 110 ]]; then
        # replacement_sort_tuples should be disabled under 9.6 and 10
        # this setting is removed under PG 11 and later
        # according to https://github.com/jberkus/annotated.conf/blob/master/annotated.csv
        set_pg_config_params "${local_config_file}" \
            replacement_sort_tuples 0
    fi

    if [[ "${db_config_template}" == starfish ]]; then
        # disconnecting lingering connections only for Starfish cluster, not sure how other clients
        # will behave (e.g. Redash)
        set_pg_config_params "${local_config_file}" \
            idle_in_transaction_session_timeout 1h
    fi

    if [[ "${pg_compound_version}" -ge 120 ]]; then
        # disable JIT as it slows down most of the queries. Probably additional tuning is required.
        set_pg_config_params "${local_config_file}" \
            jit off
    fi

}

configure_machine_specific_params() {
    local local_config_file="$1"
    local storage_type="$2"
    local pg_compound_version="$3"
    local db_config_template="$4"
    local num_cores mem_kb random_page_cost estimated_number_of_concurrent_joins effective_io_concurrency
    local work_mem_MB shared_buffers_MB effective_cache_size_MB
    local max_worker_processes max_parallel_workers max_parallel_workers_per_gather expected_sessions

    num_cores="$(getconf _NPROCESSORS_ONLN)"
    mem_kb="$(( $(getconf _PHYS_PAGES) * $(getconf PAGE_SIZE) / 1024 ))"

    # all parameters should correspond to document "Starfish PostgreSQL configuration guide" on Google Docs

    log "Configuring PostgreSQL ${pg_compound_version} for ${storage_type} storage with ${db_config_template} template"

    # effective_io_concurrency - Number of simultaneous requests that can be handled efficiently by the disk subsystem.
    #                            Set to the number of disks in your RAID array or number of I/O channels.
    #                            Currently only affects the execution of parallel bitmapscan,
    #                            but might affect other I/O operations in future versions.
    # random_page_cost - Sets the planner's estimate of the cost of a nonsequentially fetched disk page.
    #                    Sets the ratio of seek to scan time for your database storage.
    case "${storage_type}" in
        hdd)
            random_page_cost=4
            effective_io_concurrency=1
            autovacuum_vacuum_cost_limit=2000
            autovacuum_vacuum_cost_delay=20ms  # page hits: 800 MB/s page misses: 80 MB/s page dirty: 40 MB/s
            bgwriter_lru_maxpages=250
            bgwriter_delay=100ms  # max bw: 20 MB/s

        ;;
        raid|"raid (hdd)")
            random_page_cost=3
            effective_io_concurrency=8
            autovacuum_vacuum_cost_limit=5000
            autovacuum_vacuum_cost_delay=10ms  # page hits: 4 GB/s page misses: 400 MB/s page dirty: 200 MB/s
            bgwriter_lru_maxpages=300
            bgwriter_delay=50ms  # max bw: 48 MB/s
        ;;
        ssd)
            random_page_cost=1.5
            effective_io_concurrency=1000
            autovacuum_vacuum_cost_limit=10000
            autovacuum_vacuum_cost_delay=2ms  # page hits: 40 GB/s page misses: 4 GB/s page dirty: 2 GB/s
            bgwriter_lru_maxpages=500
            bgwriter_delay=20ms  # max bw: 200 MB/s
        ;;
        nvme)
            random_page_cost=1
            effective_io_concurrency=1000
            autovacuum_vacuum_cost_limit=10000
            autovacuum_vacuum_cost_delay=2ms  # page hits: 40 GB/s page misses: 4 GB/s page dirty: 2 GB/s
            bgwriter_lru_maxpages=500
            bgwriter_delay=20ms  # max bw: 200 MB/s
        ;;
        *)
            log "Invalid value for underlying storage type: ${storage_type}"
            exit 1
    esac

    if [[ "${db_config_template}" = "redash" ]]; then
        # Configure Redash DBs to use little memory if nothing else has been set
        # PG_WORK_MEM_MB and PG_SHARED_BUFFERS_MB may have been set from outside
        if [[ "${PG_WORK_MEM_MB:-0}" -eq 0 ]]; then
            PG_WORK_MEM_MB=32
        fi
        if [[ "${PG_SHARED_BUFFERS_MB:-0}" -eq 0 ]]; then
            PG_SHARED_BUFFERS_MB=512
        fi
    fi

    # work_mem sets the limit for the amount of non-shared RAM available for each query operation,
    # including sorts and hashes. This limit acts as a primitive resource control, preventing the server
    # from going into swap due to overallocation. Note that this is non-shared RAM per operation,
    # which means large complex queries can use multiple times this amount. Also, work_mem is
    # allocated by powers of two, so round to the nearest binary step.

    # this is a random estimate on how many concurrent joins we run. Each parallel worker can use up to work_mem_MB
    # but total number of parallel workers is limited to (num_cores+2) (see max_worker_processes)
    estimated_number_of_concurrent_joins="$((20 + 2 * num_cores))"
    if [[ "${PG_WORK_MEM_MB:-0}" -gt 0 ]]; then
        work_mem_MB="${PG_WORK_MEM_MB}"
    else
        work_mem_MB="$((20 * mem_kb / 100 / 1024 / estimated_number_of_concurrent_joins))"
    fi
    if [[ "${work_mem_MB}" -lt 32 ]]; then
        work_mem_MB=32
    elif [[ "${work_mem_MB}" -gt 1024 ]]; then
        work_mem_MB=1024
    fi

    # A memory quantity defining PostgreSQL's "dedicated" RAM, which is used for connection control,
    # active operations, and more. However, since PostgreSQL also needs free RAM for file system buffers,
    # sorts and maintenance operations, it is not advisable to set shared_buffers to a majority of RAM.
    if [[ "${PG_SHARED_BUFFERS_MB:-0}" -gt 0 ]]; then
        shared_buffers_MB="${PG_SHARED_BUFFERS_MB}"
    else
        shared_buffers_MB="$((3 * mem_kb / 10 / 1024))"
    fi
    if [[ "${shared_buffers_MB}" -gt 8192 ]]; then
        shared_buffers_MB=8192
    fi

    # Tells the PostgreSQL query planner how much RAM is estimated to be available for caching data,
    # in both shared_buffers and in the filesystem cache. This setting just helps the planner make
    # good cost estimates; it does not actually allocate the memory.

    # 75% recommended in https://github.com/jberkus/annotated.conf/blob/master/postgresql.10.simple.conf
    effective_cache_size_MB="$((75 * mem_kb / 100 / 1024))"

    set_pg_config_params "${local_config_file}" \
        effective_cache_size "${effective_cache_size_MB}MB" \
        effective_io_concurrency "${effective_io_concurrency}" \
        random_page_cost "${random_page_cost}" \
        shared_buffers "${shared_buffers_MB}MB" \
        work_mem "${work_mem_MB}MB" \
        autovacuum_vacuum_cost_limit "${autovacuum_vacuum_cost_limit}" \
        autovacuum_vacuum_cost_delay "${autovacuum_vacuum_cost_delay}" \
        bgwriter_lru_maxpages "${bgwriter_lru_maxpages}" \
        bgwriter_delay "${bgwriter_delay}"

    # A wild guess.
    # calculations taken from https://github.com/jberkus/annotated.conf/blob/master/extra.10.conf
    expected_sessions=8
    max_worker_processes="$(( num_cores + 2 ))"
    max_parallel_workers="${num_cores}"
    max_parallel_workers_per_gather="$(( 2 * num_cores / expected_sessions ))"

    if [[ "${pg_compound_version}" -ge 96 ]]; then
        set_pg_config_params "${local_config_file}" \
            max_worker_processes "${max_worker_processes}" \
            max_parallel_workers_per_gather "${max_parallel_workers_per_gather}"
    fi

    if [[ "${pg_compound_version}" -ge 100 ]]; then
        set_pg_config_params "${local_config_file}" \
            max_parallel_workers "${max_parallel_workers}"
    fi
}

configure_pg_for_ssl() {
    local local_config_file="$1"
    local pg_config_dir="$2"

    local pg_ssl_dir="${pg_config_dir}/ssl"
    local pg_ssl_name="pg-ssl"
    local pg_ssl_cert="${pg_ssl_dir}/${pg_ssl_name}.crt"
    local pg_ssl_key="${pg_ssl_dir}/${pg_ssl_name}.key"

    pkg_install openssl
    create_ssl_cert "${pg_ssl_dir}" "${pg_ssl_name}" "Starfish PostgreSQL"
    chown --recursive postgres.postgres "${pg_ssl_dir}"
    chmod --recursive og= "${pg_ssl_dir}"

    set_pg_config_params "${local_config_file}" \
        ssl on \
        ssl_ciphers "'ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-SHA384:ECDHE-RSA-AES256-SHA384:ECDHE-ECDSA-AES128-SHA256:ECDHE-RSA-AES128-SHA256'" \
        ssl_prefer_server_ciphers on \
        ssl_ecdh_curve "'prime256v1'" \
        ssl_cert_file "'${pg_ssl_cert}'" \
        ssl_key_file "'${pg_ssl_key}'"
}

configure_postgresql_initial() {
    local listen_address="$1"
    local port="$2"
    local db_path="$3"
    local pg_version="$4"
    local pg_cluster_name="$5"
    local db_storage_type="$6"
    local db_config_template="$7"
    local pg_config_dir local_config_file pg_compound_version

    pg_config_dir="$(get_pgconfig_dir "${db_path}" "${pg_version}" "${pg_cluster_name}")"
    local_config_file="$(get_pg_local_conf_file "${pg_config_dir}")"

    pg_compound_version="$(postgresql_compound_version "${db_path}")"

    create_local_conf "${local_config_file}" "${listen_address}" "${port}"
    configure_hba "${listen_address}" "${pg_config_dir}"
    configure_pg_for_ssl "${local_config_file}" "${pg_config_dir}"
    configure_machine_independent_params "${local_config_file}" "${pg_compound_version}" "${db_path}" "${pg_cluster_name}" "${db_config_template}"
    configure_machine_specific_params "${local_config_file}" "${db_storage_type}" "${pg_compound_version}" "${db_config_template}"
    configure_pg_to_use_local_config_file "${pg_config_dir}" "${port}"
}

enable_pg_stat_extension() {
    local port="$1"
    local db_name="$2"

    log "Enabling pg_stat_statements extension"
    psql_as_postgres "${port}" "${db_name}" "CREATE EXTENSION IF NOT EXISTS pg_stat_statements;"
    psql_as_postgres "${port}" "${db_name}" "ALTER EXTENSION pg_stat_statements UPDATE"
}

update_pgagent() {
    local port="$1"

    log "Updating pgagent extension"
    psql_as_postgres "${port}" postgres "ALTER EXTENSION pgagent UPDATE;"
}

cancel_pgagent_queries() {
    local port="${1:-${STARFISH_PG_PORT}}"

    log "Cancelling running pgagent queries"
    psql_as_postgres "${port}" postgres <<EOF
SELECT COUNT(pid) AS num_of_canceled_pgagent_queries FROM pg_stat_activity, LATERAL pg_cancel_backend(pid) WHERE usename = 'pgagent';
EOF
}

get_pg_port() {
    local pg_local_conf="$1"

    port="$(get_pg_config_param "${pg_local_conf}" port)"
    if [[ -z "${port}" ]]; then
        echo "${STARFISH_PG_PORT}"
    else
        echo "${port}"
    fi
}

reconfigure_postgresql() {
    local db_path="$1"
    local db_storage_type="$2"
    # assume that we want to reconfigure the "main" Starfish DB
    local pg_version="${3:-${STARFISH_PG_VERSION}}"
    local pg_cluster_name="${4:-${STARFISH_PG_CLUSTER_NAME}}"
    local db_config_template="${5:-${STARFISH_PG_CONFIG_TEMPLATE}}"
    local pg_config_dir local_config_file pg_compound_version pg_port
    local cancelled_pgagent_queries

    pg_config_dir="$(get_pgconfig_dir "${db_path}" "${pg_version}" "${pg_cluster_name}")"
    local_config_file="$(get_pg_local_conf_file "${pg_config_dir}")"

    pg_compound_version="$(postgresql_compound_version "${db_path}")"

    fail_if_missing_config_file "${local_config_file}"
    configure_machine_independent_params "${local_config_file}" "${pg_compound_version}" "${db_path}" "${pg_cluster_name}" "${db_config_template}"
    configure_machine_specific_params "${local_config_file}" "${db_storage_type}" "${pg_compound_version}" "${db_config_template}"

    pg_port="$(get_pg_port "${local_config_file}")"
    cancelled_pgagent_queries="$(cancel_pgagent_queries "${pg_port}")"
    if [[ "${cancelled_pgagent_queries}" -gt 0 ]]; then
        log "Cancelled ${cancelled_pgagent_queries} running pgagent queries"
    fi
    enable_pg_stat_extension "${pg_port}" "${STARFISH_PG_DB_NAME}"
    update_pgagent "${pg_port}"
}

_is_line_in_file() {
    local line="$1"
    local file="$2"

    [[ $(grep --count --fixed-strings "${line}" "${file}") -gt 0 ]]
}



centos_enable_pgdg_repos() {
    local major_version
    local pgdg_yum_repo=/etc/yum.repos.d/pgdg-redhat-all.repo
    local -a pgdg_repos

    # WARNING: this is called from post-install so can't call yum/dnf install as it will deadlock

    # if pgdg-redhat-all.repo is changed upstream need to use the latest, overwriting previous changes
    # that could've been done by this very script, disabling repos for some PostgreSQL versions
    if file_exists "${pgdg_yum_repo}".rpmnew; then
        mv --backup=numbered "${pgdg_yum_repo}" "${pgdg_yum_repo}.rpmold"
        mv --force "${pgdg_yum_repo}".rpmnew "${pgdg_yum_repo}"
    fi

    # augeas is needed for manipulating (enabling/disabling) yum repos
    # it's already available when called from post-install so it'll be no-op from post-install
    pkg_install augeas

    # pgdgXX-updates-debuginfo were renamed to pgdgXX-debuginfo upstream on 30 Nov 2021
    # so try to enable either the old or the new one
    pgdg_repos=(
        pgdg-common
        pgdg13 pgdg13-debuginfo pgdg13-updates-debuginfo
        "$@"
    )

    major_version="$(centos_get_major_version)"

    if [[ "${major_version}" -ge 8 ]]; then
        # RHEL 8 and later have "postgresql" module with own Postgres, we want official PGDG
        dnf_disable_module_if_available postgresql
        if [[ "$(centos_get_major_version)" -eq 8 ]]; then
            # https://yum.postgresql.org/news/centos8-llvm-repo-is-available/
            # In the latest 4 minor releases of Red Hat Enterprise Linux 8, Red Hat have broken compatibility of
            # PostgreSQL and GIS packages with the previous minor releases (like introducing LLVM 8 with RHEL 8.1,
            # LLVM 9 with RHEL 8.2, LLVM 10 with RHEL 8.3, and LLVM 11 with Poppler on RHEL 8.4).
            # This breaks compatibility with the previous releases, and also affects PostgreSQL updates for the users
            # who installed the llvmjit subpackage and PostGIS. The negative effects are:
            #  * RHEL: Users cannot update to the new LLVM and Poppler until the packages are rebuilt.
            #    This issue has been solved already, by updating the build servers to the new minor release immediately,
            #    and rebuilding affected packages.
            #  * Rocky Linux and CentOS, which lags behind RHEL, is also significantly affected by this breakage,
            #    blocking PostgreSQL minor updates. This is the problem that needs to be solved.
            # Today, we released a new repo called “pgdg-centos8-sysupdates” for these users.
            # This repo brings in the LLVM, CLANG and Poppler packages from latest RHEL (of course, rebuilt and signed
            # with our own key), which satisfy the llvmjit and PostGIS dependencies. Please note that this is optional,
            # because it may break other packages (if any) which depend on older versions of LLVM, CLANG and Poppler.
            # This feature is available for PostgreSQL 11 and above.
            # (Disabling rust-toolset is not a must, however dnf will otherwise throw warnings as it has a dependency to
            # the llvm-toolset module).
            pgdg_repos+=( pgdg-centos8-sysupdates )

            # rust-toolset must be first as it requires llvm-toolset
            dnf_disable_module_if_available rust-toolset llvm-toolset
        fi
    fi

    disable_all_repos "${pgdg_yum_repo}"
    enable_repos "${pgdg_yum_repo}" "${pgdg_repos[@]}"
}

centos_enable_archived_repo_for_pg_96() {
    # This is not used in product as it no longer depends on PG 9.6
    # Taken from https://yum.postgresql.org/repopackages/#pgredhatoldrepos
    tee /etc/yum.repos.d/pgdg-redhat-archive.repo >/dev/null <<'_EOF'
[pgdg96]
name=PostgreSQL 9.6 RPMs for RHEL/CentOS $releasever
baseurl=https://yum-archive.postgresql.org/9.6/redhat/rhel-$releasever-x86_64
enabled=1
gpgcheck=0
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-PGDG
_EOF
}
