/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */
-- similar to:
 -- zone_size_changes_redash_query.sql
 -- tag_in_zone_size_changes_redash_query.sql
 -- zone_tag_size_changes_redash_query.sql
-- whenever someone changes this file changing the ones above should also be considered

WITH tag_with_date_range AS (
    SELECT volume_name,
           tag,
           MIN(run_time) AS min_run_time,
           MAX(run_time) AS max_run_time
    FROM sf_reports.tags_history
    WHERE run_time >= (now() - interval '{{number_of_days_to_look_back}} days')
      AND SUBSTRING(tag, 1, length('__')) != '__'
    GROUP BY volume_name,
             tag
), tag_size_for_chosen_days AS (
    SELECT tags_history.volume_name,
           tags_history.tag,
           tags_history.run_time,
           SUM(COALESCE(tags_history.physical_size, tags_history.size)) AS size
    FROM sf_reports.tags_history AS tags_history
    INNER JOIN tag_with_date_range ON tag_with_date_range.tag = tags_history.tag
                                      AND tag_with_date_range.volume_name = tags_history.volume_name
                                      AND (tag_with_date_range.min_run_time = tags_history.run_time
                                           OR tag_with_date_range.max_run_time = tags_history.run_time)
    GROUP BY tags_history.volume_name,
             tags_history.tag,
             tags_history.run_time
), tag_size_delta AS (
    SELECT volume_name,
           tag,
           (lag(run_time) OVER (PARTITION BY tag, volume_name ORDER BY run_time))::date AS start_date,
           run_time::date AS end_date,
           size,
           size - lag(size) OVER (PARTITION BY tag, volume_name ORDER BY run_time) AS size_delta
    FROM tag_size_for_chosen_days
), tag_percentage_delta AS (
    SELECT tag,
           volume_name,
           start_date,
           end_date,
           CASE WHEN (size - size_delta) = 0 THEN
                CASE WHEN size_delta = 0 THEN 0
                ELSE 'Infinity'::float
                END
           ELSE
                (size_delta * 100) / (size - size_delta)
           END AS percentage_delta,
           (size - size_delta) AS previous_size,
           size AS current_size,
           size_delta
   FROM tag_size_delta
   WHERE size_delta IS NOT NULL AND start_date IS NOT NULL
)
SELECT '<a href="DASHBOARD_LINK(tag-detail-analytics)?p_tag='||urlencode(tag)||'">'||urlencode(tag)||'</a>' AS "tag",
       COALESCE(volume.display_name, tag_percentage_delta.volume_name) AS "volume",
       start_date AS "start date",
       end_date AS "end date",
       -- column below is printed in HTML to produce nice looking result in redash and force correct sorting order
       -- at the same time
       '<div order="' || LPAD((DENSE_RANK() OVER(ORDER BY percentage_delta ASC))::text, 10, '0')
           || '" class="' ||
           CASE
               WHEN (percentage_delta > 0) THEN 'bg-success'
               WHEN (percentage_delta < 0) THEN 'bg-danger'
               ELSE ''
           END
           || ' text-center">' || ROUND(percentage_delta) || '%</div>'
           AS "percent delta",
       ROUND(previous_size / (1024 * 1024 * 1024.0), 2) AS "previous physical size (GiB)",
       ROUND(current_size / (1024 * 1024 * 1024.0), 2) AS "current physical size (GiB)",
       ROUND(size_delta / (1024 * 1024 * 1024.0), 2) AS "delta physical size (GiB)"
FROM tag_percentage_delta
    LEFT JOIN sf_volumes.volume ON tag_percentage_delta.volume_name = sf_volumes.volume.name
WHERE ABS(percentage_delta) >= {{minimum_percent_change}}
  AND ABS(size_delta) >= {{minimum_delta_size_gb}}*(1024*1024*1024)::BIGINT
ORDER BY size_delta DESC, volume;
