/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */

/* The parameters for this query are set in the report "Scan Alert Parameters" (id scan_alert_parameters) because Redash Alerts don't support the queries with parameters */

DO $$
    BEGIN
        IF NOT EXISTS(SELECT proname FROM pg_catalog.pg_proc WHERE proname = 'get_parameters_for_redash_scan_alert') THEN
            CREATE FUNCTION get_parameters_for_redash_scan_alert()
            RETURNS TABLE(scans_back INT, heartbeat_interval TEXT, volume_interval TEXT, hb_interval INTERVAL, v_interval INTERVAL)
            AS $Body$
                SELECT 3, '12 hours', '3 days', INTERVAL '12 hours', INTERVAL '3 days';
            $Body$
            LANGUAGE sql STABLE PARALLEL SAFE;
        END IF;
    END
$$;

WITH params AS
(
    SELECT scans_back, heartbeat_interval, volume_interval
    FROM get_parameters_for_redash_scan_alert()
),
scans AS
(
    SELECT id, volume_id, type, state_name, heartbeat
    FROM sf_scans.scan
    WHERE heartbeat >= NOW() - INTERVAL '6 MONTH'
    ORDER BY volume_id, heartbeat DESC, id DESC
),
scan_arrays AS
(
    SELECT volume_id, ARRAY_AGG(state_name) AS states
    FROM scans
    GROUP BY volume_id
),
last_scans AS
(
    SELECT volume_id, states[1:(SELECT scans_back FROM params)] AS states
    FROM scan_arrays
),
monitor_scan_arrays AS
(
    SELECT volume_id, ARRAY_AGG(state_name) AS states, ARRAY_AGG(heartbeat) AS heartbeats
    FROM scans
    WHERE type ILIKE 'monitor%'
    GROUP BY volume_id
),
last_monitor_scans AS
(
    SELECT volume_id, states[1:(SELECT scans_back FROM params)] AS states, heartbeats[1:(SELECT scans_back FROM params)] AS heartbeat
    FROM monitor_scan_arrays
),
volume_scan_heartbeats AS
(
    SELECT volume_id, MAX(heartbeat) AS heartbeat
    FROM scans
    GROUP BY volume_id
),
volume_successful_scan_heartbeats AS
(
    SELECT volume_id, MAX(heartbeat) AS heartbeat
    FROM scans
    WHERE state_name = 'done'
    GROUP BY volume_id
),
volume_monitor_scans_in_progress AS
(
    SELECT volume_id
    FROM scans
    WHERE type ILIKE 'monitor%' AND state_name = 'in_progress'
    GROUP BY volume_id
),
checks AS
(
    -- warning when the last monitor scan failed and heartbeat was within the checked time range
    SELECT CONCAT('monitor scan failed within last ', (SELECT volume_interval FROM params)) AS problem, volume_id
    FROM last_monitor_scans
    WHERE states[1] = 'fail' AND heartbeat[1] >= NOW() - (SELECT volume_interval::INTERVAL FROM params)

    -- warning when the monitor scan is running but there was no heartbeat during the heartbeat-check time range.
    UNION SELECT CONCAT('monitor heartbeat more than ', (SELECT heartbeat_interval FROM params), ' ago'), volume_id
    FROM scans WHERE type ILIKE 'monitor%' AND state_name = 'in_progress' AND heartbeat < NOW() - (SELECT heartbeat_interval::INTERVAL FROM params)

    -- warning when the last scans (starting from the last one, the number decided by the parameter) failed
    UNION SELECT CONCAT((SELECT scans_back FROM params)::TEXT, ' failed scans in row'), volume_id
    FROM last_scans WHERE 'fail' = ALL(states) AND ARRAY_LENGTH(states, 1) >= (SELECT scans_back FROM params)

    -- warning when there was no scan (as a heartbeat) for the specified time range
    UNION SELECT CONCAT('volume scan more than ', (SELECT volume_interval FROM params), ' ago'), vol.id
    FROM sf_volumes.volume vol
    LEFT JOIN volume_scan_heartbeats scan ON scan.volume_id = vol.id
    WHERE scan.volume_id IS NULL OR scan.heartbeat < NOW() - (SELECT volume_interval::INTERVAL FROM params)

    -- warning when there was no successful scan (as a heartbeat) for the specified time range
    -- the monitor in_progress scans are excluded
    UNION SELECT CONCAT('volume successful scan more than ', (SELECT volume_interval FROM params), ' ago'), vol.id
    FROM sf_volumes.volume vol
    LEFT JOIN volume_successful_scan_heartbeats scan ON scan.volume_id = vol.id
    LEFT JOIN volume_monitor_scans_in_progress monitor ON monitor.volume_id = scan.volume_id
    WHERE scan.volume_id IS NULL OR (monitor.volume_id IS NULL AND scan.heartbeat < NOW() - (SELECT volume_interval::INTERVAL FROM params))

    ORDER BY volume_id, problem
),
volume_problems AS
(
    SELECT COALESCE(vol.name, checks.volume_id::TEXT) AS volume, problem
    FROM checks
    LEFT JOIN sf_volumes.volume vol ON vol.id = checks.volume_id
    ORDER BY volume, problem
),
problem_info AS (
  SELECT volume, STRING_AGG(problem, ', ') AS problems, 1 AS alert
  FROM volume_problems
  GROUP BY volume
  ORDER BY volume
)

SELECT volume, problems, alert FROM problem_info
UNION ALL SELECT 'all-volumes', 'no issues found', 0
          WHERE NOT EXISTS(SELECT 1 FROM problem_info)
