/* legal disclaimer in /opt/starfish/data/starfish/sql-copyright-and-license.md */
WITH dirs_stats AS (
    SELECT volume_id,
        COUNT(*) AS dirs,
        MAX((local_aggrs #>> '{total, size}')::BIGINT) AS max_non_recursive,
        SUM(size) AS total_dirs_size,
        COUNT(*) FILTER(WHERE (local_aggrs #>> '{total, files}')::BIGINT = 0) AS dirs_without_files,
        -- recursive aggregates count directories together with the one for which aggrs are computed, so it's 1
        -- for an empty directory
        COUNT(*) FILTER(WHERE (rec_aggrs->>'files')::BIGINT = 0 AND (rec_aggrs->>'dirs')::BIGINT = 1) AS empty_dirs,
        SUM(COALESCE(rec_aggrs->>'blocks_div_nlinks', rec_aggrs->>'blocks')::BIGINT) FILTER(WHERE path='') AS recursive_blocks_count_without_hardlinks,
        SUM(COALESCE(rec_aggrs->>'size_div_nlinks', rec_aggrs->>'size')::NUMERIC) FILTER(WHERE path='') AS recursive_size_without_hardlinks,
        MAX(depth) AS max_depth,
        ROUND(AVG(depth), 2) AS avg_depth
    FROM sf.dir_current
    GROUP BY volume_id
), files_count AS (
    SELECT
        volume_id,
        COUNT(*) FILTER(WHERE type=32768) AS files,
        COUNT(*) FILTER(WHERE type=8192) AS char_dev,
        COUNT(*) FILTER(WHERE type=24576) AS block_dev,
        COUNT(*) FILTER(WHERE type=4096) AS fifo,
        COUNT(*) FILTER(WHERE type=40960) AS symlinks,
        COUNT(*) FILTER(WHERE type=49152) AS sockets,
        SUM(size) AS total_file_size,
        MAX(size) FILTER(WHERE type=32768) AS max_file_size,
        PERCENTILE_DISC(0.5) WITHIN GROUP (ORDER BY size) FILTER(WHERE type=32768) AS file_median,
        AVG(size) FILTER(WHERE type=32768) AS avg_file_size
    FROM sf.file_current
    GROUP BY volume_id
), stats_without_volume_capacity_percent AS (
SELECT
    volumes.name AS "volume name",
    volumes.total_capacity AS "total capacity",
    dirs.recursive_blocks_count_without_hardlinks * 512 AS "volume occupied space",
    dirs.recursive_size_without_hardlinks AS "volume occupied space logical",
    dirs.max_depth AS "max depth",
    dirs.avg_depth AS "avg depth",
    files.files AS "regular files",
    files.char_dev AS "character devices",
    files.block_dev AS "block devices",
    files.fifo AS "FIFOs",
    files.symlinks AS "symbolic links",
    files.sockets AS "sockets",
    files.total_file_size AS "total file size",
    files.max_file_size AS "maximum file size",
    files.file_median AS "median file size",
    files.avg_file_size AS "average file size",
    dirs.dirs AS "directories",
    dirs.max_non_recursive AS "maximum directory non-recursive size",
    dirs.dirs_without_files AS "directories without files",
    dirs.empty_dirs AS "empty directories",
    dirs.total_dirs_size AS "total directories size",
    CASE
        WHEN COALESCE(volumes.total_capacity, 0) > 0 THEN
            GREATEST(volumes.total_capacity - dirs.recursive_blocks_count_without_hardlinks * 512, 0)
        ELSE NULL
    END AS "volume free space"
FROM dirs_stats AS dirs
    JOIN files_count AS files  ON dirs.volume_id = files.volume_id
    JOIN sf_volumes.volume AS volumes  ON dirs.volume_id = volumes.id
)
SELECT *,
         CASE WHEN "total capacity" > 0 THEN
            LEAST(ROUND(("total capacity" - "volume free space") * 100 / "total capacity", 1), 100)
         END AS "volume occupied space %",
         CASE WHEN "total capacity" > 0 THEN
            ROUND("volume free space" * 100 / "total capacity", 1)
         END AS "volume free space %"
    FROM stats_without_volume_capacity_percent;
